#' Create a VariantTrack for a custom JBrowse 2 view
#'
#' Creates the necessary configuration string for an
#' indexed VCF file so that it can be used
#' in a JBrowse custom linear genome view.
#'
#' It is important to note that while only the VCF file is
#' passed as an argument, \code{tracks_variant} assumes that a VCF
#' index of the same name is located with the file
#'
#' For example:
#'
#' \code{track_alignments("data/variants.vcf")}
#'
#' Assumes that \code{data/variants.vcf.tbi} also exists.
#'
#' This is a JBrowse 2 convention, and the default naming output of tabix
#'
#' For more information on creating an index with tabix, visit
#' \url{https://www.htslib.org/}
#'
#' @param track_data the URL to the VCF file
#' @param assembly the config string generated by \code{assembly}
#'
#' @return a character vector of stringified VariantTrack JSON configuration
#'
#' @export
#'
#' @examples
#' assembly <- assembly("https://jbrowse.org/genomes/hg19/fasta/hg19.fa.gz", bgzip = TRUE)
#'
#' track_variant("variants.vcf", assembly)
track_variant <- function(track_data, assembly) {
  check_vcf(track_data)

  type <- "VariantTrack"
  name <- get_name(track_data)
  assembly_name <- get_assembly_name(assembly)
  track_id <- stringr::str_c(assembly_name, "_", name)
  index <- stringr::str_c(track_data, ".tbi")

  as.character(
    stringr::str_glue(
      "{{ ",
      '"type": "{type}", ',
      '"name": "{name}", ',
      '"assemblyNames": ["{assembly_name}"], ',
      '"trackId": "{track_id}", ',
      '"adapter": {{ ',
      '"type": "VcfTabixAdapter", ',
      '"vcfGzLocation": {{ "uri": "{track_data}" }}, ',
      '"index": {{ "location": {{ "uri": "{index}"  }} }}',
      "}}",
      "}}"
    )
  )
}

check_vcf <- function(track_data) {
  track_non_gz <- strip_gz(track_data)
  if (!stringr::str_ends(track_non_gz, ".vcf")) {
    stop("variant data must be VCF")
  }
}
