% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/predict.SSModel.R
\name{predict.SSModel}
\alias{predict}
\alias{predict.SSModel}
\title{State Space Model Predictions}
\usage{
\method{predict}{SSModel}(object, newdata, n.ahead, interval = c("none",
  "confidence", "prediction"), level = 0.95, type = c("response", "link"),
  states = NULL, se.fit = FALSE, nsim = 0, prob = TRUE, maxiter = 50,
  filtered = FALSE, ...)
}
\arguments{
\item{object}{Object of class \code{SSModel}.}

\item{newdata}{A compatible \code{SSModel} object to be added in the end of
the old object for which the predictions are required. If omitted,
predictions are either for the past data points, or if argument
\code{n.ahead} is given, \code{n.ahead} time steps ahead.}

\item{n.ahead}{Number of steps ahead at which to predict. Only used if
\code{newdata} is omitted. Note that when using \code{n.ahead}, object
cannot contain time varying system matrices.}

\item{interval}{Type of interval calculation.}

\item{level}{Confidence level for intervals.}

\item{type}{Scale of the prediction, \code{"response"} or \code{"link"}.}

\item{states}{Which states are used in computing the predictions. Either a
numeric vector containing the indices of the corresponding states, or a
character vector defining the types of the corresponding states. Possible choices are
\code{"all"},  \code{"level"}, \code{"slope"},
\code{"trend"},  \code{"regression"}, \code{"arima"}, \code{"custom"},
\code{"cycle"} or \code{"seasonal"}, where \code{"trend"} extracts all states relating to trend.
 These can be combined. Default is \code{"all"}.}

\item{se.fit}{If TRUE, standard errors of fitted values are computed. Default is FALSE.}

\item{nsim}{Number of independent samples used in importance sampling. Used
only for non-Gaussian models.}

\item{prob}{if TRUE (default), the predictions in binomial case are
probabilities instead of counts.}

\item{maxiter}{The maximum number of iterations used in approximation Default
is 50. Only used for non-Gaussian model.}

\item{filtered}{If \code{TRUE}, compute predictions based on filtered
(one-step-ahead) estimates. Default is FALSE i.e. predictions are based on
all available observations given by user. For diffuse phase,
interval bounds and standard errors of fitted values are set to \code{-Inf}/\code{Inf}
(If the interest is in the first time points it might be useful to use
non-exact diffuse initialization.).}

\item{\dots}{Ignored.}
}
\value{
A matrix or list of matrices containing the predictions, and
  optionally standard errors.
}
\description{
Function \code{predict.SSModel} predicts the future observations of a state
space model of class \code{\link{SSModel}}.
}
\details{
For non-Gaussian models, the results depend whether importance sampling is
used (\code{nsim>0}). without simulations, the confidence intervals are based
on the Gaussian approximation of \eqn{p(\alpha | y)}. Confidence intervals in
response scale are computed in linear predictor scale, and then transformed
to response scale. The prediction intervals are not supported. With
importance sampling, the confidence intervals are computed as the empirical
quantiles from the weighted sample, whereas the prediction intervals contain
additional step of simulating the response variables from the sampling
distribution \eqn{p(y|\theta^i)}.

Predictions take account the uncertainty in state estimation
(given the prior distribution for the initial states), but not the uncertainty
of estimating the parameters in the system matrices (i.e. \eqn{Z}, \eqn{Q} etc.).
Thus the obtained confidence/prediction intervals can underestimate the true
uncertainty for short time series and/or complex models.

If no simulations are used, the standard errors in response scale are
computed using the Delta method.
}
\examples{

set.seed(1)
x <- runif(n=100,min=1,max=3)
y <- rpois(n=100,lambda=exp(x-1))
model <- SSModel(y~x,distribution="poisson")
xnew <- seq(0.5,3.5,by=0.1)
newdata <- SSModel(rep(NA,length(xnew))~xnew,distribution="poisson")
pred <- predict(model,newdata=newdata,interval="prediction",level=0.9,nsim=100)
plot(x=x,y=y,pch=19,ylim=c(0,25),xlim=c(0.5,3.5))
matlines(x=xnew,y=pred,col=c(2,2,2),lty=c(1,2,2),type="l")

model <- SSModel(Nile~SSMtrend(1,Q=1469),H=15099)
pred <- predict(model,n.ahead=10,interval="prediction",level=0.9)
pred
}

