% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/KPC.R
\name{KPCgraph}
\alias{KPCgraph}
\title{Kernel partial correlation with geometric graphs}
\usage{
KPCgraph(
  Y,
  X,
  Z,
  k = kernlab::rbfdot(1/(2 * stats::median(stats::dist(Y))^2)),
  Knn = 1,
  trans_inv = FALSE
)
}
\arguments{
\item{Y}{a matrix (n by dy)}

\item{X}{a matrix (n by dx) or \code{NULL} if \eqn{X} is empty}

\item{Z}{a matrix (n by dz)}

\item{k}{a function \eqn{k(y, y')} of class \code{kernel}. It can be the kernel implemented in \code{kernlab} e.g. Gaussian kernel: \code{rbfdot(sigma = 1)}, linear kernel: \code{vanilladot()}.}

\item{Knn}{number of nearest neighbor to use; or "MST"}

\item{trans_inv}{TRUE or FALSE. Is \eqn{k(y, y)} free of \eqn{y}?}
}
\value{
The algorithm returns a real number which is the estimated KPC.
}
\description{
Calculate the kernel partial correlation (KPC) coefficient with directed K-nearest neighbor (K-NN) graph or minimum spanning tree (MST).
}
\details{
The kernel partial correlation squared (KPC) measures the conditional dependence
between \eqn{Y} and \eqn{Z} given \eqn{X}, based on an i.i.d. sample of \eqn{(Y, Z, X)}.
It converges to the population quantity (depending on the kernel) which is between 0 and 1.
A small value indicates low conditional dependence between \eqn{Y} and \eqn{Z} given \eqn{X}, and
a large value indicates stronger conditional dependence.
If \code{X == NULL}, it returns the \code{KMAc(Y,Z,k,Knn)}, which measures the unconditional dependence between \eqn{Y} and \eqn{Z}.
Euclidean distance is used for computing the K-NN graph and the MST.
}
\examples{
library(kernlab)
n = 2000
x = rnorm(n)
z = rnorm(n)
y = x + z + rnorm(n,1,1)
KPCgraph(y,x,z,vanilladot(),Knn=1,trans_inv=FALSE)

n = 1000
x = runif(n)
z = runif(n)
y = (x + z) \%\% 1
KPCgraph(y,x,z,rbfdot(5),Knn="MST",trans_inv=TRUE)

discrete_ker = function(y1,y2) {
    if (y1 == y2) return(1)
    return(0)
}
class(discrete_ker) <- "kernel"
set.seed(1)
n = 2000
x = rnorm(n)
z = rnorm(n)
y = rep(0,n)
for (i in 1:n) y[i] = sample(c(1,0),1,prob = c(exp(-z[i]^2/2),1-exp(-z[i]^2/2)))
KPCgraph(y,x,z,discrete_ker,1)
##0.330413
}
\seealso{
\code{\link{KPCRKHS}}, \code{\link{KMAc}}, \code{\link{Klin}}
}
