#' Lioness Algorithm for Latin hypercube design
#'
#' \code{LA_LHD} returns a \code{n} by \code{k} Latin hypercube design matrix generated by lioness algorithm (LA)
#'
#' @param n A positive integer, which stands for the number of rows (or run size).
#' @param k A positive integer, which stands for the number of columns (or factor size).
#' @param m A positive integer, which stands for the number of starting lionesses agents. The default is set to be 10, and it is recommended to be no greater than 100.
#' @param N A positive integer, which stands for the number of iterations. The default is set to be 10. A large value of \code{N} will result a high CPU time, and it is recommended to be no greater than 500.
#' @param prun A probability, which stands for the probability of "prey runs away", and it is denoted as \code{p*} in the original paper. The default is set to be 1/(\code{k} - 1).
#' @param OC An optimality criterion. The default setting is "phi_p", and it could be one of the following: "phi_p", "AvgAbsCor", "MaxAbsCor", "MaxProCriterion".
#' @param p A positive integer, which is the parameter in the phi_p formula, and \code{p} is prefered to be large. The default is set to be 15.
#' @param q The default is set to be 1, and it could be either 1 or 2. If \code{q} is 1, the Manhattan (rectangular) distance will be used. If \code{q} is 2, the Euclidean distance will be used.
#' @param maxtime A positive number, which indicates the expected maximum CPU time given by user, and it is measured by minutes. For example, maxtime=3.5 indicates the CPU time will be no greater than three and half minutes. The default is set to be 5.
#'
#' @return If all inputs are logical, then the output will be a \code{n} by \code{k} LHD.
#'
#' @examples
#' #generate a 6 by 3 maximin distance LHD with the default setting
#' try=LA_LHD(n=6,k=3)
#' try
#' LHD::phi_p(try)   #calculate the phi_p of "try".
#'
#' #Another example
#' #generate a 9 by 4 nearly orthogonal LHD
#' try2=LA_LHD(n=9,k=4,OC="AvgAbsCor")
#' try2
#' LHD::AvgAbsCor(try2)  #calculate the average absolute correlation.
#' @export

LA_LHD=function(n,k,m=10,N=10,prun=1/(k-1),OC="phi_p",p=15,q=1,maxtime=5){
  #n and k are the rs and fs.
  #m: the number of starting search agents.
  #N: maximum number of iterations.
  #prun: the probability of prey runs away
  #OC: optimality criterion, the default is "phi_p", along with default p and q

  maxtime=maxtime*60  #convert minutes to seconds
  timeALL=NULL        #record all cpu time

  C=1  #Initialize counter index

  #step 1 starts
  X=rep(0,n*k*m)    #solution matrix

  dim(X)=c(n,k,m)

  for (i in 1:m) {
    X[,,i]=LHD::rLHD(n=n,k=k)
  }
  #step 1 ends

  if(OC=="phi_p"){

    #step 2 starts
    result=rep(0,m)

    for (i in 1:m) {
      result[i]=LHD::phi_p(X[,,i],p=p,q=q)
    }

    #step 2 ends

    progressbar = utils::txtProgressBar(min = 0, max = N, style = 3)

    while (C<=N) {  #step 3

      time0=Sys.time()

      temp=cbind(result,1:m)

      temp=temp[order(temp[,1]),]

      #step 4: determine the top 3 agents
      centre=X[,,temp[1,2]]

      LW=X[,,temp[2,2]]

      RW=X[,,temp[3,2]]

      #step 4 ends

      m=6*k+3

      Xnew=rep(0,n*k*m)       #new position matrix

      dim(Xnew)=c(n,k,m)

      Xnew[,,1]=centre

      Xnew[,,2]=LW

      Xnew[,,3]=RW

      #step 5 starts
      #centre troop
      index=4

      for (j in 1:k) {

        Xnew[,,index]=centre
        Xnew[,j,index]=LW[,j]
        index=index+1

        Xnew[,,index]=centre
        Xnew[,j,index]=RW[,j]
        index=index+1

      }

      #LW troop

      for (j in 1:k) {

        Xnew[,,index]=LW
        Xnew[,j,index]=centre[,j]
        index=index+1

        Xnew[,,index]=LW
        Xnew[,j,index]=RW[,j]
        index=index+1

      }

      #RW troop

      for (j in 1:k) {

        Xnew[,,index]=RW
        Xnew[,j,index]=centre[,j]
        index=index+1

        Xnew[,,index]=RW
        Xnew[,j,index]=LW[,j]
        index=index+1

      }

      #step 5 ends
      #Until here, Xnew has been fully updated

      X=Xnew

      for (i in 2:m) {      #step 6
        for (j in 1:k) {    #step 7

          z=stats::runif(1,0,1)             #step 8

          if (z<=prun){

            X[,,i]=LHD::exchange(X=X[,,i],j=j)   #step 9


          } #step 10: end if

        }   #step 11: end for
      }     #step 12: end for

      #step 13: update phi_p for all the agents
      result=rep(0,m)

      for (i in 1:m) {
        result[i]=LHD::phi_p(X[,,i],p=p,q=q)
      }

      time1=Sys.time()
      timediff=time1-time0
      timeALL=c(timeALL,timediff)

      ##########progress bar codes
      utils::setTxtProgressBar(progressbar, C)
      ##########

      if(as.numeric(sum(timeALL)+timediff)<=maxtime){C=C+1}
      if(as.numeric(sum(timeALL)+timediff)>maxtime){C=N+1}
    }


    temp=cbind(result,1:m)

    temp=temp[order(temp[,1]),]

    centre=X[,,temp[1,2]]

  }

  if(OC=="AvgAbsCor"){

    #step 2 starts
    result=rep(0,m)

    for (i in 1:m) {
      result[i]=LHD::AvgAbsCor(X[,,i])
    }

    #step 2 ends

    progressbar = utils::txtProgressBar(min = 0, max = N, style = 3)

    while (C<=N) {  #step 3

      time0=Sys.time()

      temp=cbind(result,1:m)

      temp=temp[order(temp[,1]),]

      #step 4: determine the top 3 agents
      centre=X[,,temp[1,2]]

      LW=X[,,temp[2,2]]

      RW=X[,,temp[3,2]]

      #step 4 ends

      m=6*k+3

      Xnew=rep(0,n*k*m)       #new position matrix

      dim(Xnew)=c(n,k,m)

      Xnew[,,1]=centre

      Xnew[,,2]=LW

      Xnew[,,3]=RW

      #step 5 starts
      #centre troop
      index=4

      for (j in 1:k) {

        Xnew[,,index]=centre
        Xnew[,j,index]=LW[,j]
        index=index+1

        Xnew[,,index]=centre
        Xnew[,j,index]=RW[,j]
        index=index+1

      }

      #LW troop

      for (j in 1:k) {

        Xnew[,,index]=LW
        Xnew[,j,index]=centre[,j]
        index=index+1

        Xnew[,,index]=LW
        Xnew[,j,index]=RW[,j]
        index=index+1

      }

      #RW troop

      for (j in 1:k) {

        Xnew[,,index]=RW
        Xnew[,j,index]=centre[,j]
        index=index+1

        Xnew[,,index]=RW
        Xnew[,j,index]=LW[,j]
        index=index+1

      }

      #step 5 ends
      #Until here, Xnew has been fully updated

      X=Xnew

      for (i in 2:m) {      #step 6
        for (j in 1:k) {    #step 7

          z=stats::runif(1,0,1)             #step 8

          if (z<=prun){

            X[,,i]=LHD::exchange(X=X[,,i],j=j)   #step 9


          } #step 10: end if

        }   #step 11: end for
      }     #step 12: end for

      #step 13: update AvgAbsCor for all the agents
      result=rep(0,m)

      for (i in 1:m) {
        result[i]=LHD::AvgAbsCor(X[,,i])
      }

      time1=Sys.time()
      timediff=time1-time0
      timeALL=c(timeALL,timediff)

      ##########progress bar codes
      utils::setTxtProgressBar(progressbar, C)
      ##########

      if(as.numeric(sum(timeALL)+timediff)<=maxtime){C=C+1}
      if(as.numeric(sum(timeALL)+timediff)>maxtime){C=N+1}
    }


    temp=cbind(result,1:m)

    temp=temp[order(temp[,1]),]

    centre=X[,,temp[1,2]]

  }

  if(OC=="MaxAbsCor"){

    #step 2 starts
    result=rep(0,m)

    for (i in 1:m) {
      result[i]=LHD::MaxAbsCor(X[,,i])
    }

    #step 2 ends

    progressbar = utils::txtProgressBar(min = 0, max = N, style = 3)

    while (C<=N) {  #step 3

      time0=Sys.time()

      temp=cbind(result,1:m)

      temp=temp[order(temp[,1]),]

      #step 4: determine the top 3 agents
      centre=X[,,temp[1,2]]

      LW=X[,,temp[2,2]]

      RW=X[,,temp[3,2]]

      #step 4 ends

      m=6*k+3

      Xnew=rep(0,n*k*m)       #new position matrix

      dim(Xnew)=c(n,k,m)

      Xnew[,,1]=centre

      Xnew[,,2]=LW

      Xnew[,,3]=RW

      #step 5 starts
      #centre troop
      index=4

      for (j in 1:k) {

        Xnew[,,index]=centre
        Xnew[,j,index]=LW[,j]
        index=index+1

        Xnew[,,index]=centre
        Xnew[,j,index]=RW[,j]
        index=index+1

      }

      #LW troop

      for (j in 1:k) {

        Xnew[,,index]=LW
        Xnew[,j,index]=centre[,j]
        index=index+1

        Xnew[,,index]=LW
        Xnew[,j,index]=RW[,j]
        index=index+1

      }

      #RW troop

      for (j in 1:k) {

        Xnew[,,index]=RW
        Xnew[,j,index]=centre[,j]
        index=index+1

        Xnew[,,index]=RW
        Xnew[,j,index]=LW[,j]
        index=index+1

      }

      #step 5 ends
      #Until here, Xnew has been fully updated

      X=Xnew

      for (i in 2:m) {      #step 6
        for (j in 1:k) {    #step 7

          z=stats::runif(1,0,1)             #step 8

          if (z<=prun){

            X[,,i]=LHD::exchange(X=X[,,i],j=j)   #step 9


          } #step 10: end if

        }   #step 11: end for
      }     #step 12: end for

      #step 13: update MaxAbsCor for all the agents
      result=rep(0,m)

      for (i in 1:m) {
        result[i]=LHD::MaxAbsCor(X[,,i])
      }

      time1=Sys.time()
      timediff=time1-time0
      timeALL=c(timeALL,timediff)

      ##########progress bar codes
      utils::setTxtProgressBar(progressbar, C)
      ##########

      if(as.numeric(sum(timeALL)+timediff)<=maxtime){C=C+1}
      if(as.numeric(sum(timeALL)+timediff)>maxtime){C=N+1}
    }


    temp=cbind(result,1:m)

    temp=temp[order(temp[,1]),]

    centre=X[,,temp[1,2]]

  }

  if(OC=="MaxProCriterion"){

    #step 2 starts
    result=rep(0,m)

    for (i in 1:m) {
      result[i]=LHD::MaxProCriterion(X[,,i])
    }

    #step 2 ends

    progressbar = utils::txtProgressBar(min = 0, max = N, style = 3)

    while (C<=N) {  #step 3

      time0=Sys.time()

      temp=cbind(result,1:m)

      temp=temp[order(temp[,1]),]

      #step 4: determine the top 3 agents
      centre=X[,,temp[1,2]]

      LW=X[,,temp[2,2]]

      RW=X[,,temp[3,2]]

      #step 4 ends

      m=6*k+3

      Xnew=rep(0,n*k*m)       #new position matrix

      dim(Xnew)=c(n,k,m)

      Xnew[,,1]=centre

      Xnew[,,2]=LW

      Xnew[,,3]=RW

      #step 5 starts
      #centre troop
      index=4

      for (j in 1:k) {

        Xnew[,,index]=centre
        Xnew[,j,index]=LW[,j]
        index=index+1

        Xnew[,,index]=centre
        Xnew[,j,index]=RW[,j]
        index=index+1

      }

      #LW troop

      for (j in 1:k) {

        Xnew[,,index]=LW
        Xnew[,j,index]=centre[,j]
        index=index+1

        Xnew[,,index]=LW
        Xnew[,j,index]=RW[,j]
        index=index+1

      }

      #RW troop

      for (j in 1:k) {

        Xnew[,,index]=RW
        Xnew[,j,index]=centre[,j]
        index=index+1

        Xnew[,,index]=RW
        Xnew[,j,index]=LW[,j]
        index=index+1

      }

      #step 5 ends
      #Until here, Xnew has been fully updated

      X=Xnew

      for (i in 2:m) {      #step 6
        for (j in 1:k) {    #step 7

          z=stats::runif(1,0,1)             #step 8

          if (z<=prun){

            X[,,i]=LHD::exchange(X=X[,,i],j=j)   #step 9


          } #step 10: end if

        }   #step 11: end for
      }     #step 12: end for

      #step 13: update MaxProCriterion for all the agents
      result=rep(0,m)

      for (i in 1:m) {
        result[i]=LHD::MaxProCriterion(X[,,i])
      }

      time1=Sys.time()
      timediff=time1-time0
      timeALL=c(timeALL,timediff)

      ##########progress bar codes
      utils::setTxtProgressBar(progressbar, C)
      ##########

      if(as.numeric(sum(timeALL)+timediff)<=maxtime){C=C+1}
      if(as.numeric(sum(timeALL)+timediff)>maxtime){C=N+1}
    }


    temp=cbind(result,1:m)

    temp=temp[order(temp[,1]),]

    centre=X[,,temp[1,2]]

  }


  avgtime=round(mean(timeALL),2)
  iterations=length(timeALL)

  close(progressbar)
  print(paste0("average CPU time per iteration is: ", avgtime, " seconds"))
  print(paste0("the number of iterations completed is: ", iterations))


  #step 15: return centre
  centre

}
