#' Set taxonomy summary configuration
#' @description This function set taxonomy summary configuration by assigning treatment column number, facet column number, replication column number, treatment mapping color, treatment order and facet order.
#' @param taxobj Taxonomic summary object generated by \code{\link{tax_summary}}
#' @param treat_location Numeric. Treatment column number in metafile/groupinformation.
#' @param facet_location Numeric, default:NULL. Facet column number in metafile/groupinformation.
#' @param rep_location Numeric. Replication column number in metafile/groupinformation.
#' @param subject_location Numeric, default:NULL.Subject column number in metafile/groupinformation (used for pairwise experiment).
#' @param treat_col Named character string, default:NULL. A set of aesthetic character to map treatment to.
#' @param treat_order Character string, default:NULL. The character string indicating treatment displaying order.
#' @param facet_order Character string, default:NULL. The character string indicating facet displaying order.
#'
#' @return object_config returns taxonomy summary object with configuration.
#' @export
#'
#' @author  Wang Ningqi<2434066068@qq.com>
#' @examples
#' {
#'   ### Data preparation ###
#'   data(testotu)
#'   groupinformation <- data.frame(
#'     group = c(rep("a", 10), rep("b", 10)),
#'     factor1 = rnorm(10),
#'     factor2 = rnorm(mean = 100, 10),
#'     subject = factor(c(1:10, 1:10)),
#'     group2 = c(rep("e", 5), rep("f", 5), rep("e", 5), rep("f", 5))
#'   )
#'
#'   ### Packaging metafile, community data, and taxonomy table ###
#'   test_object <- tax_summary(
#'     groupfile = groupinformation,
#'     inputtable = testotu[, 2:21],
#'     reads = TRUE,
#'     taxonomytable = testotu[, c(1, 22)]
#'   )
#'
#'   ### Object configuration ###
#'   test_object_plan1 <- object_config(
#'     taxobj = test_object,
#'     treat_location = 1,
#'     rep_location = 4
#'   )
#'
#'   ### Facet configuration ###
#'   test_object_plan2 <- object_config(
#'     taxobj = test_object,
#'     treat_location = 1,
#'     rep_location = 4,
#'     facet_location = 5
#'   )
#' }
object_config=function(taxobj,treat_location,facet_location=NULL,rep_location,subject_location=NULL,treat_col=NULL,treat_order=NULL,facet_order=NULL){
  gp=taxobj[["Groupfile"]]
  cond=unique(gp[,treat_location])
  fit=which(cond %in%treat_order)
  if(treat_location>ncol(taxobj$Groupfile)){
    stop("Invalid 'treat_location',Please check again!")
  }
  if(rep_location>ncol(taxobj$Groupfile)){
    stop("Invalid 'rep_location',Please check again!")
  }
  taxobj$Groupfile[,rep_location]=as.numeric(taxobj$Groupfile[,rep_location]) %>% suppressWarnings()

  if(length(which(is.na(taxobj$Groupfile[,rep_location])==TRUE))>1){
    stop("Invalid 'rep_location',please confirm replication column in you metadata is in 'numeric'!")
  }
  if(!is.null(treat_order)){
    if( length(fit)!=length(cond)){
      warning('treat_order does not match your treatment, please check!')
      return(NULL)
    }
  }
  configuration=list(treat_location,facet_location,rep_location,treat_col,treat_order,facet_order,subject_location)
  names(configuration)=c("treat_location","facet_location","rep_location","treat_col","treat_order","facet_order","subject_location")
  taxobj$configuration=configuration
  return(taxobj)
}
