% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Lorenz-bootstrap.R
\name{Lorenz.boot}
\alias{Lorenz.boot}
\title{Bootstrap for the (penalized) Lorenz regression}
\usage{
Lorenz.boot(object, R, boot_out_only = FALSE, ...)
}
\arguments{
\item{object}{An object with S3 class \code{"LR"} or \code{"PLR"}, i.e. the return of a call to the \code{\link{Lorenz.Reg}} function.}

\item{R}{An integer indicating the number of bootstrap replicates.}

\item{boot_out_only}{A logical determining whether the function should return raw bootstrap results. This is an advanced feature that helps save computation time in certain instances. See Details.}

\item{...}{Additional parameters corresponding to arguments passed to the function \code{\link[boot]{boot}} from the \emph{boot} library.}
}
\value{
An object of class \code{c("LR_boot", "LR")} or \code{c("PLR_boot", "PLR")}, depending on whether a non-penalized or penalized regression was fitted.

The methods \code{\link{confint.LR}} and \code{\link{confint.PLR}} are used on objects of class \code{"LR_boot"} or \code{"PLR_boot"} to construct confidence intervals for the model parameters.

For the non-penalized Lorenz regression, the returned object is a list containing the following components:
\describe{
   \item{\code{theta}}{The estimated vector of parameters. In the penalized case, it is a matrix where each row corresponds to a different selection method (e.g., BIC, bootstrap, cross-validation).}
   \item{\code{Gi.expl}}{The estimated explained Gini coefficient. In the penalized case, it is a vector, where each element corresponds to a different selection method.}
   \item{\code{LR2}}{The Lorenz-\eqn{R^2} of the regression. In the penalized case, it is a vector, where each element corresponds to a different selection method.}
   \item{\code{boot_out}}{An object of class \code{"boot"} containing the output of the bootstrap calculation.}
}
For the penalized Lorenz regression, the returned object is a list containing the following components:
\describe{
   \item{\code{path}}{See \code{\link{Lorenz.Reg}} for the original path. To this path is added the out-of-bag (OOB) score.}
   \item{\code{lambda.idx}}{A vector indicating the index of the optimal lambda obtained by each selection method.}
   \item{\code{grid.idx}}{A vector indicating the index of the optimal grid parameter obtained by each selection method.}
}
Note: in the penalized case, the returned object may have additional classes such as \code{"PLR_cv"} if cross-validation was performed and used as a selection method.
}
\description{
\code{Lorenz.boot} determines bootstrap estimators for the vector of coefficients of the single-index model, explained Gini coefficient and Lorenz-\eqn{R^2}. In the penalized case, it also provides a selection method.
}
\details{
Users that want to perform parallel computing have two options. The first and most obvious option is to use the facilities provided by the function \code{\link[boot]{boot}}.
Indeed, arguments such as \code{parallel}, \code{ncpus} and \code{cl} can be passed through the \code{...}.
Alternatively, users might want to run different instances of the function, each taking care of a portion of the bootstrap samples.
The argument \code{boot_out_only} can be set to \code{TRUE} to avoid unnecessary computations. If so, the returned object does not inherit from the class \code{"LR_boot"} or \code{"PLR_boot"}. The function simply returns the original \code{object}, to which is added the \code{boot_out} object.
If this second option is chosen, the instances have to be combined using the function \code{\link{Lorenz.boot.combine}}.
}
\section{References}{

Heuchenne, C. and A. Jacquemain (2022). Inference for monotone single-index conditional means: A Lorenz regression approach. \emph{Computational Statistics & Data Analysis 167(C)}.

Jacquemain, A., C. Heuchenne, and E. Pircalabelu (2024). A penalised bootstrap estimation procedure for the explained Gini coefficient. \emph{Electronic Journal of Statistics 18(1) 247-300}.
}

\examples{
\dontshow{
utils::example(Lorenz.Reg, echo = FALSE)
}
# Continuing the Lorenz.Reg(.) example for the non-penalized regression:
# This example is not run as it takes > 5 seconds to run.
\dontrun{
set.seed(123)
NPLR_boot <- Lorenz.boot(NPLR, R = 30)
# The method confint() is available to objects of class "LR_boot".
confint(NPLR_boot)
summary(NPLR_boot)
}
# Continuing the Lorenz.Reg(.) example for the penalized regression:
set.seed(123)
PLR_boot <- Lorenz.boot(PLR, R = 20)
# The object now inherits from the class "PLR_boot"
# Hence the methods (also) display the results obtained by bootstrap.
print(PLR_boot)
summary(PLR_boot)
coef(PLR_boot, pars.idx = "Boot")
predict(PLR_boot, pars.idx = "Boot")
plot(PLR_boot)
# Plot of the scores for each selection method depending on the grid and penalty parameters
plot(PLR_boot, type = "diagnostic")
# The method confint() is available to objects of class "PLR_boot".
confint(PLR_boot, pars.idx = "BIC") # Using the tuning parameters selected by BIC
confint(PLR_boot, pars.idx = "Boot") # Using the tuning parameters selected by bootstrap

}
\seealso{
\code{\link{Lorenz.Reg}}, \code{\link{Lorenz.GA}}, \code{\link{Lorenz.SCADFABS}}, \code{\link{Lorenz.FABS}}, \code{\link{PLR.CV}}, \code{\link[boot]{boot}}
}
