
#' @rdname SS2MOM
#' @param import_mov Logical. Import movement matrix?
#' @export
SS2OM <- function(SSdir, nsim = 48, proyears = 50, reps = 1, maxF = 3, seed = 1, interval = 1, pstar = 0.5,
                  Obs = MSEtool::Generic_Obs, Imp = MSEtool::Perfect_Imp,
                  import_mov = TRUE, gender = 1:2, seasons_to_years = TRUE, silent = FALSE,
                  Name = "OM generated by SS2OM function",
                  Source = "No source provided", Author = "No author provided",
                  report = FALSE, filename = "SS2OM", dir = tempdir(), open_file = TRUE, ...) {

  if(is.list(SSdir)) {
    replist <- SSdir
  } else {
    replist <- SS_import(SSdir, silent, ...)
  }

  if(replist$nsexes == 1) gender <- 1

  if(!silent) message("Converting SS output to MOM...")
  MOM <- SS2MOM(replist, nsim = nsim, proyears = proyears, reps = reps, maxF = maxF, seed = seed,
                interval = interval, pstar = pstar, Obs = Obs, Imp = Imp, silent = silent,
                Name = Name, Source = Source)

  if(!silent) message("Converting MOM to OM...")
  OM <- SSMOM2OM(MOM, replist, gender, import_mov, seed, silent)

  if(replist$nseasons == 1 && replist$seasduration < 1 && seasons_to_years) {
    message("Model with season as years found. Will convert to annual time step.")
    OM <- SS_seasonalyears_to_annual(OM, replist)
  }
  if(report) plot_SS2OM(OM, replist, filename, dir, open_file, silent)

  return(OM)
}

#' @rdname SS2MOM
#' @param MOM MOM object
#' @export
SSMOM2OM <- function(MOM, SSdir, gender = 1:2, import_mov = TRUE, seed = 1, silent = FALSE) {
  Factor <- NULL # variable for binding check

  if(!requireNamespace("reshape2", quietly = TRUE)) {
    stop("Package `reshape2` is required for this function. Install with `install.packages('reshape2')`", call. = FALSE)
  }

  if(is.list(SSdir)) {
    replist <- SSdir
  } else {
    replist <- SS_import(SSdir, silent)
  }

  if(replist$nsexes == 1) gender <- 1

  Stocks <- MOM@Stocks
  Fleets <- MOM@Fleets
  Obs <- MOM@Obs[[1]][[1]]
  Imp <- MOM@Imps[[1]][[1]]
  cpars <- MOM@cpars
  nsim <- MOM@nsim
  nyears <- Fleets[[1]][[1]]@nyears
  proyears <- MOM@proyears
  mainyrs <- replist$startyr:replist$endyr

  Stock <- new("Stock")
  Stock@maxage <- vapply(Stocks, slot, numeric(1), "maxage") %>% unique()
  Stock@R0 <- vapply(Stocks, slot, numeric(1), "R0") %>% sum()

  Stock@SRrel <- Stocks[[1]]@SRrel
  Stock@h <- Stocks[[1]]@h
  Stock@D <- rep(replist$current_depletion, 2)
  Stock@AC <- Stocks[[1]]@AC
  Stock@Perr <- Stocks[[1]]@Perr

  Stock@Msd <- Stock@Linfsd <- Stock@Ksd <- Stock@L50 <- Stock@L50_95 <- c(0, 0)
  Stock@Size_area_1 <- Stock@Frac_area_1 <- Stock@Prob_staying <- c(0.5, 0.5)

  # cpars
  cpars_out <- list()

  # This function grabs array tt from the cpars of the first fleet of each stock, and averages across genders
  mean_array <- function(tt) {
    lapply(cpars[gender], function(x) parse(text = paste0("x[[1]]$", tt)) %>% eval()) %>%
      simplify2array() %>% apply(1:3, mean)
  }

  cpars_out$M_ageArray <- mean_array("M_ageArray")
  cpars_out$Wt_age <- mean_array("Wt_age")
  cpars_out$Len_age <- mean_array("Len_age")
  cpars_out$LatASD <- mean_array("LatASD")

  mean_vector <- function(tt) {
    out <- lapply(cpars[gender], function(x) parse(text = paste0("x[[1]]$", tt)) %>% eval())
    if(length(out) == 2 && is.null(out[[2]])) {
      res <- out[[1]]
    } else {
      res <- simplify2array(out) %>% apply(1, mean)
    }
    return(res)
  }

  cpars_out$Linf <- mean_vector("Linf")
  cpars_out$K <- mean_vector("K")
  cpars_out$t0 <- mean_vector("t0")

  # Stock placeholders (overriden by cpars mean_arrays or mean_vectors above)
  Stock@M <- vapply(Stocks[gender], slot, numeric(2), "M") %>% apply(1, mean)
  Stock@LenCV <- vapply(Stocks[gender], slot, numeric(2), "LenCV") %>% apply(1, mean)

  Stock@Linf <- cpars_out$Linf %>% range()
  Stock@K <- cpars_out$K %>% range()
  Stock@t0 <- cpars_out$t0 %>% range()

  slot2 <- function(x, y) ifelse(length(slot(x, y)), slot(x, y), NA_real_)
  Stock@a <- vapply(Stocks[gender], slot2, numeric(1), "a") %>% mean(na.rm = TRUE)
  Stock@b <- vapply(Stocks[gender], slot2, numeric(1), "b") %>% mean(na.rm = TRUE)

  # cpars for the first gender, first fleet
  .cpars <- cpars[[1]][[1]]
  cpars_out$hs <- .cpars$hs
  #cpars_out$binWidth <- .cpars$binWidth
  cpars_out$CAL_bins <- .cpars$CAL_bins
  #cpars_out$CAL_binsmid <- .cpars$CAL_binsmid
  cpars_out$Mat_age <- .cpars$Mat_age

  # Sample recruitment deviations
  Perr_proj <- exp(sample_recruitment(log(.cpars$Perr_y), proyears,
                                      replist$sigma_R_in, Stocks[[1]]@AC[1], seed))
  cpars_out$Perr_y <- cbind(.cpars$Perr_y, Perr_proj)

  # Do movement
  if(import_mov && !is.null(replist$movement) && nrow(replist$movement)) {
    movement <- replist$movement[replist$movement$Seas == 1 & replist$movement$Gpattern == 1, ]
    if(!nrow(movement)) movement <- replist$movement[replist$movement$Seas == 1 & replist$movement$GP == 1, ]

    nareas <- length(unique(movement$Source_area))
    if(!silent) message(nareas, " area model found. Parameterizing movement matrix.")

    full_movement <- vapply(0:Stock@maxage, function(x) movement[, paste0("age", x)], numeric(nrow(movement)))

    cpars_out$mov <- array(0, c(nsim, Stock@maxage + 1, nareas, nareas))

    for(i in 1:nrow(full_movement)) {
      from <- movement$Source_area[i]
      to <- movement$Dest_area[i]
      for(j in 1:ncol(full_movement)) cpars_out$mov[1:nsim, j, from, to] <- full_movement[i, j]
    }
    #mov[is.na(mov)] <- 0
    #cpars_out$mov <- mov
  }

  # Fleet
  Fleet <- new("Fleet")
  Fleet@Name <- names(Fleets[[1]]) %>% paste(collapse = ", ")
  Fleet@nyears <- nyears
  Fleet@CurrentYr <- max(mainyrs)

  # Placeholders
  Fleet@Spat_targ <- c(1, 1)
  Fleet@Esd <- Fleet@qinc <- Fleet@qcv <- Fleet@L5 <- Fleet@LFS <- Fleet@Vmaxlen <- c(0, 0)
  Fleet@isRel <- FALSE
  Fleet@DR <- Stock@Fdisc <- c(0, 0) # No discards
  Fleet@MPA <- FALSE

  FF <- dplyr::filter(replist$ageselex, Factor == "F", Yr %in% mainyrs)
  if(nrow(FF)) { # Search for F-at-age in replist$ageselex otherwise go to replist$exploitation
    # Need to average over seasons
    # Selectivity = all dead catch
    # No discards are modeled
    FF <- dplyr::filter(replist$ageselex, Factor == "F", Yr %in% mainyrs)[, -c(1, 4, 6, 7)] %>%
      reshape2::melt(list("Fleet", "Yr", "Sex"), variable.name = "Age", value.name = "F") %>%
      group_by(Yr, Sex, Age) %>% summarise(F = sum(F)) %>% # Sum over fleets
      group_by(Yr, Age) %>% summarise(F = mean(F)) %>% # Mean over sexes
      group_by(Yr) %>% mutate(F_denom = pmax(F, 1e-8), V = F/max(F_denom)) # Get vulnerability, F_denom avoids divide-by-zero
    Find <- group_by(FF, Yr) %>% summarise(F = max(F)) %>% getElement("F")
    V_temp <- reshape2::acast(FF, Age ~ Yr, value.var = "V")
  } else {
    # Selectivity = All mortality
    Vfleet <- lapply(c(1:replist$nfleets)[replist$IsFishFleet], get_V_from_Asel2, i = gender,
                     replist = replist, mainyrs = mainyrs, maxage = Stock@maxage)

    Fapic <- replist$exploitation[, replist$FleetNames[replist$IsFishFleet] %in% colnames(replist$exploitation)][, -3] %>%
      dplyr::filter(Yr %in% mainyrs) %>% structure(names = c("Yr", "Seas", c(1:replist$nfleets)[replist$IsFishFleet])) %>%
      reshape2::melt(value.name = "FF", variable.name = "Fleet", id.vars = c("Yr", "Seas")) %>% group_by(Yr, Fleet) %>%
      summarise(FF = sum(FF)) %>% reshape2::acast(list("Yr", "Fleet"), value.var = "FF")

    F_at_age <- lapply(1:length(Vfleet), function(x) t(Vfleet[[x]]) * Fapic[, x]) %>%
      simplify2array() %>% apply(1:2, sum)

    Find <- apply(F_at_age, 1, max)
    F_denom <- pmax(Find, 1e-8)
    V_temp <- t(F_at_age/F_denom)
  }
  V <- vapply(1:ncol(V_temp), function(x) { # Grab selectivity from neighboring years if all zeros
    x <- min(x, ncol(V_temp))
    if(!sum(V_temp[, x], na.rm = TRUE) && x < ncol(V_temp)) {
      Recall(x+1)
    } else {
      return(V_temp[, x])
    }
  }, numeric(nrow(V_temp)))

  Vpro <- V[, ncol(V)] %>% matrix(nrow(V), proyears)
  cpars_out$V <- cbind(V, Vpro) %>% array(c(Stock@maxage + 1, nyears + proyears, nsim)) %>% aperm(c(3, 1 ,2))

  cpars_out$Find <- matrix(Find, nsim, nyears, byrow = TRUE)

  # Place holders
  Fleet@EffYears <- 1:nyears
  Fleet@EffLower <- Fleet@EffUpper <- Find

  OM <- suppressMessages(new("OM", Stock = Stock, Fleet = Fleet, Obs = Obs, Imp = Imp))
  OM@nsim <- MOM@nsim
  OM@proyears <- MOM@proyears
  OM@reps <- MOM@reps
  OM@maxF <- MOM@maxF
  OM@seed <- MOM@seed
  OM@interval <- MOM@interval
  OM@pstar <- MOM@pstar
  OM@cpars <- cpars_out
  return(OM)
}

#' @rdname SS2MOM
#' @export
plot_SS2OM <- function(x, SSdir, gender = 1:2,
                       filename = "SS2OM", dir = tempdir(), open_file = TRUE, silent = FALSE, ...) {
  if(missing(SSdir)) stop("SSdir not found.")

  if(inherits(x, "OM")) {
    if(!silent) message("Generating Hist object from OM...")
    OM <- x
    Hist <- runMSE(OM, Hist = TRUE, silent = silent)
  } else if(inherits(x, "Hist")) {
    Hist <- x
  } else {
    stop("Neither Hist nor OM object was found.", call. = FALSE)
  }

  if(is.list(SSdir)) {
    replist <- SSdir
  } else {
    replist <- SS_import(SSdir, silent, ...)
  }

  if(replist$nsexes == 1) gender <- 1

  rmd_file <- file.path(system.file(package = "MSEtool"), "Rmd", "SS", "SS2OM.Rmd")
  rmd <- readLines(rmd_file)

  write(rmd, file = file.path(dir, paste0(filename, ".rmd")))

  if(!silent) message("Rendering markdown file to HTML: ", file.path(dir, paste0(filename, ".html")))

  out <- rmarkdown::render(file.path(dir, paste0(filename, ".rmd")), "html_document", paste0(filename, ".html"), dir,
                           output_options = list(df_print = "paged"), quiet = TRUE)
  message("Rendering complete.")

  if(open_file) browseURL(out)
  return(invisible(out))
}
