#### All Methods in relation with the sparseVector (sub)classes


## atomicVector : classUnion (logical,integer,double,....)
setAs("atomicVector", "sparseVector",
      function(from) {
	  n <- length(from)# *is* integer for atomic vectors
	  r <- new(paste0(.V.kind(from), "sparseVector"), length = n)
	  ii <- isN0(from)
	  r@x <- from[ii]
	  r@i <- seq_len(n)[ii]
	  r
      })
## dsparseVector: currently important, as later potentially made into d..Matrix :
setAs("atomicVector", "dsparseVector",
      function(from) {
	  n <- length(from)# *is* integer for atomic vectors
	  r <- new("dsparseVector", length = n)
	  ii <- isN0(from)
	  r@x <- as.numeric(from)[ii]
	  r@i <- seq_len(n)[ii]
	  r
      })

setAs("nsparseVector", "lsparseVector",
      function(from) new("lsparseVector", i = from@i, length = from@length,
			 x = rep.int(TRUE, length(from@i))))
setAs("nsparseVector", "dsparseVector", function(from)
      as(as(from, "lsparseVector"), "dsparseVector"))
setAs("nsparseVector", "isparseVector", function(from)
      as(as(from, "lsparseVector"), "isparseVector"))
setAs("nsparseVector", "zsparseVector", function(from)
      as(as(from, "lsparseVector"), "zsparseVector"))


## "xsparseVector" : those with an 'x' slot (i.e., currently := not nsparse*)
setAs("xsparseVector", "dsparseVector",
      function(from)
      new("dsparseVector", x= as.double(from@x) , i= from@i, length= from@length))
setAs("xsparseVector", "isparseVector",
      function(from)
      new("isparseVector", x= as.integer(from@x), i= from@i, length= from@length))
setAs("xsparseVector", "lsparseVector",
      function(from)
      new("lsparseVector", x= as.logical(from@x), i= from@i, length= from@length))
setAs("xsparseVector", "zsparseVector",
      function(from)
      new("zsparseVector", x= as.complex(from@x), i= from@i, length= from@length))

setAs("xsparseVector", "nsparseVector",
      function(from) {
          if(any(is.na(from@x)))
              stop("cannot coerce 'NA's to \"nsparseVector\"")
          new("nsparseVector", i = from@i, length = from@length)
      })

setMethod("is.na", signature(x = "nsparseVector"),
	  function(x) new("nsparseVector", length = x@length))## all FALSE
setMethod("is.na", signature(x = "sparseVector"),
	  ## x is *not* "nsparse*" as that has own method
	  function(x) new("nsparseVector", i = x@i[is.na(x@x)], length= x@length))



sp2vec <- function(x, mode = .type.kind[substr(cl, 1,1)]) {
    ## sparseVector  ->  vector
    cl <- class(x)
    r <- vector(mode, x@length)
    r[x@i] <-
	if(cl != "nsparseVector") { # cheap test for 'has x slot'
	    if(is(x@x, mode)) x@x else as(x@x, mode)
	} else TRUE
    r
}

##' Construct new sparse vector , *dropping* zeros

##' @param class  character, the sparseVector class
##' @param x      numeric/logical/...:  the 'x' slot
##' @param i      integer: index of non-zero entries
##' @param length integer: the 'length' slot

##' @return a sparseVector, with 0-dropped 'x' (and 'i')
newSpV <- function(class, x, i, length) {
    if(length(x) == 1 && (li <- length(i)) != 1) ## recycle x :
	x <- rep.int(x, li)
    if(isTRUE(any(x0 <- x == 0))) {
	keep <- is.na(x) | !x0
	x <- x[keep]
	i <- i[keep]
    }
    new(class, x = x, i = i, length = length)
}
newSpVec <- function(class, x, prev)
    newSpV(class, x=x, i=prev@i, length=prev@length)

setAs("sparseVector", "vector", function(from) sp2vec(from))

setMethod("as.vector", signature(x = "sparseVector", mode = "missing"),
	  sp2vec)
setMethod("as.vector", signature(x = "sparseVector", mode = "character"),
	  sp2vec)

setMethod("as.numeric", "sparseVector", function(x) sp2vec(x, mode = "double"))
setMethod("as.logical", "sparseVector", function(x) sp2vec(x, mode = "logical"))

setAs("sparseVector", "numeric", function(from) sp2vec(from, mode = "double"))
setAs("sparseVector", "integer", function(from) sp2vec(from, mode = "integer"))
setAs("sparseVector", "logical", function(from) sp2vec(from, mode = "logical"))

## the "catch all remaining" method:
setAs("ANY", "sparseVector",
      function(from) as(as.vector(from), "sparseVector"))

setAs("diagonalMatrix", "sparseVector",
      function(from) {
	  kind <- .M.kind(from) ## currently only "l" and "d" --> have 'x'
	  n <- nrow(from)
          n2 <- as.double(n) * n
	  if(n2 > .Machine$integer.max) { ## double (i, length)
	      ii <- seq(1, by = n+1, length.out = n) ## 1-based indexing
	  } else { # integer ok
	      n2 <- as.integer(n2)
	      ii <- as.integer(seq(1L, by = n+1L, length.out = n))
	  }
	  new(paste0(kind, "sparseVector"),
	      length = n2, i = ii,
	      x = if(from@diag != "U") from@x else
		  rep.int(switch(kind, "d" = 1, "l" = TRUE, "i" = 1L, "z" = 1+0i), n))
	 })

setAs("sparseMatrix", "sparseVector",
      function(from) as(as(from, "TsparseMatrix"), "sparseVector"))

setAs("TsparseMatrix", "sparseVector",
      function(from) {
	  d <- dim(from)
	  n <- prod(d) # -> numeric, no integer overflow
          cld <- getClassDef(class(from))
	  kind <- .M.kind(from, cl = cld)
	  if(extends(cld, "symmetricMatrix"))
	      from <- as(from, "generalMatrix")
	  else if(extends(cld, "triangularMatrix") && from@diag == "U")
	      from <- .Call(Tsparse_diagU2N, from)
	  if(is_duplicatedT(from, di = d))
	      from <- uniqTsparse(from)
	  r <- new(paste0(kind, "sparseVector"), length = n)
	  r@i <- if(n < .Machine$integer.max) {
	      1L + from@i + d[1] * from@j
	  } else {
	      1 + from@i + as.double(d[1]) * from@j
	  }
	  if(kind != "n") ## have 'x' slot
	      r@x <- from@x
	  r
      })


##' <description>
##'
##' <details>
## Utility -- used in `dim<-` below, but also in  Matrix(.) :
##' @title sparseVector --> sparseMatrix constructor
##' @param x "sparseVector" object
##' @param nrow integer or missing, as in matrix(), see ?matrix
##' @param ncol (ditto)
##' @param byrow logical (see ?matrix)
##' @param check logical indicating if it needs to be checked that 'x' is a sparseVector
##' @return an object inheriting from "sparseMatrix"
##' @author Martin Maechler, 11 May 2007
spV2M <- function (x, nrow, ncol, byrow = FALSE, check = TRUE)
{
    cx <- class(x)
    if(check && !extends(cx, "sparseVector"))
	stop("'x' must inherit from \"sparseVector\"")
    if(!missing(ncol)) { ncol <- as.integer(ncol)
			 if(ncol < 0) stop("'ncol' must be >= 0") }
    if(!missing(nrow)) { nrow <- as.integer(nrow)
			 if(nrow < 0) stop("'nrow' must be >= 0") }
    n <- length(x)
    if(missing(nrow)) {
	nrow <- as.integer(
			   if(missing(ncol)) { ## both missing: --> (n x 1)
			       ncol <- 1L
			       n
			   } else {
			       if(n %% ncol != 0) warning("'ncol' is not a factor of length(x)")
			       as.integer(ceiling(n / ncol))
			   })
    } else if(missing(ncol)) {
        if(n %% nrow != 0) warning("'nrow' is not a factor of length(x)")
        ncol <- as.integer(ceiling(n / nrow))
    } else { ## both nrow and ncol specified
        n.n <- as.double(ncol) * nrow # no integer overflow
        if(n.n <  n) stop("nrow * ncol < length(x)", domain = NA)
        if(n.n != n) warning("nrow * ncol != length(x)", domain = NA)
    }
    ## now nrow * ncol >= n
    ##	   ~~~~~~~~~~~~~~~~
    cld <- getClassDef(cx)
    kind <- .M.kindC(cld)		# "d", "n", "l", "i", "z", ...
    has.x <- kind != "n"
    ## "careful_new()" :
    cNam <- paste0(kind, "gTMatrix")
    chngCl <- is.null(slotNames(newCl <- getClass(cNam, .Force=TRUE)))
    if(chngCl) { ## e.g. "igTMatrix" is not yet implemented
	if(substr(cNam,1,1) == "z")
	    stopifnot(sprintf("Class '%s' is not yet implemented", cNam))
	## coerce to "double":
	newCl <- getClass("dgTMatrix")
    }
    r <- new(newCl, Dim = c(nrow, ncol))
    ## now "compute"  the (i,j,x) slots given x@(i,x)
    i0 <- x@i - 1L
    if(byrow) { ## need as.integer(.) since <sparseVector> @ i can be double
	r@j <- as.integer(i0 %% ncol)
	r@i <- as.integer(i0 %/% ncol)
    } else { ## default{byrow = FALSE}
	r@i <- as.integer(i0 %% nrow)
	r@j <- as.integer(i0 %/% nrow)
    }
    if(has.x)
	r@x <- if(chngCl) as.numeric(x@x) else x@x
    r
}## {spV2M}

.sparseV2Mat <- function(from) spV2M(from, nrow=length(from), ncol=1L, check=FALSE)
setAs("sparseVector","Matrix", .sparseV2Mat)
setAs("sparseVector","sparseMatrix", .sparseV2Mat)
setAs("sparseVector","TsparseMatrix", .sparseV2Mat)
setAs("sparseVector","CsparseMatrix",
      function(from) .Call(Tsparse_to_Csparse, .sparseV2Mat(from), FALSE))

## This is very similar to the 'x = "sparseMatrix"' method in ./sparseMatrix.R:
setMethod("dim<-", signature(x = "sparseVector", value = "ANY"),
	  function(x, value) {
	      if(!is.numeric(value) || length(value) != 2)
		  stop("dim(.) value must be numeric of length 2")
	      if(length(x) != prod(value <- round(value)))
		  stop("dimensions don't match the number of cells")
	      spV2M(x, nrow=value[1], ncol=value[2])
	  })


setMethod("length", "sparseVector", function(x) x@length)

setMethod("show", signature(object = "sparseVector"),
   function(object) {
       n <- object@length
       cl <- class(object)
       cat(sprintf('sparse vector (nnz/length = %d/%.0f) of class "%s"\n',
		   length(object@i), as.double(n), cl))
       maxp <- max(1, getOption("max.print"))
       if(n <= maxp) {
	   prSpVector(object, maxp = maxp)
       } else { # n > maxp : will cut length of what we'll display :
	   ## cannot easily show head(.) & tail(.) because of "[1] .." printing of tail
	   prSpVector(object[seq_len(maxp)], maxp = maxp)
	   cat(" ............................",
	       "\n ........suppressing ", n - maxp,
	       " entries in show(); maybe adjust 'options(max.print= *)'",
	       "\n ............................\n\n", sep='')
       }
       invisible(object)
   })

prSpVector <- function(x, digits = getOption("digits"),
		    maxp = getOption("max.print"), zero.print = ".")
{
    cld <- getClassDef(cl <- class(x))
    stopifnot(extends(cld, "sparseVector"), maxp >= 1)
    if(is.logical(zero.print))
	zero.print <- if(zero.print) "0" else " "
##     kind <- .M.kindC(cld)
##     has.x <- kind != "n"
    n <- x@length
    if(n > 0) {
        if(n > maxp) { # n > maxp =: nn : will cut length of what we'll display :
            ## FIXME: very inefficient for very large maxp < n
            x <- x[seq_len(maxp)]       # need "[" to work ...
            n <- maxp
        }
        xi <- x@i
        is.n <- extends(cld, "nsparseVector")
        logi <- is.n || extends(cld, "lsparseVector")
        cx <- if(logi) rep.int("N", n) else character(n)
        cx[if(length(xi)) -xi else TRUE] <- zero.print
        cx[ xi] <- {
	    if(is.n) "|" else if(logi) c(":","|")[x@x + 1L] else
	    ## numeric (or --not yet-- complex): 'has.x' in any cases
	    format(x@x, digits = digits)
        }
        ## right = TRUE : cheap attempt to get better "." alignment
        print(cx, quote = FALSE, right = TRUE, max = maxp)
    }
    invisible(x) # TODO? in case of n > maxp, "should" return original x
}

## This is a simplified intI() {-> ./Tsparse.R } -- for sparseVector indexing:
intIv <- function(i, n)
{
    ## Purpose: translate numeric | logical index     into  1-based integer
    ## --------------------------------------------------------------------
    ## Arguments: i: index vector (numeric | logical) *OR* sparseVector
    ##		  n: array extent { ==	length(.) }
    if(missing(i))
	return(seq_len(n))
    ## else :
    cl.i <- getClass(class(i))
    if(extends(cl.i, "numeric")) {
	storage.mode(i) <- "integer"
	if(any(i < 0L)) {
	    if(any(i > 0L))
		stop("you cannot mix negative and positive indices")
	    seq_len(n)[i]
	} else {
	    if(length(i) && max(i) > n)
		stop("indexing out of range 0:",n)
	    if(any(z <- i == 0))
		i <- i[!z]
	    i
	}
    }
    else if (extends(cl.i, "logical")) {
	seq_len(n)[i]
    } else if(extends(cl.i, "nsparseVector")) {
	i@i # the indices are already there !
    } else if(extends(cl.i, "lsparseVector")) {
	i@i[i@x] # "drop0", i.e. FALSE; NAs ok
    } else if (extends(cl.i, "sparseVector")) { ## 'i'sparse, 'd'sparse	 (etc)
	as.integer(i@x[i@i])
    }
    else
        stop("index must be numeric, logical or sparseVector for indexing sparseVectors")
} ## intIv()


setMethod("[", signature(x = "sparseVector", i = "index"),
	  function (x, i, j, ..., drop) {
	      cld <- getClassDef(class(x))
	      has.x <- !extends(cld, "nsparseVector")
	      n <- x@length
	      ii <- intIv(i, n)
	      anyDup <- any(iDup <- duplicated(ii))
	      m <- match(x@i, ii, nomatch = 0)
	      sel <- m > 0L
	      x@length <- length(ii)
	      x@i <- m[sel]
	      if(anyDup) {
		  i.i <- match(ii[iDup], ii)
		  jm <- lapply(i.i, function(.) which(. == m))
		  sel <- c(which(sel), unlist(jm))
		  x@i <- c(x@i, rep.int(which(iDup), sapply(jm, length)))
	      }
	      if (has.x)
		  x@x <- x@x[sel]
	      x
	  })

setMethod("[", signature(x = "sparseVector", i = "lsparseVector"),
	  function (x, i, j, ..., drop) x[sort.int(i@i[i@x])])
setMethod("[", signature(x = "sparseVector", i = "nsparseVector"),
	  function (x, i, j, ..., drop) x[sort.int(i@i)])

##--- Something else:  Allow    v[ <sparseVector> ] -- exactly similarly:
if(FALSE) { ## R_FIXME: Not working, as internal "[" only dispatches on 1st argument
setMethod("[", signature(x = "atomicVector", i = "lsparseVector"),
	  function (x, i, j, ..., drop) x[sort.int(i@i[i@x])])
setMethod("[", signature(x = "atomicVector", i = "nsparseVector"),
	  function (x, i, j, ..., drop) x[sort.int(i@i)])
}

##' Implement   x[i] <- value

##' @param x  a "sparseVector"
##' @param i  an "index" (integer, logical, ..)
##' @param value

##' @return  a "sparseVector" of the same length as 'x'
## This is much analogous to replTmat in ./Tsparse.R:
replSPvec <- function (x, i, value)
{
    n <- x@length
    ii <- intIv(i, n)
    lenRepl <- length(ii)
    lenV <- length(value)
    if(lenV == 0) {
	if(lenRepl != 0)
	    stop("nothing to replace with")
	else return(x)
    }
    ## else: lenV := length(value) > 0
    if(lenRepl %% lenV != 0)
	stop("number of items to replace is not a multiple of replacement length")
    if(anyDuplicated(ii)) { ## multiple *replacement* indices: last one wins
	## TODO: in R 2.6.0 use	 duplicate(*, fromLast=TRUE)
	ir <- lenRepl:1
	keep <- match(ii, ii[ir]) == ir
	ii <- ii[keep]
	lenV <- length(value <- rep(value, length = lenRepl)[keep])
	lenRepl <- length(ii)
    }

    cld <- getClassDef(class(x))
    has.x <- !extends(cld, "nsparseVector")
    m <- match(x@i, ii, nomatch = 0)
    sel <- m > 0L

    ## the simplest case
    if(all0(value)) { ## just drop the non-zero entries
	if(any(sel)) { ## non-zero there
	    x@i <- x@i[!sel]
	    if(has.x)
		x@x <- x@x[!sel]
	}
	return(x)

    }
    ## else --	some( value != 0 ) --
    if(lenV > lenRepl)
	stop("too many replacement values")
    else if(lenV < lenRepl)
	value <- rep(value, length = lenRepl)
    ## now:  length(value) == lenRepl

    v0 <- is0(value)
    ## value[1:lenRepl]:  which are structural 0 now, which not?

    if(any(sel)) {
	## indices of non-zero entries -- WRT to subvector
	iN0 <- m[sel] ## == match(x@i[sel], ii)

	## 1a) replace those that are already non-zero with new val.
	vN0 <- !v0[iN0]
	if(any(vN0) && has.x)
	    x@x[sel][vN0] <- value[iN0[vN0]]

	## 1b) replace non-zeros with 0 --> drop entries
	if(any(!vN0)) {
	    i <- which(sel)[!vN0]
	    if(has.x)
		x@x <- x@x[-i]
	    x@i <- x@i[-i]
	}
	iI0 <- if(length(iN0) < lenRepl)
	    seq_len(lenRepl)[-iN0]
    } else iI0 <- seq_len(lenRepl)

    if(length(iI0) && any(vN0 <- !v0[iI0])) {
	## 2) add those that were structural 0 (where value != 0)
	ij0 <- iI0[vN0]
	x@i <- c(x@i, ii[ij0])
	if(has.x)
	    x@x <- c(x@x, value[ij0])
    }
    x
}

setReplaceMethod("[", signature(x = "sparseVector", i = "index", j = "missing",
				value = "replValue"),
		 replSPvec)

setReplaceMethod("[", signature(x = "sparseVector",
                                i = "sparseVector", j = "missing",
				value = "replValue"),
                 ## BTW, the important case: 'i' a *logical* sparseVector
		 replSPvec)

## Something else:  Also allow	  x[ <sparseVector> ] <- v

if(FALSE) { ## R_FIXME: Not working, as internal "[<-" only dispatches on 1st argument
## Now "the work is done" inside  intIv() :
setReplaceMethod("[", signature(x = "atomicVector",
				i = "sparseVector", j = "missing",
				value = "replValue"),
		 function (x, i, value)
		 callGeneric(x, i = intIv(i, x@length), value=value))
}

## a "method" for c(<(sparse)Vector>, <(sparse)Vector>):
## FIXME: This is not exported, nor used (nor documented)
c2v <- function(x, y) {
    ## these as(., "sp..V..") check input implicitly:
    cx <- class(x <- as(x, "sparseVector"))
    cy <- class(y <- as(y, "sparseVector"))
    if(cx != cy) { ## find "common" class; result does have 'x' slot
        cxy <- c(cx,cy)
        commType <- {
            if(all(cxy %in% c("nsparseVector", "lsparseVector")))
                "lsparseVector"
            else { # ==> "numeric" ("integer") or "complex"
                xslot1 <- function(u, cl.u)
                    if(cl.u != "nsparseVector") u@x[1] else TRUE
                switch(typeof(xslot1(x, cx) + xslot1(y, cy)),
                       ## "integer", "double", or "complex"
                       "integer" = "isparseVector",
                       "double" = "dsparseVector",
                       "complex" = "zsparseVector")
            }
        }
        if(cx != commType) x <- as(x, commType)
        if(cy != commType) y <- as(y, commType)
        cx <- commType
    }
    ## now *have* common type -- transform 'x' into result:
    nx <- x@length
    x@length <- nx + y@length
    x@i <- c(x@i, nx + y@i)
    if(cx != "nsparseVector")
        x@x <- c(x@x, y@x)
    x
}

all.equal.sparseV <- function(target, current, ...)
{
    if(!is(target, "sparseVector") || !is(current, "sparseVector")) {
	return(paste("target is ", data.class(target), ", current is ",
		     data.class(current), sep = ""))
    }
    lt <- length(target)
    lc <- length(current)
    if(lt != lc) {
	return(paste("sparseVector", ": lengths (", lt, ", ", lc, ") differ",
		     sep = ""))
    }

    t.has.x <- class(target)  != "nsparseVector"
    c.has.x <- class(current) != "nsparseVector"
    nz.t <- length(i.t <- target @i)
    nz.c <- length(i.c <- current@i)
    t.x <- if(t.has.x)	target@x else rep.int(TRUE, nz.t)
    c.x <- if(c.has.x) current@x else rep.int(TRUE, nz.c)
    if(nz.t != nz.c || any(i.t != i.c)) { ## "work" if indices are not the same
	i1.c <- setdiff(i.t, i.c)# those in i.t, not yet in i.c
	i1.t <- setdiff(i.c, i.t)
	if((n1t <- length(i1.t))) {
	    target@i <- i.t <- c(i.t, i1.t)
	    t.x <- c(t.x, rep.int(if(t.has.x) 0 else 0L, n1t))
	}
	if((n1c <- length(i1.c))) {
	    current@i <- i.c <- c(i.c, i1.c)
	    c.x <- c(c.x, rep.int(if(c.has.x) 0 else 0L, n1c))
	}
    }
    if(is.unsorted(i.t)) {  ## method="quick" {"radix" not ok for large range}
	ii <- sort.list(i.t, method = "quick", na.last=NA)
	target@i <- i.t <- i.t[ii]
	t.x <- t.x[ii]
    }
    if(is.unsorted(i.c)) {
	ii <- sort.list(i.c, method = "quick", na.last=NA)
	current@i <- i.c <- i.c[ii]
	c.x <- c.x[ii]
    }

    ## Now, we have extended both target and current
    ## *and* have sorted the respective i-slot, the i-slots should match!
    stopifnot(all(i.c == i.t))

    all.equal.numeric(c.x, t.x, ...)
} ## all.equal.sparseV


## For these, we remain sparse:
setMethod("all.equal", c(target = "sparseVector", current = "sparseVector"),
	  all.equal.sparseV)
setMethod("all.equal", c(target = "sparseVector", current = "sparseMatrix"),
	  function(target, current, ...)
	  all.equal.sparseV(target, as(current, "sparseVector"), ...))
setMethod("all.equal", c(target = "sparseMatrix", current = "sparseVector"),
	  function(target, current, ...)
	  all.equal.sparseV(as(target, "sparseVector"), current, ...))
## For the others, where one is "dense", "go to" dense rather now than later:
setMethod("all.equal", c(target = "ANY", current = "sparseVector"),
	  function(target, current, ...)
	  all.equal(target, as.vector(current), ...))
setMethod("all.equal", c(target = "sparseVector", current = "ANY"),
	  function(target, current, ...)
	  all.equal(as.vector(target), current, ...))


### rep(x, ...) -- rep() is primitive with internal default method with these args:
### -----------
### till R 2.3.1, it had  rep.default()  which we use as 'model' here.

repSpV <- function(x, times) {
    ## == rep.int(<sparseVector>, times)"
    times <- as.integer(times)# truncating as rep.default()
    n <- x@length
    has.x <- substr(class(x), 1,1) != "n" ## fast, but hackish
    ## just assign new correct slots:
    if(times <= 1) { ## be quick for {0, 1} times
        if(times < 0) stop("'times >= 0' is required")
        if(times == 0) {
            x@length <- 0L
            x@i <- integer(0)
            if(has.x) x@x <- rep.int(x@x, 0)
        }
        return(x)
    }
    n. <- as.double(n)
    if(n. * times >= .Machine$integer.max)
        n <- n. # so won't have overflow in subsequent multiplys
    x@length <- n * times
    x@i <- rep.int(x@i, times) + n * rep(0:(times-1L), each=length(x@i))
    ## := outer(x@i, 0:(times-1) * n, "+")   but a bit faster
    if(has.x) x@x <- rep.int(x@x, times)
    x
}

setMethod("rep", "sparseVector",
	  function(x, times, length.out, each, ...) {
	      if (length(x) == 0)
		  return(if(missing(length.out)) x else x[seq_len(length.out)])
	      if (!missing(each)) {
		  tm <- rep.int(each, length(x))
		  x <- rep(x, tm) # "recursively"
		  if(missing(length.out) && missing(times))
		      return(x)
	      } ## else :
	      if (!missing(length.out)) # takes precedence over times
		  times <- ceiling(length.out/length(x))
	      r <- repSpV(x, times)
	      if (!missing(length.out) && length(r) != length.out)
                  ## FIXME: for large length.out > maxInt (and very sparse r),
                  ##       the following fails, *unnecessarily*
                  ## --> need a  subset(x, i) function which works with abIndex 'i'
		  return(r[if(length.out > 0) 1:length.out else integer(0)])
	      return(r)
	  })


### Group Methods (!)
## "Ops" : ["Arith", "Compare", "Logic"]:  ---> in ./Ops.R
##                                                 -------

## "Summary" group : "max"   "min"   "range" "prod"  "sum"   "any"   "all"

setMethod("Summary", signature(x = "nsparseVector", na.rm = "ANY"),
	  function(x, ..., na.rm) { ## no 'x' slot, no NA's ..
	      n <- x@length
	      l.x <- length(x@i)
	      if(l.x == n)
		  callGeneric(rep.int(TRUE, n), ..., na.rm = na.rm)
	      else ## l.x < n :	 has some FALSE entries
		  switch(.Generic,
			 "prod" = 0,
			 "min"	= 0L,
			 "all" = FALSE,
			 "any" = l.x > 0,
			 "sum" = l.x,
			 "max" = as.integer(l.x > 0),
			 "range" = c(0L, as.integer(l.x > 0)))
	  })

## The "other" "sparseVector"s ("d", "l", "i" ..): all have an	'x' slot :
setMethod("Summary", signature(x = "sparseVector", na.rm = "ANY"),
	  function(x, ..., na.rm) {
	      n <- x@length
	      l.x <- length(x@x)
	      if(l.x == n) ## fully non-zero (and "general") - very rare but quick
		  callGeneric(x@x, ..., na.rm = na.rm)
	      else if(.Generic != "prod") {
		  logicF <- .Generic %in% c("any","all")
		  ## we rely on	 <generic>(x, NULL, y, ..) :==	<generic>(x, y, ..):
		  callGeneric(x@x, if(logicF) FALSE else 0, ..., na.rm = na.rm)
	      }
	      else { ## prod()
		  if(any(is.na(x@x))) NaN else 0
	      }
	  })


setMethod("solve", signature(a = "Matrix", b = "sparseVector"),
	  function(a, b, ...) callGeneric(a, as(b, "sparseMatrix")))

## the 'i' slot is 1-based *and* has no NA's:

setMethod("which", "nsparseVector", function(x, arr.ind) sort.int(x@i, method="quick"))
setMethod("which", "lsparseVector",
	  function(x, arr.ind) sort.int(x@i[is1(x@x)], method="quick"))
## and *error* for "dsparseVector", "i*", ...
