\name{EventPredExample}
\alias{EventPredExample}
\title{
Simulation-based event prediction in trials with an event-driven design (time-to- event endpoint)
}
\description{
Consider a Phase III trial in an oncology population that is being conducted to investigate the efficacy of an experimental treatment versus a control (best supportive care). The primary efficacy analysis evaluates the survival benefit of this treatment. Blinded data, including patient enrollment, event and patient dropout information, will be reviewed periodically throughout the trial to predict the number of events at pre-defined time points in the future.  

Assume that the current snapshot of data required for event prediction is included in the \cr \code{\link{EventPredData}} data set. This data set contains blinded patient enrollment, event and patient dropout information on 150 patients at 12 months after the study start. The blinded data, along with relevant prior information, will support the calculation of the predicted number of events  and 95\% predictive intervals up to the 24-month milestone.

Prior information needs to be provided for the event hazard rate, patient dropout hazard rate and patient enrollment rate. These parameters are assumed to follow gamma distributions. The shape and rate parameters of the gamma distributions could be specified directly or by invoking the \code{\link{EventPredPriorDistribution}} function. This function facilitates the process of prior specifications by allowing the user to provide the expected value of a parameter of interest, e.g., the event hazard rate, as well as the uncertainty parameter. The latter quantifies the amount of variability around the expected value, e.g., the uncertainty parameter of 0.1 corresponds to a low standard deviation and thus defines a high-confidence prior distribution. The expected values of the event and patient enrollment hazard rates will be computed using the assumptions that the expected median event and dropout times are equal to 15 and 80 months, respectively (the expected median dropout time corresponds to an annual dropout rate of 10\%). The expected patient enrollment rate will be set to 35 patients per month. A common uncertainty parameter of 0.3 will be used for the the event hazard rate, patient dropout hazard rate and patient enrollment rate. This uncertainty parameter will define medium-confidence prior distributions for the three parameters of interest.

A summary of event predictions can be generated by calling the \code{\link{EventPred}} function. The calculations will be performed using 1,000 simulation runs. A list of all required parameters (\code{parameters}) needs to be created as shown below and then passed to this function. A simulation report can be generated using the \code{\link{GenerateReport}} function and a graphical user interface can be launched using the \code{\link{EventPredApp}} function.
}
\arguments{No arguments}
\value{No return value}
\seealso{
  \code{\link{EventPred}}
}
\examples{
# List of all parameters
parameters = list()

# Load the built-in data set with the patient enrollment, event and dropout 
# information (EventPredData)

parameters$data_set = EventPredData

# Future time points for computing event predictions
parameters$time_points = seq(from = 12, to = 24, by = 1)

# Prior distribution for the event hazard rate based on the 
# expected median time of 15 months and the uncertainty parameter of 0.3 
parameters$event_prior_distribution = 
    EventPredPriorDistribution(expected = log(2) / 15, uncertainty = 0.3)

# Prior distribution for the patient dropout hazard rate based on the 
# expected median time of 80 months and the uncertainty parameter of 0.3
parameters$dropout_prior_distribution = 
    EventPredPriorDistribution(expected = log(2) / 80, uncertainty = 0.3)

# Prior distribution for the patient enrollment rate based on the
# expected enrollment rate of 35 patients per month and the uncertainty 
# parameter of 0.3 
parameters$enrollment_prior_distribution = 
    EventPredPriorDistribution(expected = 35, uncertainty = 0.3)

# Number of simulations, you should prefer more
parameters$nsims = 100

# Forecast the number of events at the pre-defined time points
results = EventPred(parameters)

# Generate a simulation report (remove tempfile)
GenerateReport(results, 
              tempfile("EventPred.docx", fileext=".docx"))

}