% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ms_outlook.R
\docType{class}
\name{ms_outlook}
\alias{ms_outlook}
\title{Outlook mail client}
\format{
An R6 object of class \code{ms_outlook}, inheriting from \code{ms_outlook_object}, which in turn inherits from \code{ms_object}.
}
\description{
Class representing a user's Outlook email account.
}
\section{Fields}{

\itemize{
\item \code{token}: The token used to authenticate with the Graph host.
\item \code{tenant}: The Azure Active Directory tenant for the email account.
\item \code{type}: always "Outlook account" for an Outlook email account.
\item \code{properties}: The item properties (metadata).
}
}

\section{Methods}{

\itemize{
\item \code{new(...)}: Initialize a new object. Do not call this directly; see 'Initialization' below.
\item \code{update(...)}: Update the account's properties (metadata) in Microsoft Graph.
\item \code{do_operation(...)}: Carry out an arbitrary operation on the account.
\item \code{sync_fields()}: Synchronise the R object with the account metadata in Microsoft Graph.
\item \code{create_email(...)}: Creates a new email in the Drafts folder, optionally sending it as well. See 'Creating and sending emails'.
\item \code{list_inbox_emails(...)}: List the emails in the Inbox folder. See 'Listing emails'.
\item \verb{get_inbox(),get_drafts(),get_sent_items(),get_deleted_items()}: Gets the special folder of that name. These folders are created by Outlook and exist in every email account.
\item \code{list_folders(filter=NULL, n=Inf)}: List all folders in this account.
\item \code{get_folder(folder_name, folder_id)}: Get a folder, either by the name or ID.
\item \code{create_folder(folder_name)}: Create a new folder.
\item \code{delete_folder(folder_name, folder_id, confirm=TRUE)}: Delete a folder. By default, ask for confirmation first. Note that special folders cannot be deleted.
}
}

\section{Initialization}{

Creating new objects of this class should be done via the \code{get_personal_outlook()} or \code{get_business_outlook()} functions, or the \code{get_outlook} method of the \code{\link{az_user}} class. Calling the \code{new()} method for this class only constructs the R object; it does not call the Microsoft Graph API to retrieve the account information.
}

\section{Creating and sending emails}{

To create a new email, call the \code{create_email()} method. The default behaviour is to create a new draft email in the Drafts folder, which can then be edited further to add attachments, recipients etc; or the email can be sent immediately.

The \code{create_email()} method has the following signature:\preformatted{create_email(body = "", content_type = c("text", "html"), subject = "",
             to = NULL, cc = NULL, bcc = NULL, reply_to = NULL, send_now = FALSE)
}
\itemize{
\item \code{body}: The body of the message. This should be a string or vector of strings, which will be pasted together with newlines as separators. You can also supply a message object as created by the blastula or emayili packages---see the examples below.
\item \code{content_type}: The format of the body, either "text" (the default) or HTML.
\item \code{subject}: The subject of the message.
\item \verb{to,cc,bcc,reply_to}: These should be lists of email addresses, in standard "user@host" format. You can also supply objects of class \code{\link[AzureGraph:az_user]{AzureGraph::az_user}} representing user accounts in Azure Active Directory.
\item \code{send_now}: Whether the email should be sent immediately, or saved as a draft. You can send a draft email later with its \code{send()} method.
}

This returns an object of class \code{\link{ms_outlook_email}}, which has methods for making further edits, attaching files, replying, forwarding, and (re-)sending.

You can also supply message objects as created by the blastula and emayili packages in the \code{body} argument. Note that blastula objects include attachments (if any), and emayili objects include attachments, recipients, and subject line; the corresponding arguments to \code{create_email()} will not be used in this case.
}

\section{Listing emails}{

To list the emails in the Inbox, call the \code{list_emails()} method. This returns a list of objects of class \code{\link{ms_outlook_email}}, and has the following signature:\preformatted{list_emails(by = "received desc", n = 100, pagesize = 10)
}
\itemize{
\item \code{by}: The sorting order of the message list. The possible fields are "received" (received date, the default), "from" and "subject". To sort in descending order, add a " desc". You can specify multiple sorting fields, with later fields used to break ties in earlier ones. The last sorting field is always "received desc" unless it appears earlier.
\item \verb{filter, n}: See below.
\item \code{pagesize}: The number of emails per page. You can change this to a larger number to increase throughput, at the risk of running into timeouts.
}
}

\section{List methods generally}{

All \verb{list_*} methods have \code{filter} and \code{n} arguments to limit the number of results. The former should be an \href{https://docs.microsoft.com/en-us/graph/query-parameters#filter-parameter}{OData expression} as a string to filter the result set on. The latter should be a number setting the maximum number of (filtered) results to return. The default values are \code{filter=NULL} and \code{n=100} for listing emails, and \code{n=Inf} for listing folders. If \code{n=NULL}, the \code{ms_graph_pager} iterator object is returned instead to allow manual iteration over the results.

Support in the underlying Graph API for OData queries is patchy. Not all endpoints that return lists of objects support filtering, and if they do, they may not allow all of the defined operators. If your filtering expression results in an error, you can carry out the operation without filtering and then filter the results on the client side.
}

\examples{
\dontrun{

outl <- get_personal_outlook()

##
## listing emails and folders
##

# the default: 100 most recent messages in the inbox
outl$list_emails()

# sorted by subject, then by most recent received date
outl$list_emails(by="subject")

# retrieve a specific email:
# note the Outlook ID is NOT the same as the Internet message-id
email_id <- outl$list_emails()[[1]]$properties$id
outl$get_email(email_id)

# all folders in this account (including nested folders)
outl$list_folders()

# draft (unsent) emails
dr <- outl$get_drafts()
dr$list_emails()

# sent emails
sent <- outl$get_sent_items()
sent$list_emails()

##
## creating/sending emails
##

# a simple text email with just a body (can't be sent)
outl$create_email("Hello from R")

# HTML-formatted email with all necessary fields, sent immediately
outl$create_email("<emph>Emphatic hello</emph> from R",
    content_type="html",
    to="user@example.com",
    subject="example email",
    send_now=TRUE)

# you can also create a blank email object and call its methods to add content
outl$create_email()$
    set_body("<emph>Emphatic hello</emph> from R", content_type="html")$
    set_recipients(to="user@example.com")$
    set_subject("example email")$
    add_attachment("mydocument.docx")$
    send()

# using blastula to create a HTML email with Markdown
bl_msg <- blastula::compose_email(md(
"
## Hello!

This is an email message that was generated by the blastula package.

We can use **Markdown** formatting with the `md()` function.

Cheers,

The blastula team
"),
    footer=md("Sent via Microsoft365R"))
outl$create_email(bl_msg, subject="example blastula email")


# using emayili to create an email with attachments
ey_email <- emayili::envelope(
    text="Hello from emayili",
    to="user@example.com",
    subject="example emayili email") \%>\%
    emayili::attachment("mydocument.docx") \%>\%
    emayili::attachment("mydata.xlsx")
outl$create_email(ey_email)

}
}
\seealso{
\code{\link{ms_outlook_folder}}, \code{\link{ms_outlook_email}}

\href{https://docs.microsoft.com/en-us/graph/overview}{Microsoft Graph overview},
\href{https://docs.microsoft.com/en-us/graph/api/resources/mail-api-overview?view=graph-rest-1.0}{Outlook API reference}
}
