% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mix_utils.R
\name{pseMer}
\alias{pseMer}
\title{PSE/JND from GLMM Estimates Using Bootstrap Method}
\usage{
pseMer(
  mer.obj,
  B = 200,
  FUN = NULL,
  alpha = 0.05,
  ci.type = c("norm", "basic", "perc"),
  beep = F
)
}
\arguments{
\item{mer.obj}{an object of class \code{\linkS4class{merMod}}.}

\item{B}{integer. Number of bootstrap samples.}

\item{FUN}{an optional, custom made function to specify the required parameters to be estimated.
If NULL, \code{pseMer} estimates PSE and 50\%JND of a univariable GLMM with a single intercept and slope.}

\item{alpha}{significance level of the confidence intervals. Default is 0.05 (95\% confidence interval).}

\item{ci.type}{vector of character strings representing the type of intervals required. The value 
should be any subset of the values accepted by \code{\link[boot]{boot.ci}}: c("norm","basic", "stud", "perc", "bca"). 
Specify "all" for all five types of intervals. "perc" should be always included for the summary table.}

\item{beep}{logical. If TRUE, a "ping" sound alerts that the simulation is complete. Default is FALSE.}
}
\value{
\code{pseMer} returns a list of length 3 including a summary table (estimate,
inferior and superior bounds of the confidence interval), the output of \code{\link[lme4]{bootMer}}, and that of 
\code{\link[boot]{boot.ci}}, for further analyses. Confidence intervals in the summary table are
based on the percentile method.
}
\description{
Estimates the Point of Subjective Equivalence (PSE), the Just Noticeable
Difference (JND) and the related Standard Errors by means of Bootstrap Method, 
given an object of class \code{\link[lme4]{merMod}}.
}
\details{
\code{pseMer} estimates PSE and JND (and additional user defined parameters) from a 
fitted GLMM model (class \code{\linkS4class{merMod}}).
}
\note{
A first custom function was written in 2012 for the non-CRAN package MERpsychophisics,
based on the algorithm in Moscatelli et al. (2012). The current function is a wrapper
of function \code{\link[lme4]{bootMer}} and \code{\link[boot]{boot.ci}}. 

Increasing the number of bootstrap samples (\code{B}) makes the estimate more reliable. 
However, this will also increase the duration of the computation.
}
\examples{
library(lme4)
#example 1: univariable GLMM
mod.uni = glmer(formula = cbind(Longer, Total - Longer) ~ X + (1 | Subject),
family = binomial(link = "probit"), data = simul_data)
\dontshow{BootEstim.1 <- pseMer(mod.uni, B = 5, ci.type = c("perc"))}
\donttest{BootEstim.uni <- pseMer(mod.uni, B = 100, ci.type = c("perc"))}

#example 2: specify custom parameters for multivariable model
mod.multi <- glmer(cbind(faster, slower) ~ speed * vibration + (1 + speed| subject), 
family = binomial(link = "probit"), data = vibro_exp3)
              
fun2mod = function(mer.obj){
#allocate space: 4 parameters (jnd_A, jnd_B, pse_A, pse_B)
jndpse = vector(mode = "numeric", length = 4)
names(jndpse) = c("pse_0", "pse_32","jnd_0", "jnd_32")
jndpse[1] = -fixef(mer.obj)[1]/fixef(mer.obj)[2] #pse_0
jndpse[2] = -(fixef(mer.obj)[1]+fixef(mer.obj)[3])/(fixef(mer.obj)[2]+ fixef(mer.obj)[4]) #pse_0
jndpse[3] = qnorm(0.75)/fixef(mer.obj)[2] #jnd_0
jndpse[4] = qnorm(0.75)/(fixef(mer.obj)[2]+ fixef(mer.obj)[4]) #jnd_32
return(jndpse)
}
 
\donttest{BootEstim.multi = pseMer(mod.multi, B = 100, FUN = fun2mod)}

}
\references{
Moscatelli, A., Mezzetti, M., & Lacquaniti, F. (2012). Modeling psychophysical data 
at the population-level: The generalized linear mixed model. 
Journal of Vision, 12(11):26, 1-17. doi:10.1167/12.11.26

Bates, D., Mächler, M., Bolker, B., & Walker, S. (2015). Fitting Linear Mixed-Effects 
Models Using lme4. Journal of Statistical Software, 67(1), 51. https://doi.org/10.18637/jss.v067.i01
}
\seealso{
\code{\link[lme4]{bootMer}} and \code{\link[boot]{boot.ci}} for estimation of confidence intervals with the bootstrap method. 
\code{\link{MixDelta}} for confidence intervals with delta method.
}
\keyword{Bootstrap}
\keyword{GLMM}
