% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/example_combo3.R
\name{example-combo3}
\alias{example-combo3}
\title{Three-drug combination example}
\description{
Example using a combination of two experimental drugs, with EXNEX
and differential discounting.
}
\details{
This dataset involves a hypothetical dose-escalation study of combination
therapy with three treatment components. From two previous studies
\code{HistAgent1} and \code{HistAgent2}, historical data is available on each
of the treatments as single-agents, as well as two of the two-way
combinations. However, due to a difference in treatment schedule between the
\code{Combo} study and the historical studies, a stratification (through \code{stratum_id})
is made between the groups to allow differential discounting of the
alternate-schedule data. The association is as below.

\tabular{cc}{
 group_id (j):  \tab stratum_id (s_j): \cr
 Combo (1)      \tab BID (1)         \cr
 HistAgent1 (2) \tab QD (2)          \cr
 HistAgent2 (3) \tab QD (2)
}

For additional robustness, EXNEX priors are used for all
group-level treatment components while not for the interaction
parameters. This is to limit the amount of borrowing in case of
significant heterogeneity across groups.

The complete model is as follows. As a function of doses \eqn{d_1,d_2,d_3}, the
DLT rate in group \eqn{j} is, for \eqn{j = 1,\ldots,3},
\deqn{\mbox{logit}\, \pi_j(d_1,d_2,d_3) = \mbox{logit}\Bigl( 1 - \prod_{i=1}^3 (1-\pi_{ij}(d_i))\Bigr) + \eta_{j}^{(12)}\frac{d_1}{d_1^*}\frac{d_2}{d_2^*} + \eta_{j}^{(13)}\frac{d_1}{d_1^*}\frac{d_3}{d_3^*} + \eta_{j}^{(23)}\frac{d_2}{d_2^*}\frac{d_3}{d_3^*} + \eta_{j}^{(123)}\frac{d_1}{d_1^*}\frac{d_2}{d_2^*}\frac{d_3}{d_3^*}.}

In group \eqn{j} each treatment component \eqn{i} toxicity is modeled with
logistic regression,
\deqn{\mbox{logit}\, \pi_{ij}(d_i) = \log\, \alpha_{ij} + \beta_{ij} \, \log\, \Bigl(\frac{d_i}{d_i^*}\Bigr).}
The intercept and log-slope parameters \eqn{\boldsymbol\theta_{ij} = (\log\, \alpha_{ij}, \log\, \beta_{ij})}
are are given an EXNEX prior
\deqn{\boldsymbol\theta_{ij} \sim p_{ij} \mbox{BVN}(\boldsymbol\mu_i, \boldsymbol\Sigma_{ij}) + (1-p_{ij}) \mbox{BVN}(\boldsymbol m_{ij}, \boldsymbol S_{ij}),}
where the exchangeability weights are all \eqn{p_{ij} = 0.9}.
The NEX parameters are set to \eqn{\boldsymbol m_{ij} = (\mbox{logit}(1/3), \log\, 1)},
\eqn{\boldsymbol S_{ij} = \mbox{diag}(2^2, 1^2)} for all components \eqn{i=1,2,3} and
groups \eqn{j = 1,2,3}, and the EX parameters are modeled hierarchically. The
mean of the exchangeable part has the distribution
\deqn{\boldsymbol\mu_i = (\mu_{\alpha i}, \mu_{\beta i}) \sim \mbox{BVN}(\boldsymbol m_i, \boldsymbol S_i),}
with \eqn{\boldsymbol m_i = (\mbox{logit}(1/3), \log 1)} and
\eqn{\boldsymbol S_i  = \mbox{diag}(2^2, 1^2)} for each component \eqn{i = 1,2,3}.
For differentially discounting data from each schedule (QD and BID), the
covariance parameters for the exchangeable part
\deqn{\Sigma_{ij} = \left( \begin{array}{cc}
\tau^2_{\alpha s_j i} & \rho_i \tau_{\alpha s_j i} \tau_{\beta s_j i}\\
\rho_i \tau_{\alpha s_j i} \tau_{\beta s_j i} & \tau^2_{\beta s_j i}
\end{array} \right).}
are allowed to vary across groups \eqn{j} depending on their mapping
to strata \eqn{s(j)} as described above. For stratum \eqn{s=1} (\code{BID},
which contains only the group \eqn{j = 1} (\code{Combo})), the standard
deviations are modeled as
\deqn{\tau_{\alpha 1 i} \sim \mbox{Log-Normal}(\log\,0.25, (\log 4 / 1.96)^2)}
\deqn{\tau_{\beta 1 i} \sim \mbox{Log-Normal}(\log\,0.125, (\log 4 / 1.96)^2).}
Whereas in stratum \eqn{s=2} (\code{QD}, which contains the historical groups
\eqn{j=2,3} (\code{HistData1}, \code{HistData2})), the standard deviations are
\deqn{\tau_{\alpha 2 i} \sim \mbox{Log-Normal}(\log\,0.5, (\log 4 / 1.96)^2)}
\deqn{\tau_{\beta 2 i} \sim \mbox{Log-Normal}(\log\,0.25, (\log 4 / 1.96)^2).}

For all interaction parameters \eqn{\eta_{j}^{(12)}}, \eqn{\eta_{j}^{(13)}},
\eqn{\eta_{j}^{(23)}}, and \eqn{\eta_{j}^{(123)}} (\eqn{j = 1,2,3}), the following
prior is assumed:
\deqn{\eta_{j}^{(\cdot)} \sim  \mbox{N}(\mu_{\eta}^{(\cdot)},{\tau_{\eta s_j}^{(\cdot)}}^2).}
The exchangeability weights are \eqn{p_{\eta j}^{(\cdot)} = 0.9} for all
parameters with EXNEX. Here, for each \eqn{\mu_{\eta}^{(12)}}, \eqn{\mu_{\eta}^{(13)}},
\eqn{\mu_{\eta}^{(23)}}, and \eqn{\mu_{\eta}^{(123)}}, we take
\deqn{\mu_{\eta}^{(\cdot)} \sim \mbox{N}(0, 1/2),}
and for each \eqn{\tau_{\eta s}^{(12)}}, \eqn{\tau_{\eta s}^{(13)}},
\eqn{\tau_{\eta s}^{(23)}}, and \eqn{\tau_{\eta s}^{(123)}},
\deqn{\tau_{\eta s}^{(\cdot)} \sim \mbox{Log-Normal}(\log(0.25), (\log 2 / 1.96)^2),}
for both strata \eqn{s = 1,2}.

Below is the syntax for specifying this model in \code{blrm_exnex}.
}
\examples{
## Setting up dummy sampling for fast execution of example
## Please use 4 chains and 100x more warmup & iter in practice
.user_mc_options <- options(OncoBayes2.MC.warmup=10, OncoBayes2.MC.iter=20, OncoBayes2.MC.chains=1,
                            OncoBayes2.MC.save_warmup=FALSE)

## example combo3

library(abind)

dref <- c(500, 500, 1000)
num_comp <- 3
num_inter <- choose(3,2) + 1
num_strata <- nlevels(hist_combo3$stratum_id)
num_groups <- nlevels(hist_combo3$group_id)

blrmfit <- blrm_exnex(
  cbind(num_toxicities, num_patients-num_toxicities) ~
    1 + I(log(drug_A/dref[1])) |
    1 + I(log(drug_B/dref[2])) |
    1 + I(log(drug_C/dref[3])) |
    0
  + I(drug_A/dref[1] * drug_B/dref[2])
  + I(drug_A/dref[1] * drug_C/dref[3])
  + I(drug_B/dref[2] * drug_C/dref[3])
  + I(drug_A/dref[1] * drug_B/dref[2] * drug_C/dref[3]) |
    stratum_id/group_id,
  data = hist_combo3,
  prior_EX_mu_mean_comp = matrix(c(logit(1/3), 0), nrow = num_comp, ncol = 2, TRUE),
  prior_EX_mu_sd_comp = matrix(c(2, 1), nrow = num_comp, ncol = 2, TRUE),
  prior_EX_tau_mean_comp = abind(matrix(log(  c(0.25, 0.125)), nrow = num_comp, ncol = 2, TRUE),
                                 matrix(log(2*c(0.25, 0.125)), nrow = num_comp, ncol = 2, TRUE),
                                 along = 0),
  prior_EX_tau_sd_comp = abind(matrix(log(4) / 1.96, nrow = num_comp, ncol = 2, TRUE),
                               matrix(log(4) / 1.96, nrow = num_comp, ncol = 2, TRUE),
                               along = 0),
  prior_EX_mu_mean_inter = rep(0, num_inter),
  prior_EX_mu_sd_inter = rep(sqrt(2) / 2, num_inter),
  prior_EX_tau_mean_inter = matrix(log(0.25), nrow = num_strata, ncol = num_inter),
  prior_EX_tau_sd_inter = matrix(log(2) / 1.96, nrow = num_strata, ncol = num_inter),
  prior_EX_prob_comp = matrix(0.9, nrow = num_groups, ncol = num_comp),
  prior_EX_prob_inter = matrix(1.0, nrow = num_groups, ncol = num_inter),
  prior_is_EXNEX_comp = rep(TRUE, num_comp),
  prior_is_EXNEX_inter = rep(FALSE, num_inter),
  prior_tau_dist = 1,
  prior_PD = FALSE
)
## Recover user set sampling defaults
options(.user_mc_options)

}
\references{
Neuenschwander, B., Roychoudhury, S., & Schmidli, H. 
 (2016). On the use of co-data in clinical trials. \emph{Statistics in 
 Biopharmaceutical Research}, 8(3), 345-354.
}
