% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/PLMIXfunctions.R
\name{ppcheckPLMIX_cond}
\alias{ppcheckPLMIX_cond}
\title{Conditional posterior predictive check for Bayesian mixtures of Plackett-Luce models}
\usage{
ppcheckPLMIX_cond(pi_inv, seq_G, MCMCsampleP, MCMCsampleW, top1 = TRUE,
  paired = TRUE, parallel = FALSE)
}
\arguments{
\item{pi_inv}{An object of class \code{top_ordering}, collecting the numeric \eqn{N}\eqn{\times}{x}\eqn{K} data matrix of partial orderings, or an object that can be coerced with \code{\link{as.top_ordering}}.}

\item{seq_G}{Numeric vector with the number of components of the Plackett-Luce mixtures to be assessed.}

\item{MCMCsampleP}{List of size \code{length(seq_G)}, whose generic element is a numeric \eqn{L}\eqn{\times}{x}\eqn{(G*K)} matrix with the MCMC samples of the component-specific support parameters.}

\item{MCMCsampleW}{List of size \code{length(seq_G)}, whose generic element is a numeric \eqn{L}\eqn{\times}{x}\eqn{G} matrix with the MCMC samples of the mixture weights.}

\item{top1}{Logical: whether the posterior predictive \eqn{p}-value based on the top item frequencies has to be computed. Default is \code{TRUE}.}

\item{paired}{Logical: whether the posterior predictive \eqn{p}-value based on the paired comparison frequencies has to be computed. Default is \code{TRUE}.}

\item{parallel}{Logical: whether parallelization should be used. Default is \code{FALSE}.}
}
\value{
A list with a named element:

 \item{\code{post_pred_pvalue_cond}}{ Numeric \code{length(seq_G)}\eqn{\times}{x}\eqn{2} matrix of posterior predictive \eqn{p}-values based on the top item and paired comparison frequencies. If either \code{top1} or \code{paired} argument is \code{FALSE}, the corresponding matrix entries are \code{NA}.}
}
\description{
Perform conditional posterior predictive check to assess the goodness-of-fit of Bayesian mixtures of Plackett-Luce models with a different number of components.
}
\details{
The \code{ppcheckPLMIX_cond} function returns two posterior predictive \eqn{p}-values based on two chi squared discrepancy variables involving: (i) the top item frequencies and (ii) the paired comparison frequencies. In the presence of partial sequences in the \code{pi_inv} matrix, the same missingness patterns observed in the dataset (i.e., the number of items ranked by each sample unit) are reproduced on the replicated datasets from the posterior predictive distribution. Differently from the \code{ppcheckPLMIX} function, the condional discrepancy measures are obtained by summing up the chi squared discrepancies computed on subsamples of observations with the same number of ranked items.
}
\examples{

data(d_carconf)
K <- ncol(d_carconf)

## Fit 1- and 2-component PL mixtures via MAP estimation
MAP_1 <- mapPLMIX_multistart(pi_inv=d_carconf, K=K, G=1, 
                             n_start=2, n_iter=400*1)

MAP_2 <- mapPLMIX_multistart(pi_inv=d_carconf, K=K, G=2, 
                             n_start=2, n_iter=400*2)
                                   
MAP_3 <- mapPLMIX_multistart(pi_inv=d_carconf, K=K, G=3, 
                             n_start=2, n_iter=400*3)
                                   
mcmc_iter <- 30
burnin <- 10

## Fit 1- and 2-component PL mixtures via Gibbs sampling procedure
GIBBS_1 <- gibbsPLMIX(pi_inv=d_carconf, K=K, G=1, n_iter=mcmc_iter, 
                      n_burn=burnin, init=list(p=MAP_1$mod$P_map,
                      z=binary_group_ind(MAP_1$mod$class_map,G=1)))
GIBBS_2 <- gibbsPLMIX(pi_inv=d_carconf, K=K, G=2, n_iter=mcmc_iter, 
                      n_burn=burnin, init=list(p=MAP_2$mod$P_map,
                      z=binary_group_ind(MAP_2$mod$class_map,G=2)))
GIBBS_3 <- gibbsPLMIX(pi_inv=d_carconf, K=K, G=3, n_iter=mcmc_iter, 
                      n_burn=burnin, init=list(p=MAP_3$mod$P_map,
                      z=binary_group_ind(MAP_3$mod$class_map,G=3)))
                            
## Checking goodness-of-fit of the estimated mixtures
CHECKCOND <- ppcheckPLMIX_cond(pi_inv=d_carconf, seq_G=1:3, 
                               MCMCsampleP=list(GIBBS_1$P, GIBBS_2$P, GIBBS_3$P), 
                               MCMCsampleW=list(GIBBS_1$W, GIBBS_2$W, GIBBS_3$W))
CHECKCOND$post_pred_pvalue

}
\references{
Mollica, C. and Tardella, L. (2017). Bayesian Plackett-Luce mixture models for partially ranked data. \emph{Psychometrika}, \bold{82}(2), pages 442--458, ISSN: 0033-3123, DOI: 10.1007/s11336-016-9530-0.
}
\seealso{
\code{\link{ppcheckPLMIX}}
}
\author{
Cristina Mollica and Luca Tardella
}
