# S-Plus script developed by Professor Alexander McNeil, mcneil@math.ethz.ch
# R-version adapted by Scott Ulman (scottulman@hotmail.com)
# This free script using QRMLib is distributed in the hope that it will be useful, 
# but WITHOUT ANY WARRANTY; without even the implied warranty of 
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
# GNU General Public License for more details. 

######Load the QRMlib and sp500 data set##################
#QRMlib.pdf is a help file for the functions used by QRMlib.  It is available at
#...\Program Files\R\R-2.2.1\library\QRMlib\Docs
#If you have created the QRMBook workspace and .Rprofile  as described in QRMlib.pdf
#topics 'QRMBook-workspace' and 'profileLoadLibrary', then you may comment out the
#following line:
library(QRMlib);
#if you have previously opened the sp500 data set  AND saved 
#the workspace, you may comment out the following line:
data(sp500);
#################################################


#SU: This exercise refers to the Sylized facts in section 4.1.1 of QRM, p. 117.
################# Get some index data
#Select only data after 01/01/1998 and on or before 12/31/2003:
# In R, you must have the year first in an ISO-format: 
#In version 240.10068, fCalendar uses cut() rather than cutSeries() to select a subset from timeseries:
index <- cut(sp500,"1998-01-01", "2003-12-31");

#The mk.returns() function in R differs from its S-Plus counterpart.  You MUST pass a timeSeries
#R-Metrics type object as the argument to the function or you will get an error returned.
rseries <- mk.returns(index);
#You must cast the timeSeries to a list to report the summary values (Min, 1st Quantile, Median,
#Mean, 3rd Quantile, and Max):
summary(as.list(rseries));

#Extract the data field only (not the positions which are the dates)
data <- seriesData(rseries);

#Show the number of rows.  The S-Plus numRows()must be replaced with the R-function nrow()which uses 
#only the Data values from the timeSeries object:
#numRows(rseries) is the S-Plus version:
nrow(data); # is the R-version


#Fit extracted data to student-t distribution using the function fit.st()in functionsUnivariate.R from the library.
#fit.st() returns a list telling whether convergence occurred, the parameter estimates, the std error of
#parameter estimates, the asymptotic covariance matrix, and the value of the maximize loglikelihood as 
#shown in the following list() command:
#list(converged=converged,par.ests=par.ests, par.ses=par.ses, asymp.cov=asymp.cov, ll.max=loglh.max)
tmp <- fit.st(data);
#Test if convergence occurred:
tmp$converged;
#Observe parameter estimates
tmp$par.ests;


#simulate a set of random variables from the student-t distribution using parameters estimated from the data:
set.seed(19);
tdata <- -(tmp$par.ests[2]+tmp$par.ests[3]*rt(length(data),df=tmp$par.ests[1]));
#simulate a set of random variable from a normal distribution using sample moments from data as inputs:
ndata <- rnorm(length(data),mean(data),sqrt(var(data)));


#Contrast simulated data with iid data from SP500 time series
par(mfrow=c(3,1));
#S-Plus code used the ts.plot() method and must be modified for R: use the plot() function instead.  
#Use the type="h" with plot to get histogram thin lines in R-language:
plot(data,ylim=range(ndata,tdata,data),main="Real", type="h");
plot(ndata,ylim=range(data,ndata,tdata),ylab="",main="Normal",type="h");
plot(tdata,ylim=range(data,ndata,tdata),ylab="",main="Student t",type="h");
#return to one graph per sheet:
par(mfrow=c(1,1));

par(mfcol=c(3,2));
#SU 06/15/2006: S-Plus uses ACF plots (acf.plot()) which do not seem to exist in R 
#(although ACF does). ACF stands for Auto Correlation Function.  We will merely use the R-language acf()
#function with plot=T while passing only data from time series via timeSeries@Data.
#ACF plots under S-Plus
#out <- acf(rseries,plot=F)
#acf.plot(out,main="Real")
#Replaced by these under the R-language
acf(data,plot=T, type="correlation");
acf(ndata,plot=T, type="correlation");
acf(tdata,plot=T, type="correlation");
acf(abs(data),plot=T, type="correlation");
acf(abs(ndata),plot=T, type="correlation");
acf(abs(tdata),plot=T, type="correlation");
par(mfcol=c(1,1));


#Heavy tails as usual
qqnorm(data,ylab="Real");

#Clustered extreme values
#First examine iid data generated by random simulations for the t distribution: tdata.
kval=50; #set threshold so we have 50 observations above threshold
rate <- kval/length(data); #use in the QQplot below
#find threshold using function in functionsEVT.R
u <- findthreshold(tdata,kval);

#In S-Plus, exceedance times vector is filled with index values (from 1 to 1505) from tdata vector
#where values tdata[index] exceed u, the threshold.  
# We rename it to exceedance.indices for clarity since it represents a vector position (not a time):
exceedance.indices <- (1:length(tdata))[tdata>u];
#Fill exceedance.marks vector with all data items from tdata which exceed the threshold value of u:
exceedance.marks <- tdata[tdata>u];
#Plot the exceedances (marks on the Y, indices on the X-axis)
plot(x=exceedance.indices, y=exceedance.marks, type="h",main="Clustering Evidence-Simulated t data",ylab="Large Returns",xlab="Time: 1998-2004");
#The following is the unworkable S-Plus call to ts.plot()which we have just replaced 
#ts.plot(its(exceedance.marks,exceedance.times),type="h");

gaps <- diff(exceedance.indices);
#Use the QQplot function from functionsUtility.R; be sure to use 'exp' and not 'pexp' for the 
#exponential distribution: 
QQplot(gaps,re="exp",rate=rate);
#Add a straight diagonal line to the plot to help assess the fit of the QQplot:
abline(1,1);

#SU: this is a single-sample Kolmogorov-Smirnoff goodness-of-fit test.  The S-Plus
#function call
#     ks.gof(gaps,dist="exp",rate=rate)
#  must be replaced by the R-function ks.test() contained in the stats package which will
# normally be loaded:
#In R, we must use the following where we must precede the distribution name with 'p' for probability
# distribution.  Also help("pexp") shows the parameter name is 'rate')
ks.test(gaps, "pexp", rate=rate);

#Now repeat the analysis with real data from sp500 rather than the simulated data. 
u2 <- findthreshold(data,kval);
exceedance.indices2 <- (1:length(data))[data>u2];
exceedance.marks2 <- data[data>u2];
#The following R-language plot
plot(x=exceedance.indices2, y=exceedance.marks2,type="h", main="Clustering Evidence-Real Data",ylab="Large Returns",xlab="Time: 1998-2004");
#replaces this S-Plus plot which allows numbers rather than dates to be passed.
#ts.plot(its(exceedance.marks,exceedance.times),type="h")
realgaps <- diff(exceedance.indices2);
QQplot(realgaps,re="exp",rate=rate);
abline(1,1);

#Kolgomorov Smirnov test on real data.
#In R, we must use the following where we must precede the distribution name with 'p' for probability
# distribution.  Also help("pexp") shows the parameter name is 'rate')
ks.test(realgaps, "pexp", rate=rate);




