#' @title
#' Extract tibbles from a REDCapTidieR supertibble and bind it to an environment
#'
#' @description
#' Take a supertibble generated with \code{read_redcap_tidy()}
#' and bind it to an environment. The default is the global environment.
#'
#' @returns
#' This function returns nothing as it's used solely for its side effect of
#' modifying an environment.
#'
#' @param .data A tibble generated by \code{read_redcap_tidy()}. Required.
#' @param environment The environment to add the tibbles to. Default is
#' \code{global_env()}).
#' @param tbls A vector of \code{form_name}s. Default behavior is all tibbles.
#' @param structure Specify REDCap structure to load into environment, either
#' \code{repeating} or \code{nonrepeating}. Defaults to all structures.
#'
#' @importFrom dplyr filter pull %>%
#' @importFrom rlang env_poke current_env new_environment global_env .data
#' @importFrom purrr map2 pluck
#'
#' @examples
#' # Designate a sample environment
#' sample_env <- environment()
#'
#' # Supply a `read_redcap_tidy()` output to bind
#' sample_data <- tibble::tribble(
#'   ~redcap_form_name,    ~redcap_data,   ~structure,
#'   "super_hero_powers",   list(),         "repeating",
#'   "heroes_information",  list(),         "nonrepeating"
#' )
#'
#' bind_tables(sample_data, sample_env)
#'
#' @export

bind_tables <- function(.data,
                        environment = global_env(),
                        tbls = NULL,
                        structure = NULL) {
  # Name variables
  my_tbls <- tbls
  my_structures <- structure
  env_data <- .data

  # Apply conditional loading for specific forms or structures
  if (!is.null(my_tbls)) {
    env_data <- env_data %>%
      filter(.data$redcap_form_name %in% my_tbls)
  }

  if (!is.null(my_structures)) {
    env_data <- env_data %>%
      filter(.data$structure %in% my_structures)
  }

  table_names <- env_data %>%
    pull(.data$redcap_form_name)

  # Map over table names and environment data to load into environment
  map2(.x = table_names,
       .y = env_data$redcap_data,
       .f = ~ env_poke(env = environment,
                       nm = .x,
                       value = .y)
  )
  return(invisible(NULL))
}
