% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/pibf.R
\name{pibf}
\alias{pibf}
\title{Prediction intervals with boosted forests}
\usage{
pibf(
  formula,
  traindata,
  testdata,
  alpha = 0.05,
  calibration = c("cv", "oob", FALSE),
  coverage_range = c(1 - alpha - 0.005, 1 - alpha + 0.005),
  numfolds = 5,
  params_ranger = list(num.trees = 2000, mtry = ceiling(px/3), min.node.size = 5,
    replace = TRUE),
  oob = FALSE
)
}
\arguments{
\item{formula}{Object of class \code{formula} or \code{character} describing
the model to fit.}

\item{traindata}{Training data of class \code{data.frame}.}

\item{testdata}{Test data of class \code{data.frame}.}

\item{alpha}{Confidence level. (1 - \code{alpha}) is the desired coverage
level. The default is \code{alpha} = 0.05 for the 95\% prediction interval.}

\item{calibration}{Calibration method for finding working level of
\code{alpha}, i.e. \eqn{\alpha_w}. Options are \code{"cv"}, \code{"oob"},
and \code{FALSE} standing for calibration with cross-validation, OOB
calibration, and no calibration, respectively. See below for details. The
default is \code{"cv"}.}

\item{coverage_range}{The allowed target calibration range for coverage level.
\eqn{\alpha_w} is selected such that the \code{"cv"} or \code{"oob"}
coverage is within \code{coverage_range}.}

\item{numfolds}{Number of folds for calibration with cross-validation. The
default is 5 folds.}

\item{params_ranger}{List of parameters that should be passed to
\code{ranger}. In the default parameter set, \code{num.trees} = 2000,
\code{mtry} = \eqn{px/3} (rounded up), \code{min.node.size} = 5,
\code{replace} = TRUE. See \code{ranger} for possible parameters.}

\item{oob}{Should out-of-bag (OOB) predictions and prediction intervals for
the training observations be returned?}
}
\value{
A list with the following components:

\item{pred_interval}{Prediction intervals for test data. A list containing
lower and upper bounds.}
\item{test_pred}{Bias-corrected random forest predictions for test data.}
\item{alphaw}{Working level of \code{alpha}, i.e. \eqn{\alpha_w}. If
\code{calibration = FALSE}, it returns \code{NULL}.}
\item{test_response}{If available, test response.}
\item{oob_pred_interval}{Out-of-bag (OOB) prediction intervals for train
data. Prediction intervals are built with \code{alpha}. If
\code{oob = FALSE}, it returns \code{NULL}.}
\item{oob_pred}{Bias-corrected out-of-bag (OOB) predictions for train data.
If \code{oob = FALSE}, it returns \code{NULL}.}
\item{train_response}{Train response.}
}
\description{
Constructs prediction intervals with boosted forests.
}
\section{Details}{


\strong{Calibration process}

Let (\eqn{1-\alpha}) be the target coverage level. The goal of the
calibration is to find the value of \eqn{\alpha_w}, which is the working
level of \eqn{\alpha} called by Roy and Larocque (2020), such that the
coverage level of the PIs for the training observations is closest to the
target coverage level. Two calibration procedures are provided: calibration
with cross-validation and out-of-bag (OOB) calibration.

\enumerate{
\item In calibration with CV, we apply k-fold cross-validation to form
prediction intervals for the training observations. In each fold, we split
the original training data set into training and testing sets. For the
training set, we train a one-step boosted random forest and compute the OOB
residuals. Then, for each observation in the testing set, we build a PI.
After completing CV, we compute the coverage level with the constructed PIs
and if the coverage is not within the acceptable coverage range
(\code{coverage_range}), then we apply a grid search to find the
\eqn{\alpha_w} such that \eqn{\alpha_w} is the closest to the target
\eqn{\alpha} among the set of \eqn{\alpha_w}'s that ensures the target
coverage level for the constructed PIs. Once we find the \eqn{\alpha_w}, we
use this level to build the PI for the new observations.

\item The OOB calibration procedure is proposed by Roy and Larocque (2020)
and it is the default calibration procedure of \code{rfpi()}. See details
section of \code{rfpi()} for the detailed explanation of this calibration
procedure.
}

In terms of computational time, OOB calibration is faster than calibration
with CV. However, empirical results show that OOB calibration may result in
conservative prediction intervals. Therefore, the recommended calibration
procedure for the PIBF method is calibration with CV.
}

\examples{
\donttest{
## load example data
data(BostonHousing, package = "RFpredInterval")
set.seed(2345)

## define train/test split
testindex <- 1:10
trainindex <- sample(11:nrow(BostonHousing), size = 100, replace = FALSE)
traindata <- BostonHousing[trainindex, ]
testdata <- BostonHousing[testindex, ]
px <- ncol(BostonHousing) - 1

## construct 95\% PI with "cv" calibration using 5-folds
out <- pibf(formula = medv ~ ., traindata = traindata,
  testdata = testdata, calibration = "cv", numfolds = 5,
  params_ranger = list(num.trees = 40))

## get the PI for the first observation in the testdata
c(out$pred_interval$lower[1], out$pred_interval$upper[1])

## get the bias-corrected random forest predictions for testdata
out$test_pred

## construct 90\% PI with "oob" calibration
out2 <- pibf(formula = medv ~ ., traindata = traindata,
  testdata = testdata, alpha = 0.1, calibration = "oob",
  coverage_range = c(0.89,91), params_ranger = list(num.trees = 40))

## get the PI for the testdata
out2$pred_interval

## get the working level of alpha (alphaw)
out2$alphaw
}

}
\references{
Alakus, C., Larocque, D., & Labbe, A. (2022). RFpredInterval: An
R Package for Prediction Intervals with Random Forests and Boosted Forests.
R JOURNAL, 14(1), 300-319.

Roy, M. H., & Larocque, D. (2020). Prediction intervals with
random forests. Statistical methods in medical research, 29(1), 205-229.
doi:10.1177/0962280219829885.
}
\seealso{
\code{\link{piall}} \code{\link{rfpi}}
\code{\link{print.rfpredinterval}}
}
