##/*****************************************************************************
## * SIENA: Simulation Investigation for Empirical Network Analysis
## *
## * Web: http://stat.gamma.rug.nl/siena.html
## *
## * File: siena07.r
## *
## * Description: This file contains the main controlling module for the model
## * fitting
## * Also contains utility functions used within siena07
## ****************************************************************************/

##@siena07 siena07
siena07<- function(x, batch = FALSE, verbose = FALSE, useCluster = FALSE,
                   nbrNodes = 2, initC=FALSE,
                   clusterString=rep("localhost", nbrNodes), tt=NULL,
                   parallelTesting=FALSE, ...)
{
    exitfn <- function()
    {
       if (!is.batch())
       {
           tkdestroy(tkvars$tt)
       }
       ## close the report file
       Report(close=TRUE)
    }
    on.exit(exitfn())


    time0 <-  proc.time()['elapsed']
    z <- NULL ## z is the object for all control information which may change.
    ## x is designed to be readonly. Only z is returned.

    if (useCluster)
    {
        require(snow)
        require(rlecuyer)
        x$firstg <- x$firstg * sqrt(nbrNodes)
        z$int <- nbrNodes
    }
    else
    {
        z$int <- 1
    }
    if (parallelTesting)
    {
        set.seed(1, kind='Wich')
        ## randomseed2 is for second generator needed only for parallel testing
        randomseed2 <- .Random.seed
      #  .Random.seed[2:4] <- as.integer(c(1,2,3))
       # randomseed2[2:4] <- as.integer(c(3,2,1))
        randomseed2[2:4] <- as.integer(c(1, 2, 3))
        seed <- 1
        newseed <- 1
        z$parallelTesting <- TRUE
    }
    else
    {
        randomseed2 <-  NULL
        ## x$randomSeed is the user seed, if any
        if (!is.null(x$randomSeed))
        {
            set.seed(x$randomSeed)
            seed <- x$randomSeed
        }
        else
        {
            if (exists(".Random.seed"))
            {
                rm(.Random.seed, pos=1)
            }
            newseed <- trunc(runif(1) * 1000000)
            set.seed(newseed)  ## get R to create a random number seed for me.
            seed <- NULL
        }
    }
    z$randomseed2 <- randomseed2

    ## set the global is.batch
    is.batch(batch)

    ## open the output file
    Report(open=TRUE, projname=x$projname, verbose=verbose)
    InitReports(seed, newseed)

    ## reset the globals for interrupts
    NullChecks()

    ## create the screen
    if (!is.batch())
    {
        tkvars <- siena07Gui(tt=tt)
        z$tkvars<- tkvars
        z$pb <- list(pb=tkvars$pb, pbval=0, pbmax=1)
    }
    else
    {
        z$pb <- list(pb=NULL, pbval=0, pbmax=1)
    }

    z <- robmon(z, x, useCluster, nbrNodes, initC, clusterString,...)

    time1 <-  proc.time()['elapsed']
    Report(c("Total computation time", round(time1 - time0, digits=2),
             "seconds.\n"), outf)

    if (useCluster)
        stopCluster(z$cl)

    class(z) <- "sienaFit"
    z
}
##@InitReports siena07 Print report
InitReports <- function(seed, newseed)
{
    Report("\n\n-----------------------------------\n", outf)
    Report("New Analysis started.\n", outf)
    Report(c("Date and time:", format(Sys.time(),"%d/%m/%Y %H:%M:%S")), outf)
    Report("\nNew results follow.\n", outf)
    Report("-----------------------------------\n", outf)
    rforgeRevision <-  packageDescription("RSiena",
                                          fields="Repository/R-Forge/Revision")
    if (is.na(rforgeRevision))
    {
        revision <- ""
    }
    else
    {
        revision <- paste(" R-forge revision: ", rforgeRevision, " ", sep="")
    }
    Report(c("\nSiena version ",
             packageDescription("RSiena", fields = "Version"), " (",
             format(as.Date(packageDescription("RSiena", fields = "Date")),
                    "%d %b %y"), ")",
             revision, "\n\n"), sep = '',  outf )
    Heading(1, outf, "Estimation by stochastic approximation algorithm.")
    if (is.null(seed))
    {
        Report("Random initialization of random number stream.\n", outf)
        Report(sprintf("Current random number seed is %d.\n", newseed), outf)
    }
    else
    {
        Report(sprintf("Current random number seed is %d.\n", seed), outf)
    }
}

##@AnnouncePhase siena07 Progress reporting
AnnouncePhase <- function(z, x, subphase=NULL)
{
    if (!is.batch())
    {
        tkdelete(z$tkvars$phase, 0, "end")
        tkinsert(z$tkvars$phase, 0, paste(" ", z$Phase))
        tkdelete(z$tkvars$subphase, 0, "end")
        tkdelete(z$tkvars$iteration, 0, "end")
        tkinsert(z$tkvars$iteration, 0, format(0, width=6))
    }
    if (missing(subphase))
    {
        Report(c("\nStart phase", z$Phase, "\n"), cf)
    }
    else
    {
        if (!is.batch())
        {
            tkinsert(z$tkvars$subphase, 0, paste(" ", subphase))
        }
        Report(c("\nStart phase ", z$Phase, ".", subphase, "\n"), sep="", cf)
    }
    if (z$Phase == 0)
    {
        if (!is.batch())
        {
            tkconfigure(z$tkvars$current, height=min(z$pp, 30))
            tkconfigure(z$tkvars$deviation, height=min(z$pp, 30))
            tkconfigure(z$tkvars$quasi, height=min(z$pp, 30))
        }
        n1pos <- z$n1 * (z$pp + 1)
        z$n2min0 <- 7 + z$pp
        z$n2min0 <- max(5, z$n2min0 / z$int)
        z$n2minimum<- rep(0, x$nsub)
        z$n2maximum<- rep(0, x$nsub)
    ## 2.5198421 = 2^(4/3); this gives a gain parameter of order n^(-3/4) ##
        if (x$nsub > 0)
        {
            z$n2minimum[1] <- trunc(z$n2min0 * 2.52)
            z$n2maximum[1] <- z$n2minimum[1] + 200
            if (x$nsub > 1)
            {
                for (i in 2:x$nsub)
                {
                    z$n2minimum[i] <- trunc(z$n2minimum[i-1] * 2.52)
                    z$n2maximum[i] <- z$n2minimum[i] + 200
                }
            }
        }
        z$n2partsum <- c(0, cumsum(z$n2maximum))
        n2sum <- sum(z$n2maximum)
        ##Progress bar
        pbmax <- n1pos + n2sum + x$n3
        z$n1pos<- n1pos
        if (!x$maxlike && z$FinDiff.method)
            pbmax <- pbmax + x$n3 * z$pp
        z$pb$pbval <- 0
        z$pb <- createProgressBar(z$pb, maxvalue=pbmax)
        z$pb$pbmax <- pbmax
   }
    if (z$Phase==2)
    {
        propo <- z$n1pos + z$n2partsum[subphase]
        if (propo> getProgressBar(z$pb))
            z$pb <-setProgressBar(z$pb,propo)
    }
    if (z$Phase ==3)
    {
        propo <- z$n1pos + z$n2partsum[x$nsub + 1]
        if (!z$AllUserFixed)
            z$pb <- setProgressBar(z$pb,propo)
       else
        {
            max <- x$n3
            z$pb <-createProgressBar(z$pb,max)
       }
    }
    z
}
##@roundfreq siena07 Prettify interval between progress reports
roundfreq <- function(w)
{
    vec1 <- c(1, 2, 3, 4, 31, 66, 101, 300, 500)
    vec2 <- c(1, 2, 3, 20, 50, 100, 200, 500)
    if (is.batch())
        w <- max(10,vec2[findInterval(w, vec1, all.inside=TRUE)])
    else
        w <- vec2[findInterval(w, vec1[1:7], all.inside=TRUE)]
    w
}

##@WriteOutTheta siena07 Progress reporting
WriteOutTheta <- function(z)
{
    if (!is.batch())
    {
        DisplayTheta(z)
    }
    else
    {
        Report(c("theta:", format(z$theta, digits=3), "\n"))
    }
    Report("Current parameter values:\n", cf)
    Report(format(z$theta), cf, fill=80)
}

##@DisplayThetaAutocor siena07 Progress reporting
DisplayThetaAutocor <- function(z)
{
    if (!is.batch())
    {
        DisplayTheta(z)
        tkdelete(z$tkvars$quasi, "1.0", "end")
        tkinsert(z$tkvars$quasi, "1.0", FormatString(z$pp, z$ac))
    }
    else
    {
        Report(c("theta", format(z$theta, digits=3),"\n"))
        Report(c("ac", format(z$ac, digits=3), "\n"))
  }

}
##@DisplayandWriteTheta siena07 Progress reporting
DisplayandWritetheta <- function(z)
{
    if (!is.batch())
    {
        DisplayTheta(z)
    }
    else
    {
        Report(c("theta", format(z$theta, digits=3), "\n"))
    }
}
##@DisplayTheta siena07 Progress reporting
DisplayTheta <- function(z)
{
    if (!is.batch())
    {
        tkdelete(z$tkvars$current, "1.0", "end")
        tkinsert(z$tkvars$current, "1.0", FormatString(z$pp, z$theta))
    }

}
##@FormatString siena07 Progress Reporting
FormatString <- function(pp, value)
{
    ppuse <- min(30, pp)
    nbrs <- format(1:ppuse)
    nch <- nchar(nbrs[1])
    formatstr <- paste("%", nch, "d.%", (13 - nch), ".4f\n", sep="",
                       collapse="")
    paste(sprintf(formatstr, 1:ppuse, value[1:ppuse]), collapse="")
}
##@DisplayDeviations siena07 Progress reporting
DisplayDeviations <- function(z, fra)
{
    if (!is.batch())
    {
        tkdelete(z$tkvars$deviations, "1.0", "end")
        tkinsert(z$tkvars$deviations, "1.0", FormatString(z$pp, fra))
    }
}
##@DisplayIteration siena07 Progress reporting
DisplayIteration <- function(z)
{
    if (!is.batch())
    {
        tkdelete(z$tkvars$iteration, 0, "end")
        tkinsert(z$tkvars$iteration, 0, format(z$nit, width=6))
        tcl("update")
    }
}
##@Root siena07 Safe square root for compatibility with siena3. Probably not necessary in R.
Root<- function(x)
{
    ifelse(abs(x) > 1e-36, sqrt(abs(x)), 1e-18)
}

##@getProgressBar siena07 Progress reporting
getProgressBar <- function(pb)
{
    if (is.batch())
        val <- pb$pbval
    else
        val <- as.numeric(tclvalue(tkcget(pb$pb, "-value")))
    val
}

##@setProgressBarProgress siena07 reporting
setProgressBar <- function(pb, val)
{
    if (is.batch())
    {
        pb$pbval <- val
    }
    else
    {
        tkconfigure(pb$pb, value=val)
        tcl("update")
    }
    pb
}
##@createProgressBar siena07 Progress reporting
createProgressBar <- function(pb, maxvalue)
{
    if (is.batch())
        pb$pbmax <- maxvalue
    else
        tkconfigure(pb$pb, maximum=maxvalue)
    pb
}
##@tkErrorMessage Miscellaneous Not used
tkErrorMessage <- function()
{
    tkmessageBox(geterrmessage(), icon="error")
}

##@errorHandler Miscellaneous Not used
errorHandler <- function()
{
    opts <- options()
    if (!is.batch())
    {
        options(show.error.messages=FALSE)
        options(error=tkErrorMessage)
    }
}
