% Generated by roxygen2 (4.1.1): do not edit by hand
% Please edit documentation in R/radCor.R
\name{radCor}
\alias{radCor}
\title{Radiometric Calibration and Correction}
\usage{
radCor(img, metaData, method = "apref", bandSet = "full", hazeValues,
  hazeBands, atmosphere, darkProp = 0.02, verbose)
}
\arguments{
\item{img}{raster object}

\item{metaData}{object of class ImageMetaData or a path to the meta data (MTL) file.}

\item{method}{Radiometric conversion/correction method to be used. There are currently four methods available (see Details):
"rad", "apref", "sdos", "dos", "costz".}

\item{bandSet}{Numeric or character. original Landsat band numbers or names in the form of ("B1", "B2" etc). If set to 'full' all bands in the solar (optical) region will be processed.}

\item{hazeValues}{Starting haze value, can be estimated using \link{estimateHaze}. if not provided and method is "dos" or "costz" hazeValues will be estimated in an automated fashion. Not needed for apparent reflectance.}

\item{hazeBands}{Bands corresponding to hazeValues.}

\item{atmosphere}{Character. Atmospheric characteristics. Will be estimated if not expicilty provided. Must be one of \code{"veryClear", "clear", "moderate", "hazy"} or \code{"veryHazy"}.}

\item{darkProp}{Numeric. Estimated proportion of dark pixels in the scene. Used only for automatic guessing of hazeValues (typically one would choose 1 or 2\%).}

\item{verbose}{Logical. Print status information.}
}
\value{
RasterStack with top-of-atmosphere radiance (\eqn{W/(m^2 * srad * \mu m)}), at-satellite brightness temperature (K),
top-of-atmosphere reflectance (unitless) corrected for the sun angle or at-surface reflectance (unitless).
}
\description{
Implements several different methods for radiometric calibraion and correction of Landsat data.
You can either specify a metadata file, or supply all neccesary values manually.
With proper parametrization apref and sdos should work for other sensors as well.
}
\details{
The atmospheric correction methods (sdos, dos and costz) apply to the optical (solar) region of the spectrum and do not affect the thermal band.

Dark object subtraction approaches rely on the estimation of atmospheric haze based on *dark* pixels. Dark pixels are assumed to have zero reflectance, hence the name.
It is then assumed further that any radiation originating from such *dark* pixels is due to atmospheric haze and
not the reflectance of the surface itself.

The folloiwing \code{methods} are available:
\tabular{ll}{
rad   \tab Radiance \cr
apref \tab Apparent reflectance (top-of-atmosphere reflectance) \cr
dos   \tab Dark object subtratction following Chavez (1989) \cr
costz \tab Dark object subtraction following Chavez (1996) \cr
sdos  \tab Simple dark object subtraction. Classical DOS, Lhaze must be estimated for each band separately.
}

If either "dos" or "costz"  are selected, radCor will use the atmospheric haze decay model described by Chavez (1989).
Depending on the \code{atmosphere} the following coefficients are used:
\tabular{ll}{
veryClear \tab \eqn{\lambda^{-4.0}} \cr
clear     \tab \eqn{\lambda^{-2.0}} \cr
moderate  \tab \eqn{\lambda^{-1.0}} \cr
hazy      \tab \eqn{\lambda^{-0.7}} \cr
veryHazy  \tab \eqn{\lambda^{-0.5}}
}

The implemented sun-earth distances neglect the earth's eccentricity. Instead we use a 100 year daily average (1979-2070).
}
\note{
This was originally a fork of randcorr() function in the landsat package. This version works on Raster* objects and hence is suitable for large rasters.
}
\examples{
library(raster)
## Import meta-data and bands based on MTL file
mtlFile  <- system.file("external/landsat/LT52240631988227CUB02_MTL.txt", package="RStoolbox")
metaData <- readMeta(mtlFile)
\donttest{lsat     <- stackMeta(mtlFile)}

\dontshow{lsat <- readAll(lsat)}

## Convert DN to top of atmosphere reflectance and brightness temperature
lsat_ref <- radCor(lsat, metaData = metaData, method = "apref")

## Correct DN to at-surface-reflecatance with DOS (Chavez decay model)
\donttest{lsat_sref <- radCor(lsat, metaData = metaData, method = "dos")}

## Correct DN to at-surface-reflecatance with simple DOS
## Automatic haze estimation
hazeDN    <- estimateHaze(lsat, hazeBands = 1:4, darkProp = 0.01, plot = TRUE)
lsat_sref <- radCor(lsat, metaData = metaData, method = "sdos",
                    hazeValues = hazeDN, hazeBands = 1:4)
}
\references{
S. Goslee (2011): Analyzing Remote Sensing Data in R: The landsat Package. Journal of Statistical Software 43(4).
}

