% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/unsuperClass.R
\name{unsuperClass}
\alias{unsuperClass}
\title{Unsupervised Classification}
\usage{
unsuperClass(img, nSamples = 10000, nClasses = 5, nStarts = 25,
  nIter = 100, norm = FALSE, clusterMap = TRUE,
  algorithm = "Hartigan-Wong", ...)
}
\arguments{
\item{img}{Raster* object.}

\item{nSamples}{Integer. Number of random samples to draw to fit cluster map. Only relevant if clusterMap = TRUE.}

\item{nClasses}{Integer. Number of classes.}

\item{nStarts}{Integer. Number of random starts for kmeans algorithm.}

\item{nIter}{Integer. Maximal number of iterations allowed.}

\item{norm}{Logical. If \code{TRUE} will normalize img first using \link{normImage}. Normalizing is beneficial if your predictors have different scales.}

\item{clusterMap}{Logical. Fit kmeans model to a random subset of the img (see Details).}

\item{algorithm}{Character. \link[stats]{kmeans} algorithm. One of c("Hartigan-Wong", "Lloyd", "MacQueen")}

\item{...}{further arguments to be passed to \link[raster]{writeRaster}, e.g. filename}
}
\description{
Unsupervised clustering of Raster* data using kmeans clustering
}
\details{
Clustering is done using \code{\link[stats]{kmeans}}. This can be done for all pixels of the image (\code{clusterMap=FALSE}), however this can be slow and is
not memory safe. Therefore if you have large raster data (> memory), as is typically the case with remote sensing imagery it is advisable to choose clusterMap=TRUE (the default).
This means that a kmeans cluster model is calculated based on a random subset of pixels (\code{nSamples}). Then the distance of *all* pixels to the cluster centers 
is calculated in a stepwise fashion using \code{\link[raster]{predict}}. Class assignment is based on minimum euclidean distance to the cluster centers.   

The solution of the kmeans algorithm often depends on the initial configuration of class centers which is chosen randomly. 
Therefore, kmeans is usually run with multiple random starting configurations in order to find a convergent solution from different starting configurations.
The \code{nStarts} argument allows to specify how many random starts are conducted.
}
\examples{
library(raster)
input <- brick(system.file("external/rlogo.grd", package="raster"))

## Plot 
olpar <- par(no.readonly = TRUE) # back-up par
par(mfrow=c(1,2))
plotRGB(input)

## Run classification
set.seed(25)
unC <- unsuperClass(input, nSamples = 100, nClasses = 5, nStarts = 5)
unC

## Plots
colors <- rainbow(5)
plot(unC$map, col = colors, legend = FALSE, axes = FALSE, box = FALSE)
legend(1,1, legend = paste0("C",1:5), fill = colors,
       title = "Classes", horiz = TRUE,  bty = "n")

par(olpar) # reset par
}
