// -*- mode: C++; c-indent-level: 4; c-basic-offset: 4; indent-tabs-mode: nil; -*-

#include <Rcpp.h>

#include "civil_time.h"
#include "time_zone.h"

// from examples/classic.cc
// 
std::string Format(const std::string& fmt, const std::tm& tm) {
  char buf[100];
  std::strftime(buf, sizeof(buf), fmt.c_str(), &tm);
  return buf;
}

// [[Rcpp::export]]
void example0() {
    const std::time_t now = std::time(nullptr);

    std::tm tm_utc;
    gmtime_r(&now, &tm_utc);
    Rcpp::Rcout << Format("UTC: %F %T\n", tm_utc);

    std::tm tm_local;
    localtime_r(&now, &tm_local);
    Rcpp::Rcout << Format("Local: %F %T\n", tm_local);
}

// from examples/hello.cc
// 
// [[Rcpp::export]]
int helloMoon() {
    cctz::time_zone syd;
    if (!cctz::load_time_zone("Australia/Sydney", &syd)) return -1;

    // Neil Armstrong first walks on the moon
    const auto tp1 = cctz::convert(cctz::civil_second(1969, 7, 21, 12, 56, 0), syd);

    const std::string s = cctz::format("%F %T %z", tp1, syd);
    Rcpp::Rcout << s << "\n";

    cctz::time_zone nyc;
    cctz::load_time_zone("America/New_York", &nyc);

    const auto tp2 = cctz::convert(cctz::civil_second(1969, 7, 20, 22, 56, 0), nyc);
    return tp2 == tp1 ? 0 : 1;
}


// from examples/example1.cc to examples/example4.cc

// [[Rcpp::export]]
void example1() {
    cctz::time_zone lax;
    load_time_zone("America/Los_Angeles", &lax);
    
    // Time Programming Fundamentals @cppcon
    const auto tp = cctz::convert(cctz::civil_second(2015, 9, 22, 9, 0, 0), lax);

    cctz::time_zone nyc;
    load_time_zone("America/New_York", &nyc);

    Rcpp::Rcout << cctz::format("Talk starts at %T %z (%Z)\n", tp, lax);
    Rcpp::Rcout << cctz::format("Talk starts at %T %z (%Z)\n", tp, nyc);

}

// [[Rcpp::export]]
int example2() {
    const std::string civil_string = "2015-09-22 09:35:00";

    cctz::time_zone lax;
    load_time_zone("America/Los_Angeles", &lax);
    std::chrono::system_clock::time_point tp;
    const bool ok = cctz::parse("%Y-%m-%d %H:%M:%S", civil_string, lax, &tp);
    if (!ok) return -1;

    const auto now = std::chrono::system_clock::now();
    const std::string s = now > tp ? "running long!" : "on time!";
    Rcpp::Rcout << "Talk " << s << "\n";
    return 0;
}

// [[Rcpp::export]]
void example3() {
    cctz::time_zone lax;
    load_time_zone("America/Los_Angeles", &lax);

    const auto now = std::chrono::system_clock::now();
    const cctz::civil_second cs = cctz::convert(now, lax);
    
    // First day of month, 6 months from now.
    const auto then = cctz::convert(cctz::civil_month(cs) + 6, lax);
    
    Rcpp::Rcout << cctz::format("Now: %F %T %z\n", now, lax);
    Rcpp::Rcout << cctz::format("6mo: %F %T %z\n", then, lax);
}


template <typename D>
cctz::time_point<cctz::sys_seconds> FloorDay(cctz::time_point<D> tp,
                                             cctz::time_zone tz) {
  return cctz::convert(cctz::civil_day(cctz::convert(tp, tz)), tz);
}

// [[Rcpp::export]]
void example4() {
    cctz::time_zone lax;
    load_time_zone("America/Los_Angeles", &lax);
    const auto now = std::chrono::system_clock::now();
    const auto day = FloorDay(now, lax);
    Rcpp::Rcout << cctz::format("Now: %F %T %z\n", now, lax);
    Rcpp::Rcout << cctz::format("Day: %F %T %z\n", day, lax);
}
