% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plot_OR.R
\name{plot_OR}
\alias{plot_OR}
\title{Plot Odds Ratio}
\usage{
plot_OR(
  func,
  data,
  var_name,
  color_filling = grey.colors(4, start = 0.1, end = 0.9),
  verbose = FALSE
)
}
\arguments{
\item{func}{A fitted model object with binomial family, expected to be one
of the following classes:
\itemize{
\item \verb{glm lm}             : Generalized Linear Models.
\item \verb{lognet glmnet}      : Regularized Logistic Models.
}}

\item{data}{Input data frame that was used to fit the input function
(\code{data.frame} object).}

\item{var_name}{Name of a variable to plot graphs for (\code{string} object).}

\item{color_filling}{Vector with color numbers to plot in bar plot
(\code{vector} object).
Default is \code{grey.colors(4, start=0.1, end=0.9)}.}

\item{verbose}{\code{TRUE} to print additional information such as Warnings,
\code{FALSE} otherwise (\code{bool} object). Default is \code{FALSE}.}
}
\value{
A list with the following components:
\itemize{
\item \verb{$BarPlot}    : A \code{ggplot} object that visualizes dependency of a change
in Variable values on function's Odds Ratio values.
\item \verb{$BoxPlot}    : A \code{ggplot} object that visualizes distribution of data
points for a given variable.
\item \verb{$SidebySide} : A \code{ggarrange} object containing both visualizations
side-by-side.
}
}
\description{
The function accepts input in the form of a generalized linear model (GLM)
or a glmnet object, specifically those employing binomial families, and
proceeds to generate a suite of visualizations illustrating alterations in
Odds Ratios for given predictor variable corresponding to changes between
minimum, first quartile (Q1), median (Q2), third quartile (Q3), and maximum
values observed in empirical data. These plots offer a graphical depiction
of the influence exerted by individual predictors on the odds of the outcome,
facilitating a clear interpretation of their respective significance. Such
a tool aids in comprehending the interplay between predictors and outcomes
within the logistic regression framework, particularly within the context
of empirical data distributions.
}
\examples{
### Prepare Sample Binomial Data
set.seed(42)
obs_num = 100

x1 = rnorm(obs_num)
x2 = rnorm(obs_num)
x3 = rnorm(obs_num)

prob = plogis(1 + 0.3 * x1 + 0.2 * x2 + 0.1 * x3)
y = rbinom(obs_num, 1, prob)
data = data.frame(x1, x2, x3, y)


### GLM Object Exmaple
# Get GLM model
glm_object = glm(y ~ x1 + x2 + x3,
                 family=binomial(link="logit"),
                 data=data)
summary(glm_object)

# Plot Odds Ratio graphs
plot_OR(glm_object, data, var_name="x2")$"SidebySide"


### GLMNET Object Example
require(glmnet)

# Get Lasso model
y_lasso = data$y
x_lasso = model.matrix(as.formula(paste("~",
                                        paste(colnames(subset(data,
                                                              select=-c(y))),
                                              collapse = "+"),
                                        sep = "")),
                       data=data)
x_lasso = x_lasso[,-1]
ndim_lasso = dim(x_lasso)[1]

# Select the 1se lambda from cross validation
cv_model_lasso = cv.glmnet(x_lasso, y_lasso, family="binomial", alpha=1)
lambda_lasso = cv_model_lasso$lambda.1se
plot(cv_model_lasso)

# Get a model with the specified lambda
model_lasso = glmnet(x_lasso, y_lasso, family="binomial",
                     alpha=0.5, lambda=lambda_lasso)
summary(model_lasso)

# Plot Odds Ratio graphs
plot_OR(model_lasso, data, var_name="x2")$"SidebySide"
}
