% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fforecast.R
\name{fforecast}
\alias{fforecast}
\title{Functional Singular Spectrum Analysis Recurrent and Vector Forecasting}
\usage{
fforecast(
  U,
  groups,
  len = 1,
  method = "recurrent",
  only.new = TRUE,
  tol = NULL
)
}
\arguments{
\item{U}{an object of class \code{\link{fssa}} holding the decomposition.}

\item{groups}{a list of numeric vectors where each vector includes indices
of elementary components of a group used for reconstruction and
forecasting.}

\item{len}{integer, the desired length of the forecasted FTS.}

\item{method}{a character string specifying the type of forecasting to perform:
- "recurrent" for FSSA R-forecasting.
- "vector" for FSSA V-forecasting.}

\item{only.new}{logical, if 'TRUE' then only forecasted FTS are returned, whole FTS otherwise.}

\item{tol}{a double specifying the tolerated error in the approximation of
the matrix used in forecasting algorithms.}
}
\value{
An object of class `fforecast` which is a list of objects of class
        \code{\link{funts}}, where each one corresponds to a forecasted group.
}
\description{
Perform functional singular spectrum analysis (FSSA) recurrent forecasting
(FSSA R-forecasting) or vector forecasting (FSSA V-forecasting) on univariate
or multivariate functional time series (\code{\link{funts}}) observed over a
one-dimensional domain.
}
\examples{
\dontrun{
data("Callcenter")
U <- fssa(Callcenter, L = 28)
groups <- list(1, 1:7)

## Perform FSSA R-forecast
pr_R <- fforecast(
  U = U, groups = groups, only.new = FALSE,
  len = 30, method = "recurrent"
)

plot(pr_R,  group_index = 1 )


plotly_funts(pr_R[[2]], main = "group = '1:7'")

## Perform FSSA V-forecast
pr_V <- fforecast(U = U, groups = groups, len= 30, method = "vector")

plot(pr_V, group_index = 1)

plotly_funts(pr_V[[2]], type = "3Dline" , main = "group = '1:7'")

# Multivariate forecasting example:
data("Montana")
time <- Montana$time
grid <- list(0:23, list(1:33, 1:33))
montana <- eval.funts(Montana, argvals = grid)
montana[[2]] <- array(
  scale(montana[[2]][, , ],
    center = min(montana[[2]][, , ]),
    scale = max(montana[[2]][, , ]) - min(montana[[2]][, , ])
  ),
  dim = c(33, 33, 133)
)
## Kernel density estimation of pixel intensity
NDVI <- matrix(NA, nrow = 512, ncol = 133)
for (i in 1:133) NDVI[, i] <- (density(montana[[2]][, , i], from = 0, to = 1)$y)

## Define functional objects
bs1 <- Montana$basis[[1]]

require(fda)
bs2 <- create.bspline.basis(nbasis = 15)
Y <- funts(X = list(montana[[1]], NDVI), basisobj = list(bs1, bs2),
            vnames = c("Temperature", "NDVI Density"),
            dnames = c("Time", "NDVI"),
            tname = "Date")

plotly_funts(Y,
  main = c("Temperature", "NDVI"),
  xticklocs = list(c(0, 6, 12, 18, 23), seq(1, 512, len = 9)),
  xticklabels = list(c(0, 6, 12, 18, 23), seq(0, 1, len = 9))
)

U <- fssa(Y = Y, L = 45)
plotly_funts(U$Lsingf[[1]])
plot(U$Lsingf[[2]])

groups <- list(1, 1:3)
pr_R <- fforecast(U = U, groups = groups,
                   only.new = FALSE, len = 10, method = "recurrent")
plot(pr_R)
plotly_funts(pr_R[[2]], main = "Recurrent method, group = '1:3'")

pr_V <- fforecast(U = U, groups = groups, len = 10, method = "vector")
plot(pr_V, group_index = 1)
plotly_funts(pr_V[[2]], main = "Vector method, group = '1:3'")
}

}
