% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/boot_area.R
\name{boot_area}
\alias{boot_area}
\title{Cumulative analysis of collective areas by bootstrapping}
\usage{
boot_area(
  data,
  cell.size = NA,
  R = 1000,
  percent = 50,
  quantiles = c(0.25, 0.5, 0.75)
)
}
\arguments{
\item{data}{A matrix or list of RasterLayer/SpatRaster objects. 
Each row of the matrix or each RasterLayer/SpatRaster object contains a utilisation distribution 
(or other statistics that sums to 1 - e.g. proportion of time spent).
\bold{The grid size and geographical extent must be consistent across each row of the matrix or each RasterLayer/SpatRaster object.}
The function assumes that each column of the matrix is associated with a unique geographical location or 
that each RasterLayer/SpatRaster has exactly the same geographical extent and resolution.}

\item{cell.size}{A numeric value specifying the grid cell size of the input data in metres.}

\item{R}{An integer specifying the number of iterations. A larger \emph{R} is required when the sample size is large. 
R > sample size x 100 is recommended (e.g. R > 1000 for a sample size 10).}

\item{percent}{An integer specifying the percent volume of each UD to be considered in the analysis.}

\item{quantiles}{A vector or a number to specify the quantiles to be calculated in the summary of the results.}
}
\value{
A list containing two data frames - raw results and summary (mean, sd, sem and quantiles at each sample size).
}
\description{
Function to calculate collective areas (merged x\% Utilisation Distributions or UDs) of \emph{n} individuals by bootstrapping.
}
\details{
This function calculates collective areas (e.g. 50\% UDs) of 1 to \emph{n} individuals by bootstrapping.
}
\examples{
\dontrun{

#1 Utilisation distributions of flatback turtles (n = 15).
data(ud_raster)

#2 Calculate collective areas from 3000 random permutation
area <- boot_area(ud_raster, R = 3000, percent = 50)

#3 Find the minimum sample size required to estimate the general distribution.
a <- asymptote(area, upper.degree = 10, estimator = 'glm', family = gaussian, max.asymptote = NA)

#4 Plot the mean collective area and rational function fit relative to the sample sizes.
ggplot(data = a$results, aes(x = x))+
  geom_pointrange(aes(y = y, ymin = y_lwr, ymax = y_upr)) + 
  geom_point(aes(y = y), size = 2) + 
  scale_x_continuous(breaks = seq(0, 15, 3), limits = c(2,15), name = "Animals tracked (n)") +
  scale_y_continuous(name = expression(Area~(km^2)), labels=function(x) x/1e6)
}
}
\references{
Shimada T, Thums M, Hamann M, Limpus CJ, Hays GC, FitzSimmons N, Wildermann NE, Duarte CD, Meekan MG (2021) 
Optimising sample sizes for animal distribution analysis using tracking data. 
\emph{Methods in Ecology and Evolution} 12(2):288-297 \doi{10.1111/2041-210X.13506}
}
\seealso{
\code{\link{boot_overlap}}, \code{\link{combn_overlap}}
}
\author{
Takahiro Shimada
}
