% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/semLearn.R
\name{SEMtree}
\alias{SEMtree}
\title{Tree-based structure learning methods}
\usage{
SEMtree(
  graph,
  data,
  seed,
  type = "ST",
  eweight = NULL,
  alpha = 0.05,
  verbose = FALSE,
  ...
)
}
\arguments{
\item{graph}{An igraph object.}

\item{data}{A matrix or data.frame. Rows correspond to subjects, and
columns to graph nodes (variables).}

\item{seed}{A vector of seed nodes.}

\item{type}{Tree-based structure learning method. Four algorithms 
are available:
\itemize{
\item "ST"(default). Steiner Tree (ST) identification via fast Kou's algorithm 
(Kou et al, 1981) connecting a set of seed nodes (called Terminal vertices)
with connector nodes (called Steiner vertices) from input graph as defined
in \code{graph} with minimal total distance on its edges. By default the edge
weights are based on the pairwise correlation, 1-abs(cor(j,k)). If input
graph has E(graph)$weight=1, and \code{eweight = "custom"}, ST seeks a minimum
subtree (i.e., the subtree with minimal number of edges).
\item "CAT". Causal additive trees (CAT) algorithm as in Jakobsen et al. 
(2022). The argument \code{graph} is set to NULL (i.e., no input graph is needed).
In the first step, a (univariate) generalized additive model (GAM) is employed
to estimate the residual variances, var(X(j) - [X(j)|X(k)]) for all j != k,
then use these to construct edge weights as inputs to the Chu–Liu–Edmonds’
algorithm (Chow and Liu, 1968) to recover the arborescence. Argument \code{seed}
must be specified to analyse a subset of nodes (variables) of interest.
\item "CPDAG". CLE algorithm for Skeleton Recovery and CPDAG
estimation as in Lou et al. (2021). Together with "CAT" algorithm, "CPDAG" is 
data-driven and the argument \code{graph} is set to NULL.
The key idea is to first recover the skeleton of the polytree by applying 
the CLE algorithm to the pairwise sample correlations of the data matrix.
After the skeleton is recovered, the set of all v-structures can be correctly
identified via a simple thresholding approach to pairwise sample correlations.
CPDAG can be found applying iteratively only Rule 1 of Meek (1995).
Argument \code{seed} must be specified to analyse a subset of nodes
(variables) of interest.
\item "MST". Minimum Spanning Tree (MST) identification via Prim's algorithm
(Prim, 1957). The latter finds the subset of edges that includes every vertex
of the graph (as defined in \code{graph}) such that the sum of the weights 
of the edges can be minimized. The argument \code{seed} is set to NULL (i.e.,
no seed nodes are needed).
}}

\item{eweight}{Edge weight type for igraph object can be externally derived
using \code{\link[SEMgraph]{weightGraph}} or from user-defined distances. 
This option determines the weight-to-distance transform. If set to:
\itemize{
\item "NULL" (default), edge weights will be internally computed
equal to 1 - abs(pairwise Pearson's correlation).
\item "kegg", repressing(-1), neutral(0) and activating(+1) kegg
interactions will be multiplied by "zsign" attributes, and positive
(i.e., concordant) values will be set to 1 (minimum distance), while
negative (i.e., discordant) values will be set to 2.
\item "zsign", all significant interactions (abs(zsign) > 0) will be
set to 1 (minimum distance), while non-significant (zsign=0) ones will
be set to 2.
\item "pvalue", edge p-value atributes will be transformed to the
inverse of negative base-10 logarithm, 1/(-log(E(graph)$pv)).
\item "custom", the algorithm will use the distance measure specified
by the user as "weight" edge attribute in the input graph.
}}

\item{alpha}{Threshold for rejecting a pair of node being independent in 
"CPDAG" algorithm. The latter implements a natural v-structure identification 
procedure by thresholding the pairwise sample correlations over all adjacent 
pairs of edges with some appropriate threshold. By default, 
\code{alpha = 0.05}.}

\item{verbose}{If TRUE, it shows the output tree (not recommended for large graphs).}

\item{...}{Currently ignored.}
}
\value{
An \code{igraph} object. If \code{type = "ST"}, seed nodes are 
colored in "aquamarine" and connectors in "white". If \code{type = "ST"} and
\code{type = "MST"}, edges are colored in "green" if not present in the input,
graph. If \code{type = "CPDAG"}, bidirected edges are colored in "black"
(if the algorithm is not able to establish the direction of the relationship
between x and y).
}
\description{
Four tree-based structure learning methods are implemented
with graph and data-driven algorithms.
}
\details{
A tree ia an acyclic graph with p vertices and p-1 edges. The graph method
refers to the  Steiner Tree (ST), a tree from an undirected graph that connect "seed"
with additional nodes in the "most compact" way possible. The data-driven methods
propose fast and scalable procedures based on Chu-Liu–Edmonds’ algorithm (CLE) to
recover a tree from a full graph. The first method, called Causal Additive Trees (CAT)
uses pairwise mutual weights as input for CLE algorithm to recover a directed tree
(an "arborescence"). The second one applies CLE algorithm for skeleton recovery and
extends the skeleton to a tree (a "polytree") represented by a Completed Partially
Directed Acyclic Graph (CPDAG). Finally, the Minimum Spanning Tree (MST) connecting
an undirected graph with minimal edge weights can be identified.
To note, if the input graph is a directed graph, ST and MST undirected trees are
converted in directed trees using the \code{\link[SEMgraph]{orientEdges}} function.
}
\examples{

\donttest{
# Nonparanormal(npn) transformation
als.npn <- transformData(alsData$exprs)$data

# graph-based trees
graph <- alsData$graph
seed <- V(graph)$name[sample(1:vcount(graph), 10)]
tree1 <- SEMtree(graph, als.npn, seed=seed, type="ST", verbose=TRUE)
tree2 <- SEMtree(graph, als.npn, seed=NULL, type="MST", verbose=TRUE)

# data-driven trees
V <- colnames(als.npn)[colnames(als.npn) \%in\% V(graph)$name]
tree3 <- SEMtree(NULL, als.npn, seed=V, type="CAT", verbose=TRUE)
tree4 <- SEMtree(NULL, als.npn, seed=V, type="CPDAG", alpha=0.05, verbose=TRUE)
}

}
\references{
Grassi M, Tarantino B (2023). SEMtree: tree-based structure learning methods
with structural equation models. 
Bioinformatics, 39 (6), 4829–4830 <https://doi.org/10.1093/bioinformatics/btad377>

Kou, L., Markowsky, G., Berman, L. (1981). A fast algorithm for Steiner trees. 
Acta Informatica 15, 141–145. <https://doi.org/10.1007/BF00288961>

Prim, R.C. (1957). Shortest connection networks and some generalizations Bell
System Technical Journal, 37 1389–1401. 

Chow, C.K. and Liu, C. (1968). Approximating discrete probability distributions with 
dependence trees. IEEE Transactions on Information Theory, 14(3):462–467.

Meek, C. (1995). Causal inference and causal explanation with background knowledge.
In Proceedings of the Eleventh conference on Uncertainty in artificial intelligence,
403–410.

Jakobsen, M, Shah, R., Bühlmann, P., Peters, J. (2022). 
Structure Learning for Directed Trees. arXiv:
<https://doi.org/10.48550/arxiv.2108.08871>.

Lou, X., Hu, Y., Li, X. (2022). Linear Polytree Structural Equation Models:
Structural Learning and Inverse Correlation Estimation. arXiv:
<https://doi.org/10.48550/arxiv.2107.10955>
}
\author{
Mario Grassi \email{mario.grassi@unipv.it}
}
