% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RoundViaDummy.R
\encoding{UTF8}
\name{RoundViaDummy}
\alias{RoundViaDummy}
\title{Small Count Rounding of Tabular Data}
\usage{
RoundViaDummy(
  data,
  freqVar,
  formula = NULL,
  roundBase = 3,
  singleRandom = FALSE,
  crossTable = TRUE,
  total = "Total",
  maxIterRows = 1000,
  maxIter = 1e+07,
  x = NULL,
  hierarchies = NULL,
  xReturn = FALSE,
  maxRound = roundBase - 1,
  zeroCandidates = FALSE,
  forceInner = FALSE,
  identifyNew = TRUE,
  step = 0,
  preRounded = NULL,
  leverageCheck = FALSE,
  easyCheck = TRUE,
  printInc = TRUE,
  rndSeed = 123,
  dimVar = NULL,
  ...
)
}
\arguments{
\item{data}{Input data as a data frame (inner cells)}

\item{freqVar}{Variable holding counts (name or number)}

\item{formula}{Model formula defining publishable cells. Will be used to calculate \code{x} (via \code{\link{ModelMatrix}}).
When NULL, x must be supplied.}

\item{roundBase}{Rounding base}

\item{singleRandom}{Single random draw when TRUE (instead of algorithm)}

\item{crossTable}{When TRUE, cross table in output and caculations via FormulaSums()}

\item{total}{String used to name totals}

\item{maxIterRows}{See details}

\item{maxIter}{Maximum number of iterations}

\item{x}{Dummy matrix defining publishable cells}

\item{hierarchies}{List of hierarchies, which can be converted by \code{\link{AutoHierarchies}}.
Thus, a single string as hierarchy input is assumed to be a total code.
Exceptions are \code{"rowFactor"} or \code{""}, which correspond to only using the categories in the data.}

\item{xReturn}{Dummy matrix in output when TRUE (as input parameter \code{x})}

\item{maxRound}{Inner cells contributing to original publishable cells equal to or less than maxRound will be rounded.}

\item{zeroCandidates}{When TRUE, inner cells in input with zero count (and multiple of roundBase when maxRound is in use)
contributing to publishable cells will be included as candidates to obtain roundBase value.
With vector input, the rule is specified individually for each cell.}

\item{forceInner}{When TRUE, all inner cells will be rounded. Use vector input to force individual cells to be rounded.
Can be combined with parameter zeroCandidates to allow zeros and roundBase multiples to be rounded up.}

\item{identifyNew}{When TRUE, new cells may be identified after initial rounding to ensure that no nonzero rounded
publishable cells are less than roundBase.}

\item{step}{When \code{step>1}, the original forward part of the algorithm is replaced by a kind of stepwise.
After \code{step} steps forward, backward steps may be performed. The \code{step} parameter is also used
for backward-forward iteration at the end of the algorithm; \code{step} backward steps may be performed.}

\item{preRounded}{A vector or a variable in data that contains a mixture of missing values and predetermined values of rounded inner cells.}

\item{leverageCheck}{When TRUE, all inner cells that depends linearly on the published cells and with small frequencies
(\code{<=maxRound}) will be rounded.
The computation of leverages can be very time and memory consuming.
The function \code{\link{Reduce0exact}} is called.
The default leverage limit is \code{0.999999}. Another limit can be sent as input instead of \code{TRUE}.
Checking is performed before and after (since new zeros) rounding. Extra iterations are performed when needed.}

\item{easyCheck}{A light version of the above leverage checking.
Checking is performed after rounding. Extra iterations are performed when needed.
\code{Reduce0exact} is called with \code{reduceByLeverage=FALSE} and \code{reduceByColSums=TRUE}.}

\item{printInc}{Printing iteration information to console when TRUE}

\item{rndSeed}{If non-NULL, a random generator seed to be used locally within the function without affecting the random value stream in R.}

\item{dimVar}{The main dimensional variables and additional aggregating variables. This parameter can be  useful when hierarchies and formula are unspecified.}

\item{...}{Further parameters sent to \code{\link{Hierarchies2ModelMatrix}} or \code{\link{HierarchiesAndFormula2ModelMatrix}}.
In particular, one can specify \code{removeEmpty=TRUE} to omit empty combinations.
The parameter \code{inputInOutput} can be used to specify whether to include codes from input.}
}
\value{
A list where the two first elements are two column matrices.
The first matrix consists of inner cells and the second of cells to be published.
In each matrix the first and the second column contains, respectively, original and rounded values.
By default the cross table is the third element of the output list.
}
\description{
Small count rounding via a dummy matrix and by an algorithm inspired by PLS
}
\details{
Small count rounding of necessary inner cells are performed so that all small frequencies of cross-classifications to be published
(publishable cells) are rounded. This is equivalent to changing micro data since frequencies of unique combinations are changed.
Thus, additivity and consistency are guaranteed. The matrix multiplication formula is:
\code{yPublish} \code{=} \code{t(x)} \code{\%*\%}  \code{yInner}, where \code{x} is the dummy matrix.
}
\note{
Iterations are needed since after initial rounding of identified cells, new cells are identified.
If cases of a high number of identified cells the algorithm can be too memory consuming (unless singleRandom=TRUE).
To avoid problems, not more than maxIterRows cells are rounded in each iteration.
The iteration limit (maxIter) is by default set to be high since a low number of maxIterRows may need a high number of iterations.
}
\examples{
# See similar and related examples in PLSrounding documentation
RoundViaDummy(SmallCountData("e6"), "freq")
RoundViaDummy(SmallCountData("e6"), "freq", formula = ~eu * year + geo)
RoundViaDummy(SmallCountData("e6"), "freq", hierarchies = 
   list(geo = c("EU", "@Portugal", "@Spain", "Iceland"), year = c("2018", "2019")))

RoundViaDummy(SmallCountData('z2'), 
              'ant', ~region + hovedint + fylke*hovedint + kostragr*hovedint, 10)
mf <- ~region*mnd + hovedint*mnd + fylke*hovedint*mnd + kostragr*hovedint*mnd
a <- RoundViaDummy(SmallCountData('z3'), 'ant', mf, 5)
b <- RoundViaDummy(SmallCountData('sosialFiktiv'), 'ant', mf, 4)
print(cor(b[[2]]),digits=12) # Correlation between original and rounded

# Demonstrate parameter leverageCheck 
# The 42nd inner cell must be rounded since it can be revealed from the published cells.
mf2 <- ~region + hovedint + fylke * hovedint + kostragr * hovedint
RoundViaDummy(SmallCountData("z2"), "ant", mf2, leverageCheck = FALSE)$yInner[42, ]
RoundViaDummy(SmallCountData("z2"), "ant", mf2, leverageCheck = TRUE)$yInner[42, ]

\dontrun{
# Demonstrate parameters maxRound, zeroCandidates and forceInner 
# by tabulating the inner cells that have been changed.
z4 <- SmallCountData("sosialFiktiv")
for (forceInner in c("FALSE", "z4$ant < 10")) 
  for (zeroCandidates in c(FALSE, TRUE)) 
    for (maxRound in c(2, 5)) {
      set.seed(123)
      a <- RoundViaDummy(z4, "ant", formula = mf, maxRound = maxRound, 
                         zeroCandidates = zeroCandidates, 
                         forceInner = eval(parse(text = forceInner)))
      change <- a$yInner[, "original"] != a$yInner[, "rounded"]
      cat("\n\n---------------------------------------------------\n")
      cat("      maxRound:", maxRound, "\n")
      cat("zeroCandidates:", zeroCandidates, "\n")
      cat("    forceInner:", forceInner, "\n\n")
      print(table(original = a$yInner[change, "original"], rounded = a$yInner[change, "rounded"]))
      cat("---------------------------------------------------\n")
    }
}
}
\seealso{
See the  user-friendly wrapper \code{\link{PLSrounding}}
and see \code{Round2} for rounding by other algorithm
}
