\name{condmap}
\alias{condmap}
\title{Produces a conditional 2D map from a fitted max-stable process}
\description{
  Produces a conditional 2D map from a fitted max-stable process.
}
\usage{
condmap(fitted, fix.coord, x, y, covariates = NULL, ret.per1 = 100,
ret.per2 = ret.per1, col = terrain.colors(64), plot.contour = TRUE,
\dots)
}
\arguments{
  \item{fitted}{An object of class \code{maxstab}. Most often, it will
    be the output of the function \code{\link{fitmaxstab}}.}
  \item{fix.coord}{The spatial coordinates of the location from which
    the conditional quantile is computed.}
  \item{x,y}{Numeric vector defining the grid at which the levels are
    computed.}
  \item{covariates}{An array specifying the covariates at each grid
    point defined by \code{x} and \code{y}. If \code{NULL}, no covariate
    is needed. See \code{\link{map}} to see how to build it.}
  \item{ret.per1,ret.per2}{Numerics giving the return period for which
    the quantile map is plotted. See details.}
  \item{col}{A list of colors such as that generated by 'rainbow',
    'heat.colors', 'topo.colors', 'terrain.colors' or similar
    functions.}
  \item{plot.contour}{Logical. If \code{TRUE} (default), contour lines
    are added to the plot.}
  \item{\dots}{Several arguments to be passed to the \code{\link{image}}
    function.}
}
\value{
  A plot. Additionally, a list with the details for plotting the map is
  returned invisibly.
}
\details{
  The function solves the following equation:

  \deqn{\Pr\left[Z(x_2) > z_2 | Z(x_1) > z_1 \right] =
    \frac{1}{T_2}}{Pr[Z(x_2) > z_2 | Z(x_1) > z_1] = 1 / T_2}
  where \eqn{z_1 = -1 / \log(1 - 1/T_1)}{z_1 = -1 / log(1 -
    1/T_1)}.

  In other words, it computes, given that at location \eqn{x_1}{x_1} we
  exceed the level \eqn{z_1}{z_1}, the levels which is expected to be
  exceeded in average every \eqn{T_2}{T_2} year.
}
\seealso{
  \code{\link{map}}, \code{\link{filled.contour}},
  \code{\link{heatmap}}, \code{\link{heat.colors}},
  \code{\link{topo.colors}}, \code{\link{terrain.colors}},
  \code{\link{rainbow}}
}
\examples{
##Define the coordinate of each location
n.site <- 30
locations <- matrix(runif(2*n.site, 0, 10), ncol = 2)
colnames(locations) <- c("lon", "lat")

##Simulate a max-stable process - with unit Frechet margins
data <- rmaxstab(50, locations, cov.mod = "whitmat", nugget = 0, range =
2, smooth = 1)

##Now define the spatial model for the GEV parameters
param.loc <- -10 - 4 * locations[,1] + locations[,2]^2
param.scale <- 5 + locations[,2] + locations[,1]^2 / 10
param.shape <- rep(.2, n.site)

##Transform the unit Frechet margins to GEV 
for (i in 1:n.site)
  data[,i] <- frech2gev(data[,i], param.loc[i], param.scale[i],
param.shape[i])

##Define a model for the GEV margins to be fitted
##shape ~ 1 stands for the GEV shape parameter is constant
##over the region
loc.form <- loc ~ lon + I(lat^2)
scale.form <- scale ~ lat + I(lon^2)
shape.form <- shape ~ 1

##  1- Fit a max-stable process
fitted <- fitmaxstab(data, locations, "whitmat", loc.form, scale.form,
                     shape.form, nugget = 0, std.err.type = "none")

cond.coord <- c(5.1, 5.1)
condmap(fitted, cond.coord, seq(0, 10, length = 25), seq(0,10, length
 =25), ret.per1 = 100, ret.per2 = 1.5)
points(t(cond.coord), pch = "*", col = 2, cex = 2)
}
\author{Mathieu Ribatet}
\keyword{hplot}
