test_that("check zooplankton and larvae feeding flux calculations", {

#-----------------------------------------------------------------------------------------------------------------
#
# This test uses a 'Testbed' version of the model in which all of the external drivers are set to constant values.
# We expect the model outputs to converge to a steady state under these conditions. Then we take the masses of the
# food web variables from the steady state model output, and manually derive the uptake fluxes between them.
# These manually derived fluxes are then compared with the fluxes generated in the 'flux_matrix' outputs from the model.
# So the test is checking that the flux calculations in the C-code of the package are correctly evaluated.
#
# The Testbed model is stored in /StrathE2E2/tests/testdata/models/
#
# The model has been run for >1000 years to ensure a steady state, and the end-state saved as initial conditions.
# So the model should be at steady state from time 0. In fact, the input csv file containing the initial conditions
# does not hold sufficient precision (number of decimal places) to ensure perfect steady state from the onset of a run.
# This is manifest as some tiny oscillations in the first year or so of a run before everything settles to its steady state.
# Hence we run the Testbed model for 10 years and then base the test on the final year of the run.
# 
# For convenience, the Testbed model uses the fitted parameters from the North Sea implementation of StrathE2E2.
# 
# An important point to note is that the temperature driving data for all three spatial compartmets of the Testbed
# model are set to 10C, wich matches the Q10 reference temperature in the fixed parameter inputs. This means that when manually
# calculating the uptake fluxes there is no need to compute any temperature corrections for the maximum uptake rate
# parameters. However, the C-code in the model IS doing the Q10 calculation, so this is an important aspect of the testing.
# As a consequence, it is very important not to chnage the temperature driving values in the Testbed model - they must be
# equal to the Q10 reference temperature value.
#
# This test applies the above approach to uptake by omnivorous zooplankton, carnivorous zooplankton,
# and the larvae of benthos and fish classes.
#
#-----------------------------------------------------------------------------------------------------------------
#
# The following non-exported functions in the package are needed for the test:
# source("internal.R")
# source("read_fitted_parameters.R")
# source("read_fixed_parameters.R")
# source("read_physics_drivers.R")
# source("read_physical_parameters.R")
#
#-----------------------------------------------------------------------------------------------------------------

#Some additional functions required for assembling the tests

#######################
#Function to pull the final year annual average mass of a variable out of the results
pull_final_year_mean<-function(results,vtoget){
vgot<-(elt(results$output, vtoget))
vmean<-mean(vgot[(length(vgot)-360):(length(vgot))])
vmean
}
#######################
#Function to pull flux data out of the flow matrix generated by the model
extract_model_flux<-function(matrixname,preyname,predname){
predi<-which(colnames(matrixname)==predname)
preyi<-which(colnames(matrixname)==preyname)
model_flux<-(matrixname[preyi,predi]/360)
model_flux
}
#######################
#Heterotrophic uptake function
f1<-function(a,b,k1,k2)
#          prey,pred,umax,hs
{
  return ((b*k1*a)/(a+k2))
}
#######################




#-----------------------------------------------------------------------------------------------------------------

#Run the Testbed model and pull out data from the results object

#Run the testbed model - assuming R home dir is "/GitLab/StrathE2E2/tests"
model<-e2e_read(model.name="Testbed",
                  model.variant="Const",
                  model.ident="base",
                  models.path="../testdata/models")
results <- e2e_run(model,nyears=20,csv.output=FALSE)

#Pull out the inshore and offshore final year data for phytoplankton and zoopkankton groups

#names(results$output[1:78]) # Lists the names of state varibales in the header

phyt_si<- pull_final_year_mean(results,"phyt_si")
detritus_si<- pull_final_year_mean(results,"detritus_si")
benthslar_i<- pull_final_year_mean(results,"benthslar_i")
benthclar_i<- pull_final_year_mean(results,"benthclar_i")
fishplar_i<- pull_final_year_mean(results,"fishplar_i")
fishdlar_i<- pull_final_year_mean(results,"fishdlar_i")
omni_i<-pull_final_year_mean(results,"omni_i")
carn_i<-pull_final_year_mean(results,"carn_i")

phyt_so<- pull_final_year_mean(results,"phyt_so")
detritus_so<- pull_final_year_mean(results,"detritus_so")
phyt_d<- pull_final_year_mean(results,"phyt_d")
detritus_d<- pull_final_year_mean(results,"detritus_d")
benthslar_o<- pull_final_year_mean(results,"benthslar_o")
benthclar_o<- pull_final_year_mean(results,"benthclar_o")
fishplar_o<- pull_final_year_mean(results,"fishplar_o")
fishdlar_o<- pull_final_year_mean(results,"fishdlar_o")
omni_o<- pull_final_year_mean(results,"omni_o")
carn_o<- pull_final_year_mean(results,"carn_o")

#Pull out the flow matrix from the results
flow_matrix<-elt(results$final.year.outputs,"flow_matrix_all_fluxes")

#colnames(flow_matrix)   #Lists the column names in the flow matrix
#rownames(flow_matrix)   #Lists the row names in the flow matrix

#-----------------------------------------------------------------------------------------------------------------




#Pull data out o fthe model object, ie data that have been assembled from the input csv files

#Pull out the model configuration to get areas and volumes

model.path<-model$setup$model.path
	read.model.setup(model.path)			# Models/Model/Variant/MODEL_SETUP.csv
	physical.parameters	<- read_physical_parameters(model.path)
	so_depth<-elt(physical.parameters,"so_depth")
	si_depth<-elt(physical.parameters,"si_depth")
	d_depth<-elt(physical.parameters,"d_depth")
        area_s0<-elt(physical.parameters,"x_area_s0")
        area_s1<-elt(physical.parameters,"x_area_s1")
        area_s2<-elt(physical.parameters,"x_area_s2")
        area_s3<-elt(physical.parameters,"x_area_s3")
        area_d0<-elt(physical.parameters,"x_area_d0")
        area_d1<-elt(physical.parameters,"x_area_d1")
        area_d2<-elt(physical.parameters,"x_area_d2")
        area_d3<-elt(physical.parameters,"x_area_d3")
shallowprop<-(area_s0+area_s1+area_s2+area_s3)
volume_so<-so_depth*(1-shallowprop)
volume_d<-d_depth*(1-shallowprop)
volume_si<-si_depth*(shallowprop)



# Pull out the uptake function parameters needed to manually calculate the fluxes

model.path<-model$setup$model.path
	read.model.setup(model.path)			# Models/Model/Variant/MODEL_SETUP.csv
	fitted.parameters	<- read_fitted_parameters(model.path)

u_omni<-elt(fitted.parameters,"u_omni")
h_omni<-elt(fitted.parameters,"h_omni")

PREF_phyt_omni<-elt(fitted.parameters,"PREF_phyt_omni")
PREF_det_omni<-elt(fitted.parameters,"PREF_det_omni")
PREF_benthslar_omni<-elt(fitted.parameters,"PREF_benthslar_omni")
PREF_benthclar_omni<-elt(fitted.parameters,"PREF_benthclar_omni")

#....

u_carn<-elt(fitted.parameters,"u_carn")
h_carn<-elt(fitted.parameters,"h_carn")

PREF_omni_carn<-elt(fitted.parameters,"PREF_omni_carn")
PREF_benthslar_carn<-elt(fitted.parameters,"PREF_benthslar_carn")
PREF_benthclar_carn<-elt(fitted.parameters,"PREF_benthclar_carn")
PREF_fishplar_carn<-elt(fitted.parameters,"PREF_fishplar_carn")
PREF_fishdlar_carn<-elt(fitted.parameters,"PREF_fishdlar_carn")

#....

u_benthslar<-elt(fitted.parameters,"u_benthslar")
h_benthslar<-elt(fitted.parameters,"h_benthslar")

PREF_phyt_benthslar<-elt(fitted.parameters,"PREF_phyt_benthslar")
PREF_det_benthslar<-elt(fitted.parameters,"PREF_det_benthslar")

#....

u_benthclar<-elt(fitted.parameters,"u_benthclar")
h_benthclar<-elt(fitted.parameters,"h_benthclar")

PREF_phyt_benthclar<-elt(fitted.parameters,"PREF_phyt_benthclar")
PREF_det_benthclar<-elt(fitted.parameters,"PREF_det_benthclar")

#....

u_fishplar<-elt(fitted.parameters,"u_fishplar")
h_fishplar<-elt(fitted.parameters,"h_fishplar")

PREF_omni_fishplar<-elt(fitted.parameters,"PREF_omni_fishplar")
PREF_benthslar_fishplar<-elt(fitted.parameters,"PREF_benthslar_fishplar")
PREF_benthclar_fishplar<-elt(fitted.parameters,"PREF_benthclar_fishplar")

#....

u_fishdlar<-elt(fitted.parameters,"u_fishdlar")
h_fishdlar<-elt(fitted.parameters,"h_fishdlar")

PREF_omni_fishdlar<-elt(fitted.parameters,"PREF_omni_fishdlar")
PREF_benthslar_fishdlar<-elt(fitted.parameters,"PREF_benthslar_fishdlar")
PREF_benthclar_fishdlar<-elt(fitted.parameters,"PREF_benthclar_fishdlar")

#-----------------------------------------------------------------------------------------------------------------




#Now manually calculate the fluxes betwen prey and predators

# Calculate the vertical distributions of consumers in the offshore

 prop_omni_surfo<-         ((u_omni*PREF_det_omni*detritus_so)+
                            (u_omni*PREF_phyt_omni*phyt_so))/
                           ((u_omni*PREF_det_omni*detritus_so)+
                            (u_omni*PREF_det_omni*detritus_d)+
                            (u_omni*PREF_phyt_omni*phyt_so)+
                            (u_omni*PREF_phyt_omni*phyt_d))
 prop_omni_deep<-(1-prop_omni_surfo)

 prop_benthslar_surfo<-         ((u_benthslar*PREF_det_benthslar*detritus_so)+
                            (u_benthslar*PREF_phyt_benthslar*phyt_so))/
                           ((u_benthslar*PREF_det_benthslar*detritus_so)+
                            (u_benthslar*PREF_det_benthslar*detritus_d)+
                            (u_benthslar*PREF_phyt_benthslar*phyt_so)+
                            (u_benthslar*PREF_phyt_benthslar*phyt_d))
 prop_benthslar_deep<-(1-prop_benthslar_surfo)

 prop_benthclar_surfo<-         ((u_benthclar*PREF_det_benthclar*detritus_so)+
                            (u_benthclar*PREF_phyt_benthclar*phyt_so))/
                           ((u_benthclar*PREF_det_benthclar*detritus_so)+
                            (u_benthclar*PREF_det_benthclar*detritus_d)+
                            (u_benthclar*PREF_phyt_benthclar*phyt_so)+
                            (u_benthclar*PREF_phyt_benthclar*phyt_d))
 prop_benthclar_deep<-(1-prop_benthclar_surfo)


#Now calculate the fluxes

calc_flux_phyt_omni_o<- f1(phyt_so,(omni_o*prop_omni_surfo),u_omni*PREF_phyt_omni,h_omni*(volume_so)) +
                   f1(phyt_d,(omni_o*prop_omni_deep),u_omni*PREF_phyt_omni,h_omni*(volume_d))
calc_flux_phyt_omni_i<- f1(phyt_si,(omni_i),u_omni*PREF_phyt_omni,h_omni*(volume_si))

calc_flux_phyt_omni_w <- calc_flux_phyt_omni_o + calc_flux_phyt_omni_i

model_flux_phyt_omni<-extract_model_flux(flow_matrix,"phyt","omnivzoo")

#calc_flux_phyt_omni_w
#model_flux_phyt_omni

#............................

calc_flux_det_omni_o<- f1(detritus_so,(omni_o*prop_omni_surfo),u_omni*PREF_det_omni,h_omni*(volume_so)) +
                   f1(detritus_d,(omni_o*prop_omni_deep),u_omni*PREF_det_omni,h_omni*(volume_d))
calc_flux_det_omni_i<- f1(detritus_si,(omni_i),u_omni*PREF_det_omni,h_omni*(volume_si))

calc_flux_det_omni_w <- calc_flux_det_omni_o + calc_flux_det_omni_i

model_flux_det_omni<-extract_model_flux(flow_matrix,"wcdetritus","omnivzoo")

#calc_flux_det_omni_w
#model_flux_det_omni

#............................

calc_flux_benthslar_omni_o<- f1(benthslar_o,(omni_o),u_omni*PREF_benthslar_omni,h_omni*(volume_so+volume_d))
calc_flux_benthslar_omni_i<- f1(benthslar_i,(omni_i),u_omni*PREF_benthslar_omni,h_omni*(volume_si))

calc_flux_benthslar_omni_w <- calc_flux_benthslar_omni_o + calc_flux_benthslar_omni_i

model_flux_benthslar_omni<-extract_model_flux(flow_matrix,"benthslar","omnivzoo")

#calc_flux_benthslar_omni_w
#model_flux_benthslar_omni

#............................

calc_flux_benthclar_omni_o<- f1(benthclar_o,(omni_o),u_omni*PREF_benthclar_omni,h_omni*(volume_so+volume_d))
calc_flux_benthclar_omni_i<- f1(benthclar_i,(omni_i),u_omni*PREF_benthclar_omni,h_omni*(volume_si))

calc_flux_benthclar_omni_w <- calc_flux_benthclar_omni_o + calc_flux_benthclar_omni_i

model_flux_benthclar_omni<-extract_model_flux(flow_matrix,"benthclar","omnivzoo")

#calc_flux_benthclar_omni_w
#model_flux_benthclar_omni

#............................
#............................

calc_flux_phyt_benthslar_o<- f1(phyt_so,(benthslar_o*prop_benthslar_surfo),u_benthslar*PREF_phyt_benthslar,h_benthslar*(volume_so)) +
                   f1(phyt_d,(benthslar_o*prop_benthslar_deep),u_benthslar*PREF_phyt_benthslar,h_benthslar*(volume_d))
calc_flux_phyt_benthslar_i<- f1(phyt_si,(benthslar_i),u_benthslar*PREF_phyt_benthslar,h_benthslar*(volume_si))

calc_flux_phyt_benthslar_w <- calc_flux_phyt_benthslar_o + calc_flux_phyt_benthslar_i

model_flux_phyt_benthslar<-extract_model_flux(flow_matrix,"phyt","benthslar")

#calc_flux_phyt_benthslar_w
#model_flux_phyt_benthslar

#............................

calc_flux_det_benthslar_o<- f1(detritus_so,(benthslar_o*prop_benthslar_surfo),u_benthslar*PREF_det_benthslar,h_benthslar*(volume_so)) +
                   f1(detritus_d,(benthslar_o*prop_benthslar_deep),u_benthslar*PREF_det_benthslar,h_benthslar*(volume_d))
calc_flux_det_benthslar_i<- f1(detritus_si,(benthslar_i),u_benthslar*PREF_det_benthslar,h_benthslar*(volume_si))

calc_flux_det_benthslar_w <- calc_flux_det_benthslar_o + calc_flux_det_benthslar_i

model_flux_det_benthslar<-extract_model_flux(flow_matrix,"wcdetritus","benthslar")

#calc_flux_det_benthslar_w
#model_flux_det_benthslar

#............................
#............................

calc_flux_phyt_benthclar_o<- f1(phyt_so,(benthclar_o*prop_benthclar_surfo),u_benthclar*PREF_phyt_benthclar,h_benthclar*(volume_so)) +
                   f1(phyt_d,(benthclar_o*prop_benthclar_deep),u_benthclar*PREF_phyt_benthclar,h_benthclar*(volume_d))
calc_flux_phyt_benthclar_i<- f1(phyt_si,(benthclar_i),u_benthclar*PREF_phyt_benthclar,h_benthclar*(volume_si))

calc_flux_phyt_benthclar_w <- calc_flux_phyt_benthclar_o + calc_flux_phyt_benthclar_i

model_flux_phyt_benthclar<-extract_model_flux(flow_matrix,"phyt","benthclar")

#calc_flux_phyt_benthclar_w
#model_flux_phyt_benthclar

#............................

calc_flux_det_benthclar_o<- f1(detritus_so,(benthclar_o*prop_benthclar_surfo),u_benthclar*PREF_det_benthclar,h_benthclar*(volume_so)) +
                   f1(detritus_d,(benthclar_o*prop_benthclar_deep),u_benthclar*PREF_det_benthclar,h_benthclar*(volume_d))
calc_flux_det_benthclar_i<- f1(detritus_si,(benthclar_i),u_benthclar*PREF_det_benthclar,h_benthclar*(volume_si))

calc_flux_det_benthclar_w <- calc_flux_det_benthclar_o + calc_flux_det_benthclar_i

model_flux_det_benthclar<-extract_model_flux(flow_matrix,"wcdetritus","benthclar")

#calc_flux_det_benthclar_w
#model_flux_det_benthclar

#............................
#............................

calc_flux_omni_carn_o<- f1(omni_o,(carn_o),u_carn*PREF_omni_carn,h_carn*(volume_so+volume_d))
calc_flux_omni_carn_i<- f1(omni_i,(carn_i),u_carn*PREF_omni_carn,h_carn*(volume_si))

calc_flux_omni_carn_w <- calc_flux_omni_carn_o + calc_flux_omni_carn_i

model_flux_omni_carn<-extract_model_flux(flow_matrix,"omnivzoo","carnzoo")

#calc_flux_omni_carn_w
#model_flux_omni_carn

#............................

calc_flux_benthslar_carn_o<- f1(benthslar_o,(carn_o),u_carn*PREF_benthslar_carn,h_carn*(volume_so+volume_d))
calc_flux_benthslar_carn_i<- f1(benthslar_i,(carn_i),u_carn*PREF_benthslar_carn,h_carn*(volume_si))

calc_flux_benthslar_carn_w <- calc_flux_benthslar_carn_o + calc_flux_benthslar_carn_i

model_flux_benthslar_carn<-extract_model_flux(flow_matrix,"benthslar","carnzoo")

#calc_flux_benthslar_carn_w
#model_flux_benthslar_carn

#............................

calc_flux_benthclar_carn_o<- f1(benthclar_o,(carn_o),u_carn*PREF_benthclar_carn,h_carn*(volume_so+volume_d))
calc_flux_benthclar_carn_i<- f1(benthclar_i,(carn_i),u_carn*PREF_benthclar_carn,h_carn*(volume_si))

calc_flux_benthclar_carn_w <- calc_flux_benthclar_carn_o + calc_flux_benthclar_carn_i

model_flux_benthclar_carn<-extract_model_flux(flow_matrix,"benthclar","carnzoo")

#calc_flux_benthclar_carn_w
#model_flux_benthclar_carn

#............................

calc_flux_fishplar_carn_o<- f1(fishplar_o,(carn_o),u_carn*PREF_fishplar_carn,h_carn*(volume_so+volume_d))
calc_flux_fishplar_carn_i<- f1(fishplar_i,(carn_i),u_carn*PREF_fishplar_carn,h_carn*(volume_si))

calc_flux_fishplar_carn_w <- calc_flux_fishplar_carn_o + calc_flux_fishplar_carn_i

model_flux_fishplar_carn<-extract_model_flux(flow_matrix,"pfishlar","carnzoo")

#calc_flux_fishplar_carn_w
#model_flux_fishplar_carn

#............................

calc_flux_fishdlar_carn_o<- f1(fishdlar_o,(carn_o),u_carn*PREF_fishdlar_carn,h_carn*(volume_so+volume_d))
calc_flux_fishdlar_carn_i<- f1(fishdlar_i,(carn_i),u_carn*PREF_fishdlar_carn,h_carn*(volume_si))

calc_flux_fishdlar_carn_w <- calc_flux_fishdlar_carn_o + calc_flux_fishdlar_carn_i

model_flux_fishdlar_carn<-extract_model_flux(flow_matrix,"dfishlar","carnzoo")

#calc_flux_fishdlar_carn_w
#model_flux_fishdlar_carn

#............................
#............................

calc_flux_omni_fishplar_o<- f1(omni_o,(fishplar_o),u_fishplar*PREF_omni_fishplar,h_fishplar*(volume_so+volume_d))
calc_flux_omni_fishplar_i<- f1(omni_i,(fishplar_i),u_fishplar*PREF_omni_fishplar,h_fishplar*(volume_si))

calc_flux_omni_fishplar_w <- calc_flux_omni_fishplar_o + calc_flux_omni_fishplar_i

model_flux_omni_fishplar<-extract_model_flux(flow_matrix,"omnivzoo","pfishlar")

#calc_flux_omni_fishplar_w
#model_flux_omni_fishplar

#............................

calc_flux_benthslar_fishplar_o<- f1(benthslar_o,(fishplar_o),u_fishplar*PREF_benthslar_fishplar,h_fishplar*(volume_so+volume_d))
calc_flux_benthslar_fishplar_i<- f1(benthslar_i,(fishplar_i),u_fishplar*PREF_benthslar_fishplar,h_fishplar*(volume_si))

calc_flux_benthslar_fishplar_w <- calc_flux_benthslar_fishplar_o + calc_flux_benthslar_fishplar_i

model_flux_benthslar_fishplar<-extract_model_flux(flow_matrix,"benthslar","pfishlar")

#calc_flux_benthslar_fishplar_w
#model_flux_benthslar_fishplar

#............................

calc_flux_benthclar_fishplar_o<- f1(benthclar_o,(fishplar_o),u_fishplar*PREF_benthclar_fishplar,h_fishplar*(volume_so+volume_d))
calc_flux_benthclar_fishplar_i<- f1(benthclar_i,(fishplar_i),u_fishplar*PREF_benthclar_fishplar,h_fishplar*(volume_si))

calc_flux_benthclar_fishplar_w <- calc_flux_benthclar_fishplar_o + calc_flux_benthclar_fishplar_i

model_flux_benthclar_fishplar<-extract_model_flux(flow_matrix,"benthclar","pfishlar")

#calc_flux_benthclar_fishplar_w
#model_flux_benthclar_fishplar

#............................
#............................

calc_flux_omni_fishdlar_o<- f1(omni_o,(fishdlar_o),u_fishdlar*PREF_omni_fishdlar,h_fishdlar*(volume_so+volume_d))
calc_flux_omni_fishdlar_i<- f1(omni_i,(fishdlar_i),u_fishdlar*PREF_omni_fishdlar,h_fishdlar*(volume_si))

calc_flux_omni_fishdlar_w <- calc_flux_omni_fishdlar_o + calc_flux_omni_fishdlar_i

model_flux_omni_fishdlar<-extract_model_flux(flow_matrix,"omnivzoo","dfishlar")

#calc_flux_omni_fishdlar_w
#model_flux_omni_fishdlar

#............................

calc_flux_benthslar_fishdlar_o<- f1(benthslar_o,(fishdlar_o),u_fishdlar*PREF_benthslar_fishdlar,h_fishdlar*(volume_so+volume_d))
calc_flux_benthslar_fishdlar_i<- f1(benthslar_i,(fishdlar_i),u_fishdlar*PREF_benthslar_fishdlar,h_fishdlar*(volume_si))

calc_flux_benthslar_fishdlar_w <- calc_flux_benthslar_fishdlar_o + calc_flux_benthslar_fishdlar_i

model_flux_benthslar_fishdlar<-extract_model_flux(flow_matrix,"benthslar","dfishlar")

#calc_flux_benthslar_fishdlar_w
#model_flux_benthslar_fishdlar

#............................

calc_flux_benthclar_fishdlar_o<- f1(benthclar_o,(fishdlar_o),u_fishdlar*PREF_benthclar_fishdlar,h_fishdlar*(volume_so+volume_d))
calc_flux_benthclar_fishdlar_i<- f1(benthclar_i,(fishdlar_i),u_fishdlar*PREF_benthclar_fishdlar,h_fishdlar*(volume_si))

calc_flux_benthclar_fishdlar_w <- calc_flux_benthclar_fishdlar_o + calc_flux_benthclar_fishdlar_i

model_flux_benthclar_fishdlar<-extract_model_flux(flow_matrix,"benthclar","dfishlar")

#calc_flux_benthclar_fishdlar_w
#model_flux_benthclar_fishdlar

#............................


#Implement testthat checks...

expect_equal(model_flux_phyt_omni,calc_flux_phyt_omni_w , 1e-7)
expect_equal(model_flux_det_omni,calc_flux_det_omni_w , 1e-7)
expect_equal(model_flux_benthslar_omni,calc_flux_benthslar_omni_w , 1e-7)
expect_equal(model_flux_benthclar_omni,calc_flux_benthclar_omni_w , 1e-7)

expect_equal(model_flux_omni_carn,calc_flux_omni_carn_w , 1e-7)
expect_equal(model_flux_benthslar_carn,calc_flux_benthslar_carn_w , 1e-7)
expect_equal(model_flux_benthclar_carn,calc_flux_benthclar_carn_w , 1e-7)
expect_equal(model_flux_fishplar_carn,calc_flux_fishplar_carn_w , 1e-7)
expect_equal(model_flux_fishdlar_carn,calc_flux_fishdlar_carn_w , 1e-7)

expect_equal(model_flux_phyt_benthslar,calc_flux_phyt_benthslar_w , 1e-7)
expect_equal(model_flux_det_benthslar,calc_flux_det_benthslar_w , 1e-6)

expect_equal(model_flux_phyt_benthclar,calc_flux_phyt_benthclar_w , 1e-7)
expect_equal(model_flux_det_benthclar,calc_flux_det_benthclar_w , 1e-7)

expect_equal(model_flux_omni_fishplar,calc_flux_omni_fishplar_w , 1e-7)
expect_equal(model_flux_benthslar_fishplar,calc_flux_benthslar_fishplar_w , 1e-7)
expect_equal(model_flux_benthclar_fishplar,calc_flux_benthclar_fishplar_w , 1e-7)

expect_equal(model_flux_omni_fishdlar,calc_flux_omni_fishdlar_w , 1e-7)
expect_equal(model_flux_benthslar_fishdlar,calc_flux_benthslar_fishdlar_w , 1e-7)
expect_equal(model_flux_benthclar_fishdlar,calc_flux_benthclar_fishdlar_w , 1e-7)

})

