% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/as.parameters.R
\name{as.parameters}
\alias{as.parameters}
\alias{as.parameters.numeric}
\alias{as.parameters.matrix}
\alias{as.parameters.list}
\alias{as.parameters.data.frame}
\title{Converting to parameters-objects}
\usage{
as.parameters(..., distr = NULL)

\method{as.parameters}{numeric}(..., distr)

\method{as.parameters}{matrix}(x, distr, ...)

\method{as.parameters}{list}(x, distr, ...)

\method{as.parameters}{data.frame}(x, formula, distr, ...)
}
\arguments{
\item{...}{parameters of distribution. This can be named vectors or lists, matrices, or data.frames. See examples below.}

\item{distr}{character giving the distribution. Function of name
q\"distr\" has to be available.}

\item{x}{numeric vector, matrix, list, or data.frame of parameters.}

\item{formula}{if \code{x} is data.frame a formula has to be given.}
}
\value{
object of class parameters, see parameters help page.
}
\description{
Convert vector, matrix, list, or data.frame to parameters-objects.
}
\section{Methods (by class)}{
\itemize{
\item \code{numeric}: as.parameters for numeric data vectors

\item \code{matrix}: as.parameters for numeric data matrices

\item \code{list}: as.parameters for numeric data lists

\item \code{data.frame}: as.parameters for numeric data.frames
}}

\examples{
# Vector input:
as.parameters(loc = 3, scale = 2, shape = .4, distr = "gev")
as.parameters(c(loc = 3, scale = 2, shape = .4), distr = "gev")

# Names can be shortened if unambiguous:
as.parameters(l = 3, sc = 2, sh = .4, distr = "gev")
as.parameters(m = 3, s = 1, distr = "norm")

# Wrong or ambiguous names lead to errors!
\dontrun{
as.parameters(l = 3, s = 2, s = .4, distr = "gev")
as.parameters(loc2 = 3, scale = 2, shape = .4, distr = "gev")
}

# If no names are given, a warning is given and they are guessed for gev, gpd, gum, and ln3.
as.parameters(3, 2, .4, distr = "gev")
as.parameters(c(3, 2, .4), distr = "gev")
\dontrun{
as.parameters(3, 2, .2, .4, distr = "gev") #=> doesn't work
}

# Matrix input:
# Parameters in matrices must have either matching rownames or colnames!
as.parameters(cbind(loc = 10, scale = 4, shape = seq(0, .4, .1)), distr = "gev")
as.parameters(rbind(loc = 10, scale = 4, shape = seq(0, .4, .1)), distr = "ln3")

# If no names are given, a guess is made based on number of rows
# or cols according to distribution (and a warning is given).
as.parameters(matrix(1:9, nr = 3), distr = "gev")
as.parameters(matrix(1:8, nc = 2), distr = "gum")


# The same principles apply for list input and data.frames:

# List input:
as.parameters(list(list(mean = 2, sd = 1), list(mean = 0, sd = 1)), distr = "norm")
as.parameters(list(c(m = 2, s = 1), c(m = 0, s = 1)), distr = "norm")
as.parameters(list(c(loc = 2, scale = 1), c(0, 1)), distr = "gum")
\dontrun{
as.parameters(list(c(loc = 2, scale = 1), c(0, 1, 2)), distr = "gum")
}

# Dataframe input:
xdat <- data.frame(station = c(1, 2), mean = c(2, 0), sd = c(1, 1))
as.parameters(xdat, cbind(mean, sd) ~ station, distr = "norm")
as.parameters(xdat, . ~ station, distr = "norm")
as.parameters(xdat, cbind(mean, sd) ~ ., distr = "norm")

xdat <- data.frame(station = c(1, 2), m = c(2, 0), s = c(1, 1))
as.parameters(xdat, cbind(m, s) ~ station, distr = "norm")
\dontrun{
as.parameters(xdat, cbind(m, s) ~ station, distr = "gev")
}

###

# Results of as.parameters can be used in the normal TLMoments-scheme:
# they can be transfered to quantiles or to TLMoments.

xdat <- data.frame(station = c(1, 2), mean = c(2, 0), sd = c(1, 1))
quantiles(as.parameters(xdat, cbind(mean, sd) ~ ., distr = "norm"), c(.99))

# quantile estimation
p <- as.parameters(loc = 3, scale = 2, shape = .4, distr = "gev")
quantiles(p, c(.9, .95))
p <- as.parameters(cbind(loc = 10, scale = 4, shape = seq(0, .4, .1)), distr = "gev")
quantiles(p, c(.9, .95))
p <- as.parameters(list(list(mean = 2, sd = 1), list(mean = 0, sd = 1)), distr = "norm")
quantiles(p, c(.95, .975))

# With magrittr
library(magrittr)
as.parameters(loc = 3, scale = 2, shape = .4, distr = "gev") \%>\% quantiles(c(.9, .99))
}
\seealso{
\code{\link{parameters}}
}
