#' @title Tensor time series data generation.
#' @description Function to generate a random sample of time series tensor factor model, based on econometrics assumptions.
#' @details Input tensor dimension and rank of core tensor, return a sample of tensor time series generated by factor model.
#' @param K The number of modes for the tensor time series.
#' @param n Length of time series.
#' @param d Dimensions of each mode of the tensor, written in a vector of length \code{K}.
#' @param r Rank of core tensors, written in a vector of length \code{K}.
#' @param re Dimension of the cross-sectional common error.
#' @param eta Quantities controlling factor strengths in each factor loading matrix, written in a list of \code{K} vectors.
#' @param u Quantities controlling range of elements in each factor loading matrix, written in a list of \code{K} vectors.
#' @return A list containing the following: \cr
#' \code{X}: the generated tensor time series, stored in a 'Tensor' object defined in \pkg{rTensor}, where mode-1 is the time mode \cr
#' \code{A}: a list of \code{K} factor loading matrices \cr
#' \code{F_ts}: time series of core tensor, stored in a 'Tensor' object, where mode-1 is the time mode \cr
#' \code{E_ts}: time series of error tensor, stored in a 'Tensor' object, where mode-1 is the time mode \cr
#' @export
#' @import rTensor MASS stats
#' @importFrom pracma randortho
#' @examples
#' \donttest{
#' set.seed(10)
#' K = 2
#' n = 100
#' d = c(40,40)
#' r = c(2,2)
#' re = 10
#' eta = list(c(0,0),c(0,0))
#' u = list(c(-2,2),c(-2,2))
#' Data_test = tensor_data_gen(K,n,d,r,re,eta,u)
#'
#' X = Data_test$X
#' A = Data_test$A
#' F_ts = Data_test$F_ts
#' E_ts = Data_test$E_ts
#'
#' X@modes
#' F_ts@modes
#' E_ts@modes
#' dim(A[[1]])
#' }













#################################### Function to Generate data #########################################
## This function generates a tensor time series in the form of a 'Tensor' object
tensor_data_gen = function(    K              # the number of modes for the tensor time series
                             , n              # length of time series
                             , d              # dimensions in each mode, written in a vector of length K. e.g. c(40,40)
                             , r              # rank of core tensors, written in a vector of length K. e.g. c(2,2)
                             , re             # dimension of the cross-sectional common error
                             , eta            # quantities controlling factor strengths in A_k, written in a list of K vectors. e.g. eta = list(c(0,0.2),c(0,0.2)) for Setting (I). See the paper for more details
                             , u)             # quantities controlling range of elements in A_k, written in a list of K vectors. e.g. eta = list(c(-2,2),c(-2,2)) for Setting (a). See the paper for more details

  # output : a list of four elements:
  # output$X: a 'Tensor' object, where mode-1 is the time mode.
  # output$A: a list of K factor loading matrices
  # output$F: time series of core tensor
  # output$E: time series of error tensor
{
  # Generate non-zero mean
  mu = as.tensor(array(rnorm(prod(d)),d))


  ## Generate the core tensor (independent AR process)
  f_series = as.tensor(array(numeric(),c(n,r)))
  unfold_f = unfold(f_series,row_idx = 1, col_idx = 2:(K+1))
  for (j in 1:prod(r))
  {
    unfold_f[,j] = arima.sim(n = n, n.start = 6, list(ar = c(0.7,0.3,-0.4,0.2,-0.1)))
  }
  f_series = fold(unfold_f,row_idx = 1, col_idx = 2:(K+1),modes = c(n,r))


  ## Generate factor loading matrices
  A = list()

  for (k in 1:K)
  {
    U_k = matrix(runif(d[k] * r[k], u[[k]][1], u[[k]][2]), d[k], r[k])
    R_k = if (r[k] == 1) matrix(d[k]^-eta[[k]]) else {diag(d[k]^-eta[[k]])}
    A_k = U_k %*% R_k
    A[[k]] = A_k
  }


  ## Generate the error tensors
  main_d = 1 # Error generation dimension
  common_e = array(numeric(),c(n,re))          # cross-sectional common error
  idio_e = as.tensor(array(numeric(),c(n,d)))

  d_k = d[main_d]
  d_minus_k = prod(d)/d_k

  # generate cross-sectional common error
  for (i in 1:re)
  {
    common_e[,i] = arima.sim(n = n, n.start = 6, list(ar = c(-0.7,-0.3,-0.4,0.2,0.1)))
  }

  # idiosyncratic error
  unfold_idio_e = unfold(idio_e,row_idx = 1, col_idx = 2:(K+1))
  for (j in 1:prod(d))
  {
    unfold_idio_e[,j] = arima.sim(n = n, n.start = 6, list(ar = c(0.8,0.4, -0.4,0.2,-0.1)))
  }
  idio_e = fold(unfold_idio_e,row_idx = 1, col_idx = 2:(K+1),modes = c(n,d))


  # sparse loading for cross-sectional common error
  Psi_k = array(numeric(),c(d_minus_k,d_k,re))
  for (i in 1:d_minus_k)
  {
    Psi_k[i,,] = rbinom(1,1,0.3) * matrix(rnorm(d_k * re), d_k, re)  # Sparse
  }

  # within-sectional dependence for idiosyncratic error
  Sigma_idio = array(numeric(),c(d_minus_k,d_k,d_k))
  for (i in 1:d_minus_k)
  {
    random_evectors = randortho(d_k)
    Sigma_idio[i,,] = random_evectors %*% diag(runif(d_k,1,3)) %*% t(random_evectors)
  }

  # generate the whole error tensor time series
  e_series = as.tensor(array(numeric(),c(n,d)))
  unfold_e = unfold(e_series,row_idx = c(main_d + 1, 1), col_idx = c(1:(K+1))[-c(1, main_d + 1)])
  unfold_idio_e_d_minus_k = unfold(idio_e,row_idx = c(main_d + 1, 1), col_idx = c(1:(K+1))[-c(1, main_d + 1)])
  for (t in 1:n)
  {
    for (j in 1:d_minus_k)
    {
      eigen_Sigma <- eigen(Sigma_idio[j,,])
      V <- eigen_Sigma$vectors
      sqrt_Sigma_idio <- V %*% diag(sqrt(eigen_Sigma$values)) %*% t(V)
      unfold_e[((t-1)*d_k + 1):(t*d_k)][,j] = Psi_k[j,,] %*% common_e[t,] + sqrt_Sigma_idio %*% unfold_idio_e_d_minus_k[((t-1)*d_k + 1):(t*d_k),][,j]@data ## The d_k times d_minus_k matrix at the t time, and the jth column


    }
  }
  e_series = fold(unfold_e,row_idx = c(main_d + 1, 1), col_idx = c(1:(K+1))[-c(1, main_d + 1)], modes = c(n,d))


  # generate the tensor time series X
  X = as.tensor(array(numeric(),c(n,d)))

  # For now, we can only generate the data for K = 2 or K = 3, but can be easily extended with more codes if needed.
  if (K == 2)
  {
    for (t in 1:n)
    {
      temp_tensor = as.tensor(array(f_series[t,,]@data,r))
      for (k in 1:K)
      {
        temp_tensor = ttm(tnsr = temp_tensor, mat = A[[k]], m = k)
      }
      X[t,,] = temp_tensor + e_series[t,,]  + mu
    }
  } else if (K == 3)
  {
    for (t in 1:n)
    {
      temp_tensor = as.tensor(array(f_series[t,,,]@data,r))
      for (k in 1:K)
      {
        temp_tensor = ttm(tnsr = temp_tensor, mat = A[[k]], m = k)
      }
      X[t,,,] = temp_tensor + e_series[t,,,]  + mu
    }
  } else
  {
    print('K too large, need to modify last part of the function')
  }


  return(list('X'=X,'A'=A,'F_ts'=f_series,'E_ts'=e_series))
}
