/******************************************************************************
 * @section MODIFICATION
 *
 * Modification by Ruida Zhong for the R package VICmodel on Sep 18th, 2018:
 * Macro `ERROR` is rename by `VIC_ERROR` to avoid redefine.
 *
 * @section DESCRIPTION
*
* Calculates the interception and subsequent release of by the forest canopy
* using an energy balance approach.
*
* @section LICENSE
*
* The Variable Infiltration Capacity (VIC) macroscale hydrological model
* Copyright (C) 2016 The Computational Hydrology Group, Department of Civil
* and Environmental Engineering, University of Washington.
*
* The VIC model is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License
* as published by the Free Software Foundation; either version 2
* of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License along with
* this program; if not, write to the Free Software Foundation, Inc.,
* 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
******************************************************************************/

#include <vic_run.h>

/******************************************************************************
* @brief    Calculate snow interception and release by the canopy
******************************************************************************/
int
snow_intercept(double             Dt,
               double             F,
               double             LAI,
               double             Le,
               double             LongOverIn, // incominf LW from sky
               double             LongUnderOut, // incoming LW from understroy
               double             MaxInt, // maximum interception capacity
               double             ShortOverIn, // incoming SW to overstory
               double             Tcanopy, // canopy air temperature
               double             bare_albedo, // albedo of snow-free ground
               double            *AdvectedEnergy,
               double            *AlbedoOver, // overstory albedo
               double            *IntRain, // intercepted rain
               double            *IntSnow, // intercepted snow
               double            *LatentHeat, // latent heat from overstory
               double            *LatentHeatSub, // sublimation heat from overstory
               double            *LongOverOut, // longwave emitted by canopy
               double            *MeltEnergy,
               double            *NetLongOver,
               double            *NetShortOver,
               double            *Ra,
               double            *Ra_used,
               double            *RainFall,
               double            *SensibleHeat,
               double            *SnowFall,
               double            *Tfoliage,
               bool              *Tfoliage_fbflag,
               unsigned          *Tfoliage_fbcount,
               double            *TempIntStorage,
               double            *VaporMassFlux,
               double            *Wind,
               double            *displacement,
               double            *ref_height,
               double            *roughness,
               double            *root,
               int                UnderStory,
               int                band,
               int                iveg,
               int                month,
               int                hidx,
               unsigned short     veg_class,
               double            *CanopLayerBnd,
               double            *dryFrac,
               force_data_struct *force,
               layer_data_struct *layer,
               soil_con_struct   *soil_con,
               veg_var_struct    *veg_var)
{
    extern option_struct     options;
    extern parameters_struct param;

    /* double AdvectedEnergy; */         /* Energy advected by the rain (W/m2) */
    double                   BlownSnow; /* Depth of snow blown of the canopy (m) */
    double                   DeltaSnowInt; /* Change in the physical swe of snow
                                              interceped on the branches. (m) */
    double                   Drip; /* Amount of drip from intercepted snow as a
                                      result of snowmelt (m) */
    double                   ExcessSnowMelt; /* Snowmelt in excess of the water holding
                                                capacity of the tree (m) */
    double                   InitialSnowInt; /* Initial intercepted snow (m) */
    double                   IntRainOrg;
    double                   MaxWaterInt; /* Water interception capacity (m) */
    double                   MaxSnowInt; /* Snow interception capacity (m) */
    double                   NetRadiation;
    double                   PotSnowMelt; /* Potential snow melt (m) */
    double                   RainThroughFall; /* Amount of rain reaching to the ground (m)
                                               */
    double                   RefreezeEnergy; /* Energy available for refreezing or melt */
    double                   ReleasedMass; /* Amount of mass release of intercepted snow
                                              (m) */
    /* double SensibleHeat; */           /* Sensible heat flux (W/m2) */
    double                   SnowThroughFall; /* Amount of snow reaching to the ground (m)
                                               */
    double                   Imax1; /* maxium water intecept regardless of temp */
    double                   IntRainFract; /* Fraction of intercpeted water which is
                                              liquid */
    double                   IntSnowFract; /* Fraction of intercepted water which is
                                              solid */
    double                   Overload; /* temp variable to calculated structural
                                          overloading */
    double                   Qnet; /* temporary storage of energy balance
                                      error */
    double                   Tupper;
    double                   Tlower;
    double                   Evap;
    double                   OldTfoliage;

    double                   AirDens;
    double                   EactAir;
    double                   Press; // atmospheric pressure
    double                   Vpd; // vapor pressure defficit
    double                   shortwave; //
    double                   Catm; //

    AirDens = force->density[hidx];
    EactAir = force->vp[hidx];
    Press = force->pressure[hidx];
    Vpd = force->vpd[hidx];
    shortwave = force->shortwave[hidx];
    if (options.CARBON) {
        Catm = force->Catm[hidx];
    }
    else {
        Catm = MISSING;
    }

    /* Initialize Tfoliage_fbflag */
    *Tfoliage_fbflag = 0;

    /* Convert Units from VIC (mm -> m) */
    *RainFall /= MM_PER_M;
    *SnowFall /= MM_PER_M;
    *IntRain /= MM_PER_M;
    MaxInt /= MM_PER_M;
    IntRainOrg = *IntRain;

    /* Initialize Drip, H2O balance, and mass release variables. */
    *IntSnow /= F;
    *IntRain /= F;

    InitialSnowInt = *IntSnow;

    Drip = 0.0;
    ReleasedMass = 0.0;

    OldTfoliage = *Tfoliage;

    /* Determine the maximum snow interception water equivalent.
       Kobayashi, D., 1986, Snow Accumulation on a Narrow Board,
       Cold Regions Science and Technology, (13), pp. 239-245.
       Figure 4. */

    Imax1 = 4.0 * param.VEG_LAI_SNOW_MULTIPLIER * LAI;

    if ((*Tfoliage) < -1.0 && (*Tfoliage) > -3.0) {
        MaxSnowInt = ((*Tfoliage) * 3.0 / 2.0) + (11.0 / 2.0);
    }
    else if ((*Tfoliage) > -1.0) {
        MaxSnowInt = 4.0;
    }
    else {
        MaxSnowInt = 1.0;
    }

    /* therefore LAI_ratio decreases as temp decreases */

    MaxSnowInt *= param.VEG_LAI_SNOW_MULTIPLIER * LAI;

    /* Calculate snow interception. */

    if (MaxSnowInt > 0) {
        DeltaSnowInt = (1 - *IntSnow / MaxSnowInt) * *SnowFall;
        if (DeltaSnowInt + *IntSnow > MaxSnowInt) {
            DeltaSnowInt = MaxSnowInt - *IntSnow;
        }
        if (DeltaSnowInt < 0.0) {
            DeltaSnowInt = 0.0;
        }
    }
    else {
        DeltaSnowInt = 0.0;
    }

    /* Reduce the amount of intercepted snow if windy and cold.
       Ringyo Shikenjo Tokyo, #54, 1952.
       Bulletin of the Govt. Forest Exp. Station,
       Govt. Forest Exp. Station, Meguro, Tokyo, Japan.
       FORSTX 634.9072 R475r #54.
       Page 146, Figure 10.

       Reduce the amount of intercepted snow if snowing, windy, and
       cold (< -3 to -5 C).
       Schmidt and Troendle 1992 western snow conference paper. */

    if ((*Tfoliage) < -3.0 && DeltaSnowInt > 0.0 && Wind[1] > 1.0) {
        BlownSnow = (0.2 * Wind[1] - 0.2) * DeltaSnowInt;
        if (BlownSnow >= DeltaSnowInt) {
            BlownSnow = DeltaSnowInt;
        }
        DeltaSnowInt -= BlownSnow;
    }

    /* now update snowfall and total accumulated intercepted snow amounts */

    if (*IntSnow + DeltaSnowInt > Imax1) {
        DeltaSnowInt = 0.0;
    }

    /* pixel depth    */
    SnowThroughFall = (*SnowFall - DeltaSnowInt) * F + (*SnowFall) * (1 - F);

    // Snow in canopy too thin for EB calculations; let it fall through
    if (*SnowFall == 0 && *IntSnow < param.SNOW_MIN_SWQ_EB_THRES) {
        SnowThroughFall += *IntSnow;
        DeltaSnowInt -= *IntSnow;
    }

    /* physical depth */
    *IntSnow += DeltaSnowInt;
    if (*IntSnow < DBL_EPSILON) {
        *IntSnow = 0.0;
    }

    /* Calculate amount of rain intercepted on branches and stored in
       intercepted snow. */

    /* physical depth */
    MaxWaterInt = param.SNOW_LIQUID_WATER_CAPACITY * (*IntSnow) + MaxInt;

    if ((*IntRain + *RainFall) <= MaxWaterInt) {
        /* physical depth */
        *IntRain += *RainFall;
        /* pixel depth */
        RainThroughFall = *RainFall * (1 - F);
    }
    else {
        /* pixel depth */
        RainThroughFall = (*IntRain + *RainFall - MaxWaterInt) * F +
                          (*RainFall * (1 - F));
        /* physical depth */
        *IntRain = MaxWaterInt;
    }

    // Liquid water in canopy too thin for EB calculations; let it fall through
    if (*RainFall == 0 && *IntRain < param.SNOW_MIN_SWQ_EB_THRES) {
        RainThroughFall += *IntRain;
        *IntRain = 0.0;
    }

    /* at this point we have calculated the amount of snowfall intercepted and
       the amount of rainfall intercepted.  These values have been
       appropriately subtracted from SnowFall and RainFall to determine
       SnowThroughfall and RainThroughfall.  However, we can end up with the
       condition that the total intercepted rain plus intercepted snow is
       greater than the maximum bearing capacity of the tree regardless of air
       temp (Imax1).  The following routine will adjust *IntRain and *IntSnow
       by triggering mass release due to overloading.  Of course since *IntRain
       and *IntSnow are mixed, we need to slough them of as fixed fractions  */

    if (*IntRain + *IntSnow > Imax1) { /*then trigger structural unloading*/
        Overload = (*IntSnow + *IntRain) - Imax1;
        IntRainFract = *IntRain / (*IntRain + *IntSnow);
        IntSnowFract = *IntSnow / (*IntRain + *IntSnow);
        *IntRain = *IntRain - Overload * IntRainFract;
        *IntSnow = *IntSnow - Overload * IntSnowFract;
        RainThroughFall = RainThroughFall + (Overload * IntRainFract) * F;
        SnowThroughFall = SnowThroughFall + (Overload * IntSnowFract) * F;
    }

    // If we've lost all intercepted moisture, we've essentially lost the thermal
    // mass of the canopy and Tfoliage should be equal to Tcanopy
    if (*IntRain + *IntSnow < DBL_EPSILON) {
        *Tfoliage = Tcanopy;
    }

    /* Calculate the net radiation at the canopy surface, using the canopy
       temperature.  The outgoing longwave is subtracted twice, because the
       canopy radiates in two directions */

    Tupper = Tlower = MISSING;

    if (*IntSnow > 0 || *SnowFall > 0) {
        /* Snow present or accumulating in the canopy */

        *AlbedoOver = param.SNOW_NEW_SNOW_ALB; // albedo of intercepted snow in canopy
        *NetShortOver = (1. - *AlbedoOver) * ShortOverIn; // net SW in canopy

        Qnet = solve_canopy_energy_bal(0., Dt, soil_con->elevation,
                                       soil_con->max_moist, soil_con->Wcr,
                                       soil_con->Wpwp, soil_con->frost_fract,
                                       AirDens, EactAir, Press, Le, Tcanopy,
                                       Vpd, shortwave, Catm, dryFrac, &Evap,
                                       Ra, Ra_used, *RainFall, Wind, veg_class,
                                       displacement, ref_height, roughness,
                                       root, CanopLayerBnd, IntRainOrg,
                                       *IntSnow, IntRain, layer, veg_var,
                                       LongOverIn, LongUnderOut, *NetShortOver,
                                       AdvectedEnergy, LatentHeat,
                                       LatentHeatSub, LongOverOut, NetLongOver,
                                       &NetRadiation, &RefreezeEnergy,
                                       SensibleHeat, VaporMassFlux);

        if (Qnet != 0) {
            /* Intercepted snow not melting - need to find temperature */
            Tupper = 0;
            if ((*Tfoliage) <= 0.) {
                Tlower = (*Tfoliage) - param.SNOW_DT;
            }
            else {
                Tlower = -param.SNOW_DT;
            }
        }
        else {
            *Tfoliage = 0.; // intercepted snow is melting
        }
    }
    else {
        /* No snow in canopy */
        *AlbedoOver = bare_albedo;
        *NetShortOver = (1. - *AlbedoOver) * ShortOverIn; // net SW in canopy
        Qnet = -9999;
        Tupper = (*Tfoliage) + param.SNOW_DT;
        Tlower = (*Tfoliage) - param.SNOW_DT;
    }

    if (Tupper != MISSING && Tlower != MISSING) {
        *Tfoliage = root_brent(Tlower, Tupper,
                               func_canopy_energy_bal, Dt,
                               soil_con->elevation, soil_con->max_moist,
                               soil_con->Wcr, soil_con->Wpwp,
                               soil_con->frost_fract,
                               AirDens, EactAir, Press, Le,
                               Tcanopy, Vpd, shortwave, Catm, dryFrac,
                               &Evap, Ra, Ra_used, *RainFall, Wind,
                               veg_class, displacement, ref_height,
                               roughness, root, CanopLayerBnd, IntRainOrg,
                               *IntSnow,
                               IntRain, layer, veg_var,
                               LongOverIn, LongUnderOut,
                               *NetShortOver,
                               AdvectedEnergy,
                               LatentHeat, LatentHeatSub,
                               LongOverOut, NetLongOver, &NetRadiation,
                               &RefreezeEnergy, SensibleHeat,
                               VaporMassFlux);

        if (*Tfoliage <= -998) {
            if (options.TFALLBACK) {
                *Tfoliage = OldTfoliage;
                *Tfoliage_fbflag = 1;
                (*Tfoliage_fbcount)++;
            }
            else {
                Qnet = error_calc_canopy_energy_bal(*Tfoliage, band, month, Dt,
                                                    soil_con->elevation,
                                                    soil_con->max_moist,
                                                    soil_con->Wcr,
                                                    soil_con->Wpwp,
                                                    soil_con->depth,
                                                    soil_con->frost_fract,
                                                    AirDens, EactAir, Press, Le,
                                                    Tcanopy, Vpd, shortwave,
                                                    Catm, dryFrac,
                                                    &Evap, Ra, Ra_used,
                                                    *RainFall, Wind, UnderStory,
                                                    iveg,
                                                    veg_class, displacement,
                                                    ref_height,
                                                    roughness, root,
                                                    CanopLayerBnd, IntRainOrg,
                                                    *IntSnow,
                                                    IntRain, layer, veg_var,
                                                    LongOverIn, LongUnderOut,
                                                    *NetShortOver,
                                                    AdvectedEnergy,
                                                    LatentHeat, LatentHeatSub,
                                                    LongOverOut, NetLongOver,
                                                    &NetRadiation,
                                                    &RefreezeEnergy,
                                                    SensibleHeat,
                                                    VaporMassFlux);
                return(VIC_ERROR);
            }
        }

        Qnet = solve_canopy_energy_bal(*Tfoliage, Dt, soil_con->elevation,
                                       soil_con->max_moist, soil_con->Wcr,
                                       soil_con->Wpwp, soil_con->frost_fract,
                                       AirDens, EactAir, Press, Le, Tcanopy,
                                       Vpd, shortwave, Catm, dryFrac, &Evap,
                                       Ra, Ra_used, *RainFall, Wind, veg_class,
                                       displacement, ref_height, roughness,
                                       root, CanopLayerBnd, IntRainOrg,
                                       *IntSnow, IntRain, layer, veg_var,
                                       LongOverIn, LongUnderOut, *NetShortOver,
                                       AdvectedEnergy, LatentHeat,
                                       LatentHeatSub, LongOverOut, NetLongOver,
                                       &NetRadiation, &RefreezeEnergy,
                                       SensibleHeat, VaporMassFlux);
    }

    if (*IntSnow <= 0) {
        RainThroughFall = veg_var->throughfall / MM_PER_M;
    }

    RefreezeEnergy *= Dt;

    /* if RefreezeEnergy is positive it means energy is available to melt the
       intercepted snow in the canopy.  If it is negative, it means that
       intercepted water will be refrozen */

    /* Update maximum water interception storage */

    MaxWaterInt = param.SNOW_LIQUID_WATER_CAPACITY * (*IntSnow) + MaxInt;

    /* Convert the vapor mass flux from a flux to a depth per interval */
    *VaporMassFlux *= Dt;

    if (*Tfoliage == 0) {
        if (-(*VaporMassFlux) > *IntRain) {
            *VaporMassFlux = -(*IntRain);
            *IntRain = 0.;
        }
        else {
            *IntRain += *VaporMassFlux;
        }

        if (RefreezeEnergy < 0) {
            /* intercepted snow is ripe, melt can occur */
            PotSnowMelt = min((-RefreezeEnergy / CONST_LATICE / CONST_RHOFW),
                              *IntSnow);
            *MeltEnergy -= (CONST_LATICE * PotSnowMelt * CONST_RHOFW) / (Dt);
        }
        else {
            /* snow temperature is below freezing, no melt occurs */
            PotSnowMelt = 0;
            *MeltEnergy -= (CONST_LATICE * PotSnowMelt * CONST_RHOFW) / (Dt);
        }

        if ((*IntRain + PotSnowMelt) <= MaxWaterInt) {
            *IntSnow -= PotSnowMelt;
            *IntRain += PotSnowMelt;
            PotSnowMelt = 0.0;
        }
        else {
            ExcessSnowMelt = PotSnowMelt + *IntRain - MaxWaterInt;

            *IntSnow -= MaxWaterInt - (*IntRain);
            *IntRain = MaxWaterInt;
            if (*IntSnow < 0.0) {
                *IntSnow = 0.0;
            }

            if (SnowThroughFall > 0.0 &&
                InitialSnowInt <= param.VEG_MIN_INTERCEPTION_STORAGE) {
                /* Water in excess of MaxWaterInt has been generated.  If it is
                   snowing and there was little intercepted snow at the beginning
                   of the time step ( <= MIN_INTERCEPTION_STORAGE), then allow the
                   snow to melt as it is intercepted */
                Drip += ExcessSnowMelt;
                *IntSnow -= ExcessSnowMelt;
                if (*IntSnow < 0.0) {
                    *IntSnow = 0.0;
                }
            }
            else {
                /* Else, SnowThroughFall = 0.0 or SnowThroughFall > 0.0 and there is a
                   substantial amount of intercepted snow at the beginning of the time
                   step ( > MIN_INTERCEPTION_STORAGE).  Snow melt may generate mass
                   release. */
                *TempIntStorage += ExcessSnowMelt;
            }

            MassRelease(IntSnow, TempIntStorage, &ReleasedMass, &Drip);
        }

        /* If intercepted snow has melted, add the water it held to drip */

        MaxWaterInt = param.SNOW_LIQUID_WATER_CAPACITY * (*IntSnow) + MaxInt;
        if (*IntRain > MaxWaterInt) {
            Drip += *IntRain - MaxWaterInt;
            *IntRain = MaxWaterInt;
        }
    }
    else {
        /* Reset *TempIntStorage to 0.0 when energy balance is negative */

        *TempIntStorage = 0.0;

        /* Refreeze as much surface water as you can */

        if (-RefreezeEnergy > -(*IntRain) * CONST_LATICE) {
            *IntSnow += fabs(RefreezeEnergy) / CONST_LATICE;
            *IntRain -= fabs(RefreezeEnergy) / CONST_LATICE;

            *MeltEnergy += (fabs(RefreezeEnergy) * CONST_RHOFW) / (Dt);

            RefreezeEnergy = 0.0;
        }
        else {
            /* All of the water in the surface layer has been frozen. */

            *IntSnow += *IntRain;

            /* Energy released by freezing of intercepted water is added to the
               MeltEnergy */

            *MeltEnergy += (CONST_LATICE * (*IntRain) * CONST_RHOFW) / (Dt);
            *IntRain = 0.0;
        }

        if (-(*VaporMassFlux) > *IntSnow) {
            *VaporMassFlux = -(*IntSnow);
            *IntSnow = 0.0;
        }
        else {
            *IntSnow += *VaporMassFlux;
        }
    }

    *IntSnow *= F;
    *IntRain *= F;
    *MeltEnergy *= F;
    *VaporMassFlux *= F;
    Drip *= F;
    ReleasedMass *= F;

    if (*IntSnow == 0 && *IntRain > MaxInt) {
        // if snow has melted, make sure canopy is not retaining extra water
        RainThroughFall += *IntRain - MaxInt;
        *IntRain = MaxInt;
    }

    /* Calculate intercepted H2O balance. */
    *RainFall = RainThroughFall + Drip;
    *SnowFall = SnowThroughFall + ReleasedMass;

    /* Convert Units to VIC (m -> mm) */
    *VaporMassFlux *= -1.;
    *RainFall *= MM_PER_M;
    *SnowFall *= MM_PER_M;
    *IntRain *= MM_PER_M;

    /*** FIX THIS ***/
    *MeltEnergy = RefreezeEnergy / Dt;

    return(0);
}

/******************************************************************************
* @brief    Dummy function to make a direct call to func_canopy_energy_bal()
*           possible.
******************************************************************************/
double
solve_canopy_energy_bal(double Tfoliage,
                        ...)
{
    va_list ap;
    double  Qnet;

    va_start(ap, Tfoliage);
    Qnet = func_canopy_energy_bal(Tfoliage, ap);
    va_end(ap);

    return Qnet;
}

/******************************************************************************
* @brief    Dummy function to make a direct call to
*           error_print_canopy_energy_bal() possible.
******************************************************************************/
double
error_calc_canopy_energy_bal(double Tfoliage,
                             ...)
{
    va_list ap;
    double  Qnet;

    va_start(ap, Tfoliage);
    Qnet = error_print_canopy_energy_bal(Tfoliage, ap);
    va_end(ap);

    return Qnet;
}

/******************************************************************************
* @brief    Print snow pack energy balance terms
******************************************************************************/
double
error_print_canopy_energy_bal(double  Tfoliage,
                              va_list ap)
{
    extern option_struct options;

    /* General Model Parameters */
    int                  band;
    int                  month;

    double               delta_t;
    double               elevation;

    double              *Wmax;
    double              *Wcr;
    double              *Wpwp;
    double              *depth;
    double              *frost_fract;

    /* Atmopheric Condition and Forcings */
    double               AirDens;
    double               EactAir;
    double               Press;
    double               Le;
    double               Tcanopy;
    double               Vpd;
    double               shortwave;
    double               Catm;
    double              *dryFrac;

    double              *Evap;
    double              *Ra;
    double              *Ra_used;
    double               Rainfall;
    double              *Wind;

    /* Vegetation Terms */
    int                  UnderStory;
    int                  iveg;
    unsigned int         veg_class;

    double              *displacement;
    double              *ref_height;
    double              *roughness;

    double              *root;
    double              *CanopLayerBnd;

    /* Water Flux Terms */
    double               IntRain;
    double               IntSnow;

    double              *Wdew;

    layer_data_struct   *layer;
    veg_var_struct      *veg_var;

    /* Energy Flux Terms */
    double               LongOverIn;
    double               LongUnderOut;
    double               NetShortOver;

    double              *AdvectedEnergy;
    double              *LatentHeat;
    double              *LatentHeatSub;
    double              *LongOverOut;
    double              *NetLongOver;
    double              *NetRadiation;
    double              *RefreezeEnergy;
    double              *SensibleHeat;
    double              *VaporMassFlux;

    size_t               cidx;

    /** Read variables from variable length argument list **/

    /* General Model Parameters */
    band = (int) va_arg(ap, int);
    month = (int) va_arg(ap, int);

    delta_t = (double) va_arg(ap, double);
    elevation = (double) va_arg(ap, double);

    Wmax = (double *) va_arg(ap, double *);
    Wcr = (double *) va_arg(ap, double *);
    Wpwp = (double *) va_arg(ap, double *);
    depth = (double *) va_arg(ap, double *);
    frost_fract = (double *) va_arg(ap, double *);

    /* Atmopheric Condition and Forcings */
    AirDens = (double) va_arg(ap, double);
    EactAir = (double) va_arg(ap, double);
    Press = (double) va_arg(ap, double);
    Le = (double) va_arg(ap, double);
    Tcanopy = (double) va_arg(ap, double);
    Vpd = (double) va_arg(ap, double);
    shortwave = (double) va_arg(ap, double);
    Catm = (double) va_arg(ap, double);
    dryFrac = (double *) va_arg(ap, double *);

    Evap = (double *) va_arg(ap, double *);
    Ra = (double *) va_arg(ap, double *);
    Ra_used = (double *) va_arg(ap, double *);
    Rainfall = (double) va_arg(ap, double);
    Wind = (double *) va_arg(ap, double *);

    /* Vegetation Terms */
    UnderStory = (int) va_arg(ap, int);
    iveg = (int) va_arg(ap, int);
    veg_class = (unsigned int) va_arg(ap, unsigned int);

    displacement = (double *) va_arg(ap, double *);
    ref_height = (double *) va_arg(ap, double *);
    roughness = (double *) va_arg(ap, double *);

    root = (double *) va_arg(ap, double *);
    CanopLayerBnd = (double *) va_arg(ap, double *);

    /* Water Flux Terms */
    IntRain = (double) va_arg(ap, double);
    IntSnow = (double) va_arg(ap, double);

    Wdew = (double *) va_arg(ap, double *);

    layer = (layer_data_struct *) va_arg(ap, layer_data_struct *);
    veg_var = (veg_var_struct *) va_arg(ap, veg_var_struct *);

    /* Energy Flux Terms */
    LongOverIn = (double) va_arg(ap, double);
    LongUnderOut = (double) va_arg(ap, double);
    NetShortOver = (double) va_arg(ap, double);

    AdvectedEnergy = (double *) va_arg(ap, double *);
    LatentHeat = (double *) va_arg(ap, double *);
    LatentHeatSub = (double *) va_arg(ap, double *);
    LongOverOut = (double *) va_arg(ap, double *);
    NetLongOver = (double *) va_arg(ap, double *);
    NetRadiation = (double *) va_arg(ap, double *);
    RefreezeEnergy = (double *) va_arg(ap, double *);
    SensibleHeat = (double *) va_arg(ap, double *);
    VaporMassFlux = (double *) va_arg(ap, double *);

    /** Print variable info */
    log_warn("snow_intercept failed to converge to a solution "
             "in root_brent.  Variable values will be dumped to the "
             "screen, check for invalid values.");

    /* General Model Parameters */
    fprintf(LOG_DEST, "band = %i\n", band);
    fprintf(LOG_DEST, "month = %i\n", month);

    fprintf(LOG_DEST, "delta_t = %f\n", delta_t);
    fprintf(LOG_DEST, "elevation = %f\n", elevation);

    fprintf(LOG_DEST, "*Wmax = %f\n", *Wmax);
    fprintf(LOG_DEST, "*Wcr = %f\n", *Wcr);
    fprintf(LOG_DEST, "*Wpwp = %f\n", *Wpwp);
    fprintf(LOG_DEST, "*depth = %f\n", *depth);
    fprintf(LOG_DEST, "frost_fract = %f\n", *frost_fract);

    /* Atmopheric Condition and Forcings */
    fprintf(LOG_DEST, "AirDens = %f\n", AirDens);
    fprintf(LOG_DEST, "EactAir = %f\n", EactAir);
    fprintf(LOG_DEST, "Press = %f\n", Press);
    fprintf(LOG_DEST, "Le = %f\n", Le);
    fprintf(LOG_DEST, "Ra = [%f, %f]\n", Ra[1], Ra[UnderStory]);
    fprintf(LOG_DEST, "Ra_used = %f\n", *Ra_used);
    fprintf(LOG_DEST, "Tcanopy = %f\n", Tcanopy);
    fprintf(LOG_DEST, "Vpd = %f\n", Vpd);
    fprintf(LOG_DEST, "shortwave = %f\n", shortwave);
    fprintf(LOG_DEST, "Catm = %f\n", Catm);
    fprintf(LOG_DEST, "dryFrac = %f\n", *dryFrac);

    fprintf(LOG_DEST, "Evap = %f\n", *Evap);
    fprintf(LOG_DEST, "Rainfall = %f\n", Rainfall);
    fprintf(LOG_DEST, "Wind = [%f, %f]\n", Wind[1], Wind[UnderStory]);

    /* Vegetation Terms */
    fprintf(LOG_DEST, "UnderStory = %i\n", UnderStory);
    fprintf(LOG_DEST, "iveg = %i\n", iveg);
    fprintf(LOG_DEST, "veg_class = %i\n", veg_class);

    fprintf(LOG_DEST, "displacement = [%f, %f]\n", displacement[1],
            displacement[UnderStory]);
    fprintf(LOG_DEST, "ref_height = [%f, %f]\n",
            ref_height[1], ref_height[UnderStory]);
    fprintf(LOG_DEST, "roughness = [%f, %f]\n",
            roughness[1], roughness[UnderStory]);

    fprintf(LOG_DEST, "root = %f\n", *root);

    if (options.CARBON) {
        fprintf(LOG_DEST, "CanopLayerBnd =");
        for (cidx = 0; cidx < options.Ncanopy; cidx++) {
            fprintf(LOG_DEST, " %f", CanopLayerBnd[cidx]);
        }
        fprintf(LOG_DEST, "\n");
    }

    /* Water Flux Terms */
    fprintf(LOG_DEST, "IntRain = %f\n", IntRain);
    fprintf(LOG_DEST, "IntSnow = %f\n", IntSnow);

    fprintf(LOG_DEST, "Wdew = %f\n", *Wdew);

    write_layer(layer, iveg, frost_fract);
    write_vegvar(&(veg_var[0]), iveg);

    fprintf(LOG_DEST, "Tfoliage = %f\n", Tfoliage);

    /* Energy Flux Terms */
    fprintf(LOG_DEST, "LongOverIn = %f\n", LongOverIn);
    fprintf(LOG_DEST, "LongUnderOut = %f\n", LongUnderOut);
    fprintf(LOG_DEST, "NetShortOver = %f\n", NetShortOver);

    fprintf(LOG_DEST, "*AdvectedEnergy = %f\n", *AdvectedEnergy);
    fprintf(LOG_DEST, "*LatentHeat = %f\n", *LatentHeat);
    fprintf(LOG_DEST, "*LatentHeatSub = %f\n", *LatentHeatSub);
    fprintf(LOG_DEST, "*LongOverOut = %f\n", *LongOverOut);
    fprintf(LOG_DEST, "*NetLongOver = %f\n", *NetLongOver);
    fprintf(LOG_DEST, "*NetRadiation = %f\n", *NetRadiation);
    fprintf(LOG_DEST, "*RefreezeEnergy = %f\n", *RefreezeEnergy);
    fprintf(LOG_DEST, "*SensibleHeat = %f\n", *SensibleHeat);
    fprintf(LOG_DEST, "*VaporMassFlux = %f\n", *VaporMassFlux);

    log_warn("Finished dumping snow_intercept variables.\n"
             "Try increasing SNOW_DT to get model to complete cell.\n"
             "Then check output for instabilities.");

    return(VIC_ERROR);
}
