library(testthat)
# aqua_data
aqua_data_single_peptide<-data.table("run_id"=c(1,1,1,1,2,2,2,2),"protein_id"=c("PROTEINA","PROTEINA","PROTEINB","PROTEINB","PROTEINA","PROTEINA","PROTEINB","PROTEINB"),"peptide_id"=c("PEPTIDEK","PEPTIDEK(UniMod:259)","PEPTIDEC","PEPTIDEC","PEPTIDEK","PEPTIDEK(UniMod:259)","PEPTIDEC","PEPTIDEC"),"peptide_sequence"=c("PEPTIDEK","PEPTIDEK","PEPTIDEC","PEPTIDEC","PEPTIDEK","PEPTIDEK","PEPTIDEC","PEPTIDEC"),"precursor_charge"=c(2,2,2,3,2,3,2,3),"peptide_intensity"=c(100,80,60,40,95,75,55,35),concentration=c("?",10,"?","?","?",20,"?","?"),stringsAsFactors=FALSE)

aqua_data_multi_peptide<-data.table("run_id"=c(1,1,1,1,2,2,2,2),"protein_id"=c("PROTEINA","PROTEINA","PROTEINA","PROTEINA","PROTEINA","PROTEINA","PROTEINA","PROTEINA"),"peptide_id"=c("PEPTIDEK","PEPTIDEK(UniMod:259)","PEPTIDEC","PEPTIDEC(UniMod:259)","PEPTIDEK","PEPTIDEK(UniMod:259)","PEPTIDEC","PEPTIDEC(UniMod:259)"),"peptide_sequence"=c("PEPTIDEK","PEPTIDEK","PEPTIDEC","PEPTIDEC","PEPTIDEK","PEPTIDEK","PEPTIDEC","PEPTIDEC"),"precursor_charge"=c(2,2,2,3,2,3,2,3),"peptide_intensity"=c(100,80,60,40,95,75,55,35),concentration=c("?",10,"?",10,"?",20,"?",20),stringsAsFactors=FALSE)

# ProteinInference.default
test_that("ProteinInference.default: Absolute abundance estimation of endogenous proteins with spiked-in SIS peptides: Single peptides / Concentration", {
	expect_that(ProteinInference.default(aqua_data_single_peptide, peptide_method = "top", peptide_topx = 1, peptide_strictness = "loose", peptide_summary = "mean", transition_topx = 3, transition_strictness = "loose", transition_summary = "sum", fasta = NA, model = NA, combine_precursors = FALSE, consensus_peptides = TRUE, consensus_transitions = TRUE)$concentration,equals(c("12.5","?","25.3333333333333","?")))
	expect_that(ProteinInference.default(aqua_data_single_peptide, peptide_method = "top", peptide_topx = 1, peptide_strictness = "loose", peptide_summary = "mean", transition_topx = 3, transition_strictness = "loose", transition_summary = "sum", fasta = NA, model = NA, combine_precursors = FALSE, consensus_peptides = TRUE, consensus_transitions = TRUE)$response,equals(c(100,60,95,55)))
	expect_that(ProteinInference.default(aqua_data_multi_peptide, peptide_method = "top", peptide_topx = 2, peptide_strictness = "loose", peptide_summary = "mean", transition_topx = 3, transition_strictness = "loose", transition_summary = "sum", fasta = NA, model = NA, combine_precursors = FALSE, consensus_peptides = TRUE, consensus_transitions = TRUE)$concentration,equals(c("13.75","28.3809523809524")))
	expect_that(ProteinInference.default(aqua_data_multi_peptide, peptide_method = "top", peptide_topx = 2, peptide_strictness = "loose", peptide_summary = "mean", transition_topx = 3, transition_strictness = "loose", transition_summary = "sum", fasta = NA, model = NA, combine_precursors = FALSE, consensus_peptides = TRUE, consensus_transitions = TRUE)$response,equals(c(80,75)))
})

# ProteinInference.default
test_that("ProteinInference.default: More complex data", {
	data(UPS2MS)
	set.seed(131)
	REP1<-UPS2_SRM[sample(854,800),]
	REP1[which(REP1$protein_id %in% unique(REP1$protein_id)[1:10]),]$concentration<-"?"
	REP1$run_id<-"UPS2_SRM_REP1"

	REP2<-UPS2_SRM[sample(854,800),]
	REP2[which(REP2$protein_id %in% unique(REP2$protein_id)[1:10]),]$concentration<-"?"
	REP2$run_id<-"UPS2_SRM_REP2"

	REP3<-UPS2_SRM[sample(854,800),]
	REP3[which(REP3$protein_id %in% unique(REP3$protein_id)[1:10]),]$concentration<-"?"
	REP3$run_id<-"UPS2_SRM_REP3"

	REP3<-REP3[-which(REP3$protein_id %in% c("P01031ups|CO5_HUMAN_UPS","P55957ups|BID_HUMAN_UPS")),]

	UPS2_SRM_REP<-rbind(REP1,REP2,REP3)

	expect_that(ProteinInference.default(UPS2_SRM_REP,consensus_proteins=TRUE,consensus_peptide=TRUE,consensus_transitions=TRUE),equals(structure(list(run_id = c("UPS2_SRM_REP1", "UPS2_SRM_REP1", "UPS2_SRM_REP1", "UPS2_SRM_REP1", "UPS2_SRM_REP1", "UPS2_SRM_REP1", "UPS2_SRM_REP1", "UPS2_SRM_REP1", "UPS2_SRM_REP1", "UPS2_SRM_REP1", "UPS2_SRM_REP1", "UPS2_SRM_REP1", "UPS2_SRM_REP1", "UPS2_SRM_REP1", "UPS2_SRM_REP1", "UPS2_SRM_REP1", "UPS2_SRM_REP1", "UPS2_SRM_REP1", "UPS2_SRM_REP1", "UPS2_SRM_REP1", "UPS2_SRM_REP1", "UPS2_SRM_REP1", "UPS2_SRM_REP1", "UPS2_SRM_REP1", "UPS2_SRM_REP1", "UPS2_SRM_REP1", "UPS2_SRM_REP2", "UPS2_SRM_REP2", "UPS2_SRM_REP2", "UPS2_SRM_REP2", "UPS2_SRM_REP2", "UPS2_SRM_REP2", "UPS2_SRM_REP2", "UPS2_SRM_REP2", "UPS2_SRM_REP2", "UPS2_SRM_REP2", "UPS2_SRM_REP2", "UPS2_SRM_REP2", "UPS2_SRM_REP2", "UPS2_SRM_REP2", "UPS2_SRM_REP2", "UPS2_SRM_REP2", "UPS2_SRM_REP2", "UPS2_SRM_REP2", "UPS2_SRM_REP2", "UPS2_SRM_REP2", "UPS2_SRM_REP2", "UPS2_SRM_REP2", "UPS2_SRM_REP2", "UPS2_SRM_REP2", "UPS2_SRM_REP2", "UPS2_SRM_REP2", "UPS2_SRM_REP3", "UPS2_SRM_REP3", "UPS2_SRM_REP3", "UPS2_SRM_REP3", "UPS2_SRM_REP3", "UPS2_SRM_REP3", "UPS2_SRM_REP3", "UPS2_SRM_REP3", "UPS2_SRM_REP3", "UPS2_SRM_REP3", "UPS2_SRM_REP3", "UPS2_SRM_REP3", "UPS2_SRM_REP3", "UPS2_SRM_REP3", "UPS2_SRM_REP3", "UPS2_SRM_REP3", "UPS2_SRM_REP3", "UPS2_SRM_REP3", "UPS2_SRM_REP3", "UPS2_SRM_REP3", "UPS2_SRM_REP3", "UPS2_SRM_REP3", "UPS2_SRM_REP3", "UPS2_SRM_REP3", "UPS2_SRM_REP3", "UPS2_SRM_REP3"), protein_id = c("O76070ups|SYUG_HUMAN_UPS", "P00167ups|CYB5_HUMAN_UPS", "P00709ups|LALBA_HUMAN_UPS", "P00915ups|CAH1_HUMAN_UPS", "P00918ups|CAH2_HUMAN_UPS", "P01008ups|ANT3_HUMAN_UPS", "P01127ups|PDGFB_HUMAN_UPS", "P01133ups|EGF_HUMAN_UPS", "P01344ups|IGF2_HUMAN_UPS", "P02144ups|MYG_HUMAN_UPS", "P02753ups|RETBP_HUMAN_UPS", "P02768ups|ALBU_HUMAN_UPS", "P04040ups|CATA_HUMAN_UPS", "P06732ups|KCRM_HUMAN_UPS", "P12081ups|SYHC_HUMAN_UPS", "P15559ups|NQO1_HUMAN_UPS", "P16083ups|NQO2_HUMAN_UPS", "P41159ups|LEP_HUMAN_UPS", "P61626ups|LYSC_HUMAN_UPS", "P61769ups|B2MG_HUMAN_UPS", "P62937ups|PPIA_HUMAN_UPS", "P62988ups|UBIQ_HUMAN_UPS", "P63165ups|SUMO1_HUMAN_UPS", "P68871ups|HBB_HUMAN_UPS", "P69905ups|HBA_HUMAN_UPS", "Q06830ups|PRDX1_HUMAN_UPS", "O76070ups|SYUG_HUMAN_UPS", "P00167ups|CYB5_HUMAN_UPS", "P00709ups|LALBA_HUMAN_UPS", "P00915ups|CAH1_HUMAN_UPS", "P00918ups|CAH2_HUMAN_UPS", "P01008ups|ANT3_HUMAN_UPS", "P01127ups|PDGFB_HUMAN_UPS", "P01133ups|EGF_HUMAN_UPS", "P01344ups|IGF2_HUMAN_UPS", "P02144ups|MYG_HUMAN_UPS", "P02753ups|RETBP_HUMAN_UPS", "P02768ups|ALBU_HUMAN_UPS", "P04040ups|CATA_HUMAN_UPS", "P06732ups|KCRM_HUMAN_UPS", "P12081ups|SYHC_HUMAN_UPS", "P15559ups|NQO1_HUMAN_UPS", "P16083ups|NQO2_HUMAN_UPS", "P41159ups|LEP_HUMAN_UPS", "P61626ups|LYSC_HUMAN_UPS", "P61769ups|B2MG_HUMAN_UPS", "P62937ups|PPIA_HUMAN_UPS", "P62988ups|UBIQ_HUMAN_UPS", "P63165ups|SUMO1_HUMAN_UPS", "P68871ups|HBB_HUMAN_UPS", "P69905ups|HBA_HUMAN_UPS", "Q06830ups|PRDX1_HUMAN_UPS", "O76070ups|SYUG_HUMAN_UPS", "P00167ups|CYB5_HUMAN_UPS", "P00709ups|LALBA_HUMAN_UPS", "P00915ups|CAH1_HUMAN_UPS", "P00918ups|CAH2_HUMAN_UPS", "P01008ups|ANT3_HUMAN_UPS", "P01127ups|PDGFB_HUMAN_UPS", "P01133ups|EGF_HUMAN_UPS", "P01344ups|IGF2_HUMAN_UPS", "P02144ups|MYG_HUMAN_UPS", "P02753ups|RETBP_HUMAN_UPS", "P02768ups|ALBU_HUMAN_UPS", "P04040ups|CATA_HUMAN_UPS", "P06732ups|KCRM_HUMAN_UPS", "P12081ups|SYHC_HUMAN_UPS", "P15559ups|NQO1_HUMAN_UPS", "P16083ups|NQO2_HUMAN_UPS", "P41159ups|LEP_HUMAN_UPS", "P61626ups|LYSC_HUMAN_UPS", "P61769ups|B2MG_HUMAN_UPS", "P62937ups|PPIA_HUMAN_UPS", "P62988ups|UBIQ_HUMAN_UPS", "P63165ups|SUMO1_HUMAN_UPS", "P68871ups|HBB_HUMAN_UPS", "P69905ups|HBA_HUMAN_UPS", "Q06830ups|PRDX1_HUMAN_UPS"), concentration = c("0.449", "44.9", "4.49", "?", "?", "?", "0.449", "44.9", "0.449", "?", "?", "?", "44.9", "?", "4.49", "44.9", "4.49", "449", "4.49", "0.449", "44.9", "449", "?", "?", "449", "?", "0.449", "44.9", "4.49", "?", "449", "0.449", "?", "44.9", "0.449", "?", "4.49", "449", "?", "4.49", "?", "?", "?", "449", "4.49", "0.449", "?", "449", "?", "449", "?", "44.9", "0.449", "?", "?", "449", "449", "0.449", "?", "44.9", "0.449", "44.9", "?", "?", "44.9", "4.49", "?", "?", "4.49", "?", "?", "0.449", "44.9", "449", "44.9", "449", "?", "44.9"), response = c(161978.666666667, 13113548.6666667, 568833.666666667, 37676253.3333333, 25882206.3333333, 74258.3333333333, 46324, 3684505, 89774, 9086594.33333333, 645838.666666667, 73089941.6666667, 3486194.5, 549538, 507207, 2309451.5, 589519.666666667, 41890138.6666667, 256876, 28007, 11569710.6666667, 26079521.3333333, 4224907.33333333, 24090440.6666667, 17080367.3333333, 3852184.66666667, 161978.666666667, 13113548.6666667, 568833.666666667, 37676253.3333333, 25882206.3333333, 74258.3333333333, 46324, 3684505, 89774, 9086594.33333333, 645838.666666667, 73089941.6666667, 3486194.5, 549538, 507207, 2309451.5, 589519.666666667, 41890138.6666667, 256876, 28007, 11569710.6666667, 26079521.3333333, 4224907.33333333, 24090440.6666667, 17080367.3333333, 3852184.66666667, 161978.666666667, 13113548.6666667, 568833.666666667, 37676253.3333333, 25882206.3333333, 74258.3333333333, 46324, 3684505, 89774, 9086594.33333333, 645838.666666667, 73089941.6666667, 3486194.5, 549538, 507207, 2309451.5, 589519.666666667, 41890138.6666667, 256876, 28007, 11569710.6666667, 26079521.3333333, 4224907.33333333, 24090440.6666667, 17080367.3333333, 3852184.66666667)), .Names = c("run_id", "protein_id", "concentration", "response"), row.names = c(NA, -78L), class = "data.frame")))

	expect_that(ProteinInference.default(UPS2_SRM_REP,consensus_proteins=FALSE,consensus_peptide=TRUE,consensus_transitions=TRUE),equals(structure(list(run_id = c("UPS2_SRM_REP1", "UPS2_SRM_REP1", "UPS2_SRM_REP1", "UPS2_SRM_REP1", "UPS2_SRM_REP1", "UPS2_SRM_REP1", "UPS2_SRM_REP1", "UPS2_SRM_REP1", "UPS2_SRM_REP1", "UPS2_SRM_REP1", "UPS2_SRM_REP1", "UPS2_SRM_REP1", "UPS2_SRM_REP1", "UPS2_SRM_REP1", "UPS2_SRM_REP1", "UPS2_SRM_REP1", "UPS2_SRM_REP1", "UPS2_SRM_REP1", "UPS2_SRM_REP1", "UPS2_SRM_REP1", "UPS2_SRM_REP1", "UPS2_SRM_REP1", "UPS2_SRM_REP1", "UPS2_SRM_REP1", "UPS2_SRM_REP1", "UPS2_SRM_REP1", "UPS2_SRM_REP1", "UPS2_SRM_REP1", "UPS2_SRM_REP2", "UPS2_SRM_REP2", "UPS2_SRM_REP2", "UPS2_SRM_REP2", "UPS2_SRM_REP2", "UPS2_SRM_REP2", "UPS2_SRM_REP2", "UPS2_SRM_REP2", "UPS2_SRM_REP2", "UPS2_SRM_REP2", "UPS2_SRM_REP2", "UPS2_SRM_REP2", "UPS2_SRM_REP2", "UPS2_SRM_REP2", "UPS2_SRM_REP2", "UPS2_SRM_REP2", "UPS2_SRM_REP2", "UPS2_SRM_REP2", "UPS2_SRM_REP2", "UPS2_SRM_REP2", "UPS2_SRM_REP2", "UPS2_SRM_REP2", "UPS2_SRM_REP2", "UPS2_SRM_REP2", "UPS2_SRM_REP2", "UPS2_SRM_REP2", "UPS2_SRM_REP2", "UPS2_SRM_REP2", "UPS2_SRM_REP3", "UPS2_SRM_REP3", "UPS2_SRM_REP3", "UPS2_SRM_REP3", "UPS2_SRM_REP3", "UPS2_SRM_REP3", "UPS2_SRM_REP3", "UPS2_SRM_REP3", "UPS2_SRM_REP3", "UPS2_SRM_REP3", "UPS2_SRM_REP3", "UPS2_SRM_REP3", "UPS2_SRM_REP3", "UPS2_SRM_REP3", "UPS2_SRM_REP3", "UPS2_SRM_REP3", "UPS2_SRM_REP3", "UPS2_SRM_REP3", "UPS2_SRM_REP3", "UPS2_SRM_REP3", "UPS2_SRM_REP3", "UPS2_SRM_REP3", "UPS2_SRM_REP3", "UPS2_SRM_REP3", "UPS2_SRM_REP3", "UPS2_SRM_REP3"), protein_id = c("O76070ups|SYUG_HUMAN_UPS", "P00167ups|CYB5_HUMAN_UPS", "P00709ups|LALBA_HUMAN_UPS", "P00915ups|CAH1_HUMAN_UPS", "P00918ups|CAH2_HUMAN_UPS", "P01008ups|ANT3_HUMAN_UPS", "P01031ups|CO5_HUMAN_UPS", "P01127ups|PDGFB_HUMAN_UPS", "P01133ups|EGF_HUMAN_UPS", "P01344ups|IGF2_HUMAN_UPS", "P02144ups|MYG_HUMAN_UPS", "P02753ups|RETBP_HUMAN_UPS", "P02768ups|ALBU_HUMAN_UPS", "P04040ups|CATA_HUMAN_UPS", "P06732ups|KCRM_HUMAN_UPS", "P12081ups|SYHC_HUMAN_UPS", "P15559ups|NQO1_HUMAN_UPS", "P16083ups|NQO2_HUMAN_UPS", "P41159ups|LEP_HUMAN_UPS", "P55957ups|BID_HUMAN_UPS", "P61626ups|LYSC_HUMAN_UPS", "P61769ups|B2MG_HUMAN_UPS", "P62937ups|PPIA_HUMAN_UPS", "P62988ups|UBIQ_HUMAN_UPS", "P63165ups|SUMO1_HUMAN_UPS", "P68871ups|HBB_HUMAN_UPS", "P69905ups|HBA_HUMAN_UPS", "Q06830ups|PRDX1_HUMAN_UPS", "O76070ups|SYUG_HUMAN_UPS", "P00167ups|CYB5_HUMAN_UPS", "P00709ups|LALBA_HUMAN_UPS", "P00915ups|CAH1_HUMAN_UPS", "P00918ups|CAH2_HUMAN_UPS", "P01008ups|ANT3_HUMAN_UPS", "P01031ups|CO5_HUMAN_UPS", "P01127ups|PDGFB_HUMAN_UPS", "P01133ups|EGF_HUMAN_UPS", "P01344ups|IGF2_HUMAN_UPS", "P02144ups|MYG_HUMAN_UPS", "P02753ups|RETBP_HUMAN_UPS", "P02768ups|ALBU_HUMAN_UPS", "P04040ups|CATA_HUMAN_UPS", "P06732ups|KCRM_HUMAN_UPS", "P12081ups|SYHC_HUMAN_UPS", "P15559ups|NQO1_HUMAN_UPS", "P16083ups|NQO2_HUMAN_UPS", "P41159ups|LEP_HUMAN_UPS", "P55957ups|BID_HUMAN_UPS", "P61626ups|LYSC_HUMAN_UPS", "P61769ups|B2MG_HUMAN_UPS", "P62937ups|PPIA_HUMAN_UPS", "P62988ups|UBIQ_HUMAN_UPS", "P63165ups|SUMO1_HUMAN_UPS", "P68871ups|HBB_HUMAN_UPS", "P69905ups|HBA_HUMAN_UPS", "Q06830ups|PRDX1_HUMAN_UPS", "O76070ups|SYUG_HUMAN_UPS", "P00167ups|CYB5_HUMAN_UPS", "P00709ups|LALBA_HUMAN_UPS", "P00915ups|CAH1_HUMAN_UPS", "P00918ups|CAH2_HUMAN_UPS", "P01008ups|ANT3_HUMAN_UPS", "P01127ups|PDGFB_HUMAN_UPS", "P01133ups|EGF_HUMAN_UPS", "P01344ups|IGF2_HUMAN_UPS", "P02144ups|MYG_HUMAN_UPS", "P02753ups|RETBP_HUMAN_UPS", "P02768ups|ALBU_HUMAN_UPS", "P04040ups|CATA_HUMAN_UPS", "P06732ups|KCRM_HUMAN_UPS", "P12081ups|SYHC_HUMAN_UPS", "P15559ups|NQO1_HUMAN_UPS", "P16083ups|NQO2_HUMAN_UPS", "P41159ups|LEP_HUMAN_UPS", "P61626ups|LYSC_HUMAN_UPS", "P61769ups|B2MG_HUMAN_UPS", "P62937ups|PPIA_HUMAN_UPS", "P62988ups|UBIQ_HUMAN_UPS", "P63165ups|SUMO1_HUMAN_UPS", "P68871ups|HBB_HUMAN_UPS", "P69905ups|HBA_HUMAN_UPS", "Q06830ups|PRDX1_HUMAN_UPS"), concentration = c("0.449", "44.9", "4.49", "?", "?", "?", "449", "0.449", "44.9", "0.449", "?", "?", "?", "44.9", "?", "4.49", "44.9", "4.49", "449", "0.449", "4.49", "0.449", "44.9", "449", "?", "?", "449", "?", "0.449", "44.9", "4.49", "?", "449", "0.449", "449", "?", "44.9", "0.449", "?", "4.49", "449", "?", "4.49", "?", "?", "?", "449", "0.449", "4.49", "0.449", "?", "449", "?", "449", "?", "44.9", "0.449", "?", "?", "449", "449", "0.449", "?", "44.9", "0.449", "44.9", "?", "?", "44.9", "4.49", "?", "?", "4.49", "?", "?", "0.449", "44.9", "449", "44.9", "449", "?", "44.9"), response = c(161978.666666667, 13113548.6666667, 568833.666666667, 37676253.3333333, 25882206.3333333, 74258.3333333333, 15900608.3333333, 46324, 3684505, 89774, 9086594.33333333, 645838.666666667, 73089941.6666667, 3486194.5, 549538, 507207, 2309451.5, 589519.666666667, 41890138.6666667, 19155.3333333333, 256876, 28007, 11569710.6666667, 26079521.3333333, 4224907.33333333, 24090440.6666667, 17080367.3333333, 3852184.66666667, 161978.666666667, 13113548.6666667, 568833.666666667, 37676253.3333333, 25882206.3333333, 74258.3333333333, 15900608.3333333, 46324, 3684505, 89774, 9086594.33333333, 645838.666666667, 73089941.6666667, 3486194.5, 549538, 507207, 2309451.5, 589519.666666667, 41890138.6666667, 19155.3333333333, 256876, 28007, 11569710.6666667, 26079521.3333333, 4224907.33333333, 24090440.6666667, 17080367.3333333, 3852184.66666667, 161978.666666667, 13113548.6666667, 568833.666666667, 37676253.3333333, 25882206.3333333, 74258.3333333333, 46324, 3684505, 89774, 9086594.33333333, 645838.666666667, 73089941.6666667, 3486194.5, 549538, 507207, 2309451.5, 589519.666666667, 41890138.6666667, 256876, 28007, 11569710.6666667, 26079521.3333333, 4224907.33333333, 24090440.6666667, 17080367.3333333, 3852184.66666667)), .Names = c("run_id", "protein_id", "concentration", "response"), row.names = c(NA, -82L), class = "data.frame")))

	expect_that(length(unique(ProteinInference.default(UPS2_SRM_REP,consensus_proteins=TRUE,consensus_peptide=TRUE,consensus_transitions=TRUE)$protein_id)),equals(26))

	expect_that(length(unique(ProteinInference.default(UPS2_SRM_REP,consensus_proteins=FALSE,consensus_peptide=TRUE,consensus_transitions=TRUE)$protein_id)),equals(28))

	expect_that(length(unique(ProteinInference.default(UPS2_SRM_REP,consensus_proteins=FALSE,consensus_peptide=FALSE,consensus_transitions=TRUE)$protein_id)),equals(28))

	expect_that(length(unique(ProteinInference.default(UPS2_SRM_REP,consensus_proteins=FALSE,consensus_peptide=FALSE,consensus_transitions=FALSE)$protein_id)),equals(28))

	expect_that(length(unique(ProteinInference.default(UPS2_SRM_REP,consensus_proteins=FALSE,consensus_peptide=TRUE,consensus_transitions=FALSE)$protein_id)),equals(28))

	expect_that(length(unique(ProteinInference.default(UPS2_SRM_REP,consensus_proteins=TRUE,consensus_peptide=TRUE,consensus_transitions=FALSE)$protein_id)),equals(26))
})

# peptide_data
peptide_data<-data.table("run_id"=c(1,1,1,1,2,2,2,2),"protein_id"=c("PROTEINA","PROTEINA","PROTEINB","PROTEINB","PROTEINA","PROTEINA","PROTEINB","PROTEINB"),"peptide_id"=c("ANDPEPTIDEA","PEPTIDEK","PEPTIDEC","PEPTIDEC","PEPTIDEK","PEPTIDEK","PEPTIDEC","PEPTIDEC"),"peptide_sequence"=c("ANDPEPTIDEA","PEPTIDEK","PEPTIDEC","PEPTIDEC","PEPTIDEK","PEPTIDEK","PEPTIDEC","PEPTIDEC"),"precursor_charge"=c(2,2,2,3,2,3,2,3),"peptide_intensity"=c(100,80,60,40,95,75,55,35),concentration=c("?","?","?","?","?","?","?","?"),stringsAsFactors=FALSE)

# protein_inference.ProteinInference
test_that("protein_inference.ProteinInference: Consensus peptide selection", {
	expect_that(protein_inference.ProteinInference(peptide_data, peptide_method = "top", peptide_topx = 3, peptide_strictness = "loose", peptide_summary = "mean", fasta = NA, model = NA, combine_precursors = FALSE, consensus_peptides = TRUE)$response,equals(c(80,50,95,45)))
	expect_that(protein_inference.ProteinInference(peptide_data, peptide_method = "top", peptide_topx = 2, peptide_strictness = "loose", peptide_summary = "mean", fasta = NA, model = NA, combine_precursors = FALSE, consensus_peptides = TRUE)$response,equals(c(80,50,95,45)))
	expect_that(protein_inference.ProteinInference(peptide_data, peptide_method = "top", peptide_topx = 1, peptide_strictness = "loose", peptide_summary = "mean", fasta = NA, model = NA, combine_precursors = FALSE, consensus_peptides = TRUE)$response,equals(c(80,60,95,55)))
	expect_that(protein_inference.ProteinInference(peptide_data, peptide_method = "top", peptide_topx = 3, peptide_strictness = "loose", peptide_summary = "mean", fasta = NA, model = NA, combine_precursors = FALSE, consensus_peptides = FALSE)$response,equals(c(90,50,85,45)))
	expect_that(protein_inference.ProteinInference(peptide_data, peptide_method = "top", peptide_topx = 2, peptide_strictness = "loose", peptide_summary = "mean", fasta = NA, model = NA, combine_precursors = FALSE, consensus_peptides = FALSE)$response,equals(c(90,50,85,45)))
	expect_that(protein_inference.ProteinInference(peptide_data, peptide_method = "top", peptide_topx = 1, peptide_strictness = "loose", peptide_summary = "mean", fasta = NA, model = NA, combine_precursors = FALSE, consensus_peptides = FALSE)$response,equals(c(100,60,95,55)))
})

test_that("protein_inference.ProteinInference: Combine precursors", {
	expect_that(protein_inference.ProteinInference(peptide_data, peptide_method = "top", peptide_topx = 1, peptide_strictness = "loose", peptide_summary = "mean", fasta = NA, model = NA, combine_precursors = FALSE, consensus_peptides = FALSE)$response,equals(c(100,60,95,55)))
	expect_that(protein_inference.ProteinInference(peptide_data, peptide_method = "top", peptide_topx = 1, peptide_strictness = "loose", peptide_summary = "mean", fasta = NA, model = NA, combine_precursors = TRUE, consensus_peptides = FALSE)$response,equals(c(100,100,170,90)))
})

test_that("protein_inference.ProteinInference: Summarization of peptides", {
	expect_that(protein_inference.ProteinInference(peptide_data, peptide_method = "top", peptide_topx = 2, peptide_strictness = "loose", peptide_summary = "sum", fasta = NA, model = NA, combine_precursors = FALSE, consensus_peptides = FALSE)$response,equals(c(180,100,170,90)))
	expect_that(protein_inference.ProteinInference(peptide_data, peptide_method = "top", peptide_topx = 2, peptide_strictness = "loose", peptide_summary = "mean", fasta = NA, model = NA, combine_precursors = FALSE, consensus_peptides = FALSE)$response,equals(c(90,50,85,45)))
	expect_that(protein_inference.ProteinInference(peptide_data, peptide_method = "top", peptide_topx = 2, peptide_strictness = "loose", peptide_summary = "median", fasta = NA, model = NA, combine_precursors = FALSE, consensus_peptides = FALSE)$response,equals(c(90,50,85,45)))
})

test_that("protein_inference.ProteinInference: topx selection criterion", {
	expect_that(protein_inference.ProteinInference(peptide_data, peptide_method = "top", peptide_topx = 1, peptide_strictness = "loose", peptide_summary = "mean", fasta = NA, model = NA, combine_precursors = FALSE, consensus_peptides = FALSE)$response,equals(c(100,60,95,55)))
	expect_that(protein_inference.ProteinInference(peptide_data, peptide_method = "top", peptide_topx = 2, peptide_strictness = "loose", peptide_summary = "mean", fasta = NA, model = NA, combine_precursors = FALSE, consensus_peptides = FALSE)$response,equals(c(90,50,85,45)))
	expect_that(protein_inference.ProteinInference(peptide_data, peptide_method = "top", peptide_topx = 3, peptide_strictness = "loose", peptide_summary = "mean", fasta = NA, model = NA, combine_precursors = FALSE, consensus_peptides = FALSE)$response,equals(c(90,50,85,45)))
})

test_that("protein_inference.ProteinInference: loose / strict selection criterion", {
	expect_that(protein_inference.ProteinInference(peptide_data, peptide_method = "top", peptide_topx = 3, peptide_strictness = "loose", peptide_summary = "mean", fasta = NA, model = NA, combine_precursors = FALSE, consensus_peptides = FALSE)$response,equals(c(90,50,85,45)))
	expect_that(protein_inference.ProteinInference(peptide_data, peptide_method = "top", peptide_topx = 3, peptide_strictness = "strict", peptide_summary = "mean", fasta = NA, model = NA, combine_precursors = FALSE, consensus_peptides = FALSE)$response,equals(c(0,0,0,0)))
})

test_that("protein_inference.ProteinInference: all peptide_method", {
	expect_that(protein_inference.ProteinInference(peptide_data, peptide_method = "all", peptide_summary = "mean", fasta = NA, model = NA, combine_precursors = FALSE, consensus_peptides = FALSE)$response,equals(c(90,50,85,45)))
	expect_that(protein_inference.ProteinInference(peptide_data, peptide_method = "all", peptide_summary = "median", fasta = NA, model = NA, combine_precursors = FALSE, consensus_peptides = FALSE)$response,equals(c(90,50,85,45)))
	expect_that(protein_inference.ProteinInference(peptide_data, peptide_method = "all", peptide_summary = "sum", fasta = NA, model = NA, combine_precursors = FALSE, consensus_peptides = FALSE)$response,equals(c(180,100,170,90)))
	expect_that(protein_inference.ProteinInference(peptide_data, peptide_method = "all", peptide_summary = "mean", fasta = NA, model = NA, combine_precursors = FALSE, consensus_peptides = TRUE)$response,equals(c(80,50,95,45)))
	expect_that(protein_inference.ProteinInference(peptide_data, peptide_method = "all", peptide_summary = "median", fasta = NA, model = NA, combine_precursors = FALSE, consensus_peptides = TRUE)$response,equals(c(80,50,95,45)))
	expect_that(protein_inference.ProteinInference(peptide_data, peptide_method = "all", peptide_summary = "sum", fasta = NA, model = NA, combine_precursors = FALSE, consensus_peptides = TRUE)$response,equals(c(80,100,95,90)))
})

test_that("protein_inference.ProteinInference: iBAQ peptide_method", {
	data(UPS2MS)

	expect_that(protein_inference.ProteinInference(data.table(UPS2_LFQ), peptide_method = "iBAQ", peptide_summary = "sum", fasta = system.file("extdata","UPS2.fasta",package="aLFQ"), model = NA, combine_precursors = FALSE, consensus_peptides = TRUE)$response,equals(c(25106081.31,125856278.00,87587312.93,95134656.33,17871715.00,6529159.17,28423.72,208145399.64,14028046.12,171435.00,347060.15,3673181.00,431140.67,156229765.07,1045140.00,12268096.30,60432526.25,4489209.67,67995597.05,37791831.53,16983291.54)))
	expect_that(protein_inference.ProteinInference(peptide_data, peptide_method = "iBAQ", peptide_summary = "sum", fasta = system.file("extdata","example.fasta",package="aLFQ"), model = NA, combine_precursors = FALSE, consensus_peptides = TRUE)$response,equals(c(40,100,47.5,90)))
})

test_that("protein_inference.ProteinInference: APEX peptide_method", {
	set.seed(131)

	data(APEXMS)

	APEX_ORBI<-head(APEX_ORBI,50) # Remove this line for real applications
	APEX_ORBI.af <- apexFeatures(APEX_ORBI)
	APEX_ORBI.apex <- APEX(data=APEX_ORBI.af)

	expect_that(dim(protein_inference.ProteinInference(peptide_data, peptide_method = "APEX", peptide_summary = "sum", fasta = system.file("extdata","example.fasta",package="aLFQ"), model = APEX_ORBI.apex, combine_precursors = FALSE, consensus_peptides = TRUE))[2],equals(4))
	expect_that(dim(protein_inference.ProteinInference(peptide_data, peptide_method = "APEX", peptide_summary = "sum", fasta = system.file("extdata","example.fasta",package="aLFQ"), model = APEX_ORBI.apex, combine_precursors = FALSE, consensus_peptides = FALSE))[2],equals(4))
})

test_that("protein_inference.ProteinInference: NSAF peptide_method", {
	expect_that(protein_inference.ProteinInference(peptide_data, peptide_method = "NSAF", peptide_summary = "sum", fasta = system.file("extdata","example.fasta",package="aLFQ"), model = NA, combine_precursors = FALSE, consensus_peptides = TRUE)$response,equals(c(0.1441441,0.4279279,0.1818182,0.4090909), tolerance = .001))
})

# transition_data
transition_data<-data.table("run_id"=c(1,1,1,1,2,2,2,2,3,3,3,3),"protein_id"=c("prot3","prot3","prot2","prot2","prot1","prot1","prot2","prot2","prot1","prot1","prot2","prot2"),"peptide_id"=c("pep1","pep1","pep2","pep2","pep1","pep1","pep2","pep2","pep1","pep1","pep2","pep2"),"peptide_sequence"=c("PEPTIDEA","PEPTIDEA","PEPTIDEC","PEPTIDEC","PEPTIDEA","PEPTIDEA","PEPTIDEC","PEPTIDEC","PEPTIDEA","PEPTIDEA","PEPTIDEC","PEPTIDEC"),"precursor_charge"=c(2,2,3,3,2,2,3,3,2,2,3,3),"transition_id"=c(1,2,3,4,5,2,3,4,5,2,3,4),"transition_intensity"=c(100,80,60,40,95,75,55,35,95,75,55,35),concentration=c("?","?","?","?","?","?","?","?","?","?","?","?"),stringsAsFactors=FALSE)

# peptide_inference.ProteinInference
test_that("peptide_inference.ProteinInference: Consensus protein / transition selection", {
	expect_that(peptide_inference.ProteinInference(transition_data, transition_topx = 3, transition_strictness = "loose", transition_summary = "sum", consensus_proteins = TRUE, consensus_transitions = TRUE)$peptide_intensity,equals(c(100,90,90)))
	expect_that(peptide_inference.ProteinInference(transition_data, transition_topx = 2, transition_strictness = "loose", transition_summary = "sum", consensus_proteins = TRUE, consensus_transitions = TRUE)$peptide_intensity,equals(c(100,90,90)))
	expect_that(peptide_inference.ProteinInference(transition_data, transition_topx = 1, transition_strictness = "loose", transition_summary = "sum", consensus_proteins = TRUE, consensus_transitions = TRUE)$peptide_intensity,equals(c(60,55,55)))
	expect_that(peptide_inference.ProteinInference(transition_data, transition_topx = 3, transition_strictness = "loose", transition_summary = "sum", consensus_proteins = TRUE, consensus_transitions = FALSE)$peptide_intensity,equals(c(100,180,170,90,170,90)))
	expect_that(peptide_inference.ProteinInference(transition_data, transition_topx = 2, transition_strictness = "loose", transition_summary = "sum", consensus_proteins = TRUE, consensus_transitions = FALSE)$peptide_intensity,equals(c(100,180,170,90,170,90)))
	expect_that(peptide_inference.ProteinInference(transition_data, transition_topx = 1, transition_strictness = "loose", transition_summary = "sum", consensus_proteins = TRUE, consensus_transitions = FALSE)$peptide_intensity,equals(c(60,100,95,55,95,55)))
})

test_that("peptide_inference.ProteinInference: Consensus transition selection", {
	expect_that(peptide_inference.ProteinInference(transition_data, transition_topx = 3, transition_strictness = "loose", transition_summary = "sum", consensus_proteins = FALSE, consensus_transitions = TRUE)$peptide_intensity,equals(c(100,180,170,90,170,90)))
	expect_that(peptide_inference.ProteinInference(transition_data, transition_topx = 2, transition_strictness = "loose", transition_summary = "sum", consensus_proteins = FALSE, consensus_transitions = TRUE)$peptide_intensity,equals(c(100,180,170,90,170,90)))
	expect_that(peptide_inference.ProteinInference(transition_data, transition_topx = 1, transition_strictness = "loose", transition_summary = "sum", consensus_proteins = FALSE, consensus_transitions = TRUE)$peptide_intensity,equals(c(60,100,95,55,95,55)))
	expect_that(peptide_inference.ProteinInference(transition_data, transition_topx = 3, transition_strictness = "loose", transition_summary = "sum", consensus_proteins = FALSE, consensus_transitions = FALSE)$peptide_intensity,equals(c(100,180,170,90,170,90)))
	expect_that(peptide_inference.ProteinInference(transition_data, transition_topx = 2, transition_strictness = "loose", transition_summary = "sum", consensus_proteins = FALSE, consensus_transitions = FALSE)$peptide_intensity,equals(c(100,180,170,90,170,90)))
	expect_that(peptide_inference.ProteinInference(transition_data, transition_topx = 1, transition_strictness = "loose", transition_summary = "sum", consensus_proteins = FALSE, consensus_transitions = FALSE)$peptide_intensity,equals(c(60,100,95,55,95,55)))
})


test_that("peptide_inference.ProteinInference: Summarization of peptides", {
	expect_that(peptide_inference.ProteinInference(transition_data, transition_topx = 3, transition_strictness = "loose", transition_summary = "sum", consensus_proteins = TRUE, consensus_transitions = FALSE)$peptide_intensity,equals(c(100,180,170,90,170,90)))
	expect_that(peptide_inference.ProteinInference(transition_data, transition_topx = 3, transition_strictness = "loose", transition_summary = "mean", consensus_proteins = TRUE, consensus_transitions = FALSE)$peptide_intensity,equals(c(50,90,85,45,85,45)))
	expect_that(peptide_inference.ProteinInference(transition_data, transition_topx = 3, transition_strictness = "loose", transition_summary = "median", consensus_proteins = TRUE, consensus_transitions = FALSE)$peptide_intensity,equals(c(50,90,85,45,85,45)))
})

test_that("peptide_inference.ProteinInference: topx selection criterion", {
	expect_that(peptide_inference.ProteinInference(transition_data, transition_topx = 3, transition_strictness = "loose", transition_summary = "sum", consensus_proteins = TRUE, consensus_transitions = FALSE)$peptide_intensity,equals(c(100,180,170,90,170,90)))
	expect_that(peptide_inference.ProteinInference(transition_data, transition_topx = 1, transition_strictness = "loose", transition_summary = "sum", consensus_proteins = TRUE, consensus_transitions = FALSE)$peptide_intensity,equals(c(60,100,95,55,95,55)))
})

test_that("peptide_inference.ProteinInference: loose / strict selection criterion", {
	expect_that(peptide_inference.ProteinInference(transition_data, transition_topx = 3, transition_strictness = "loose", transition_summary = "sum", consensus_proteins = TRUE, consensus_transitions = FALSE)$peptide_intensity,equals(c(100,180,170,90,170,90)))
	expect_that(peptide_inference.ProteinInference(transition_data, transition_topx = 3, transition_strictness = "strict", transition_summary = "sum", consensus_proteins = TRUE, consensus_transitions = FALSE)$peptide_intensity,equals(c(0,0,0,0,0,0)))
})

# peptide_inference.ProteinInference
test_that("peptide_inference.ProteinInference: More complex data", {
	data(UPS2MS)
	set.seed(131)
	REP1<-UPS2_SRM[sample(854,800),]
	REP1[which(REP1$protein_id %in% unique(REP1$protein_id)[1:10]),]$concentration<-"?"
	REP1$run_id<-"UPS2_SRM_REP1"

	REP2<-UPS2_SRM[sample(854,800),]
	REP2[which(REP2$protein_id %in% unique(REP2$protein_id)[1:10]),]$concentration<-"?"
	REP2$run_id<-"UPS2_SRM_REP2"

	REP3<-UPS2_SRM[sample(854,800),]
	REP3[which(REP3$protein_id %in% unique(REP3$protein_id)[1:10]),]$concentration<-"?"
	REP3$run_id<-"UPS2_SRM_REP3"

	REP3<-REP3[-which(REP3$protein_id %in% c("P01031ups|CO5_HUMAN_UPS","P55957ups|BID_HUMAN_UPS")),]
	REP3<-REP3[-which(REP3$transition_id %in% c("y5","y4","y8","y5")),]

	UPS2_SRM_REP<-data.table(rbind(REP1,REP2,REP3))

	expect_that(peptide_inference.ProteinInference(UPS2_SRM_REP, transition_topx = 3, transition_strictness = "loose", transition_summary = "sum", consensus_proteins = TRUE, consensus_transitions = TRUE)$peptide_intensity,equals(c(25590.3333333333, 19487.8333333333, 94881.6666666667, 178646.666666667, 6015030.33333333, 627038.333333333, 11635244.6666667, 736551, 1073336.33333333, 11974505, 3786579, 218245.666666667, 24880.3333333333, 72675.1666666667, 65261.5, 218894, 5963012.66666667, 8408505, 22734358, 12761857.3333333, 20272967.3333333, 35512577.6666667, 10497302, 3865748.33333333, 241238.333333333, 16276914.3333333, 15558374.6666667, 6822498.83333333, 1277186.66666667, 11749037.6666667, 21737299.6666667, 1846759, 1660423, 7720861.33333333, 13014537.3333333, 14559.3333333333, 67646, 63101.3333333333, 22536, 30531, 3684505, 59372.3333333333, 35372, 37802.3333333333, 1035790, 453902.333333333, 446332.333333333, 365018.666666667, 1809386.66666667, 9011571, 645838.666666667, 106781.333333333, 11963.3333333333, 65084.3333333333, 74667.6666666667, 63633, 17660313.3333333, 59667539.3333333, 24996562.3333333, 7886728.16666667, 2654978, 6584263.33333333, 66764209.8333333, 37522615, 2694749.66666667, 400530, 300637, 1759926, 784371, 1644341.33333333, 2054531.5, 239703.333333333, 3074869, 1143588.33333333, 63168.6666666667, 423062, 1354893.66666667, 169604, 110562.666666667, 25090, 140672.333333333, 173531.666666667, 40671, 370075.666666667, 82473.3333333333, 187492.666666667, 309984.333333333, 113787.666666667, 93361.6666666667, 142269.166666667, 179003.333333333, 244661.333333333, 151754.666666667, 548598.833333333, 1117559.5, 775374.333333333, 1002640, 244527, 600798, 17368.6666666667, 237515.666666667, 47599.6666666667, 87266.5, 561006.666666667, 6744154.16666667, 4181555.33333333, 11341209, 12986979, 1359312.66666667, 17666858.6666667, 144644.666666667, 2355843.16666667, 41890138.6666667, 133970.333333333, 114110.666666667, 124021.333333333, 208360.333333333, 6543.33333333333, 18754.3333333333, 15237.3333333333, 446034.333333333, 1050469.33333333, 11569710.6666667, 302367.333333333, 163573.333333333, 1064324, 10907174.3333333, 8312024.66666667, 24264809, 797923.333333333, 953073, 974526.666666667, 870937, 376157.5, 3739507.66666667, 13866442.1666667, 1214256.33333333, 387412.333333333, 130142, 11279567.6666667, 6411995.66666667, 1353513.33333333, 19411749.6666667, 18222098, 3377765, 1290837, 3380810, 870910.333333333, 15916711, 106819, 2900604.66666667, 7956447, 2826125, 1371032.66666667, 1778378, 2036743.33333333, 670967.333333333, 25590.3333333333, 19487.8333333333, 94881.6666666667, 178646.666666667, 6015030.33333333, 627038.333333333, 11635244.6666667, 736551, 1073336.33333333, 11974505, 3786579, 218245.666666667, 24880.3333333333, 72675.1666666667, 65261.5, 218894, 5963012.66666667, 8408505, 22734358, 12761857.3333333, 20272967.3333333, 35512577.6666667, 10497302, 3865748.33333333, 241238.333333333, 16276914.3333333, 15558374.6666667, 6822498.83333333, 1277186.66666667, 11749037.6666667, 21737299.6666667, 1846759, 1660423, 7720861.33333333, 13014537.3333333, 14559.3333333333, 67646, 63101.3333333333, 22536, 30531, 3684505, 59372.3333333333, 35372, 37802.3333333333, 1035790, 453902.333333333, 446332.333333333, 365018.666666667, 1809386.66666667, 9011571, 645838.666666667, 106781.333333333, 11963.3333333333, 65084.3333333333, 74667.6666666667, 63633, 17660313.3333333, 59667539.3333333, 24996562.3333333, 7886728.16666667, 2654978, 6584263.33333333, 66764209.8333333, 37522615, 2694749.66666667, 400530, 300637, 1759926, 784371, 1644341.33333333, 2054531.5, 239703.333333333, 3074869, 1143588.33333333, 63168.6666666667, 423062, 1354893.66666667, 169604, 110562.666666667, 25090, 140672.333333333, 173531.666666667, 40671, 370075.666666667, 82473.3333333333, 187492.666666667, 309984.333333333, 113787.666666667, 93361.6666666667, 142269.166666667, 179003.333333333, 244661.333333333, 151754.666666667, 548598.833333333, 1117559.5, 775374.333333333, 1002640, 244527, 600798, 17368.6666666667, 237515.666666667, 47599.6666666667, 87266.5, 561006.666666667, 6744154.16666667, 4181555.33333333, 11341209, 12986979, 1359312.66666667, 17666858.6666667, 144644.666666667, 2355843.16666667, 41890138.6666667, 133970.333333333, 114110.666666667, 124021.333333333, 208360.333333333, 6543.33333333333, 18754.3333333333, 15237.3333333333, 446034.333333333, 1050469.33333333, 11569710.6666667, 302367.333333333, 163573.333333333, 1064324, 10907174.3333333, 8312024.66666667, 24264809, 797923.333333333, 953073, 974526.666666667, 870937, 376157.5, 3739507.66666667, 13866442.1666667, 1214256.33333333, 387412.333333333, 130142, 11279567.6666667, 6411995.66666667, 1353513.33333333, 19411749.6666667, 18222098, 3377765, 1290837, 3380810, 870910.333333333, 15916711, 106819, 2900604.66666667, 7956447, 2826125, 1371032.66666667, 1778378, 2036743.33333333, 670967.333333333, 25590.3333333333, 19487.8333333333, 94881.6666666667, 178646.666666667, 6015030.33333333, 627038.333333333, 11635244.6666667, 736551, 1073336.33333333, 11974505, 3786579, 218245.666666667, 24880.3333333333, 72675.1666666667, 65261.5, 218894, 5963012.66666667, 8408505, 22734358, 12761857.3333333, 20272967.3333333, 35512577.6666667, 10497302, 3865748.33333333, 241238.333333333, 16276914.3333333, 15558374.6666667, 6822498.83333333, 1277186.66666667, 11749037.6666667, 21737299.6666667, 1846759, 1660423, 7720861.33333333, 13014537.3333333, 14559.3333333333, 67646, 63101.3333333333, 22536, 30531, 3684505, 59372.3333333333, 35372, 37802.3333333333, 1035790, 453902.333333333, 446332.333333333, 365018.666666667, 1809386.66666667, 9011571, 645838.666666667, 106781.333333333, 11963.3333333333, 65084.3333333333, 74667.6666666667, 63633, 17660313.3333333, 59667539.3333333, 24996562.3333333, 7886728.16666667, 2654978, 6584263.33333333, 66764209.8333333, 37522615, 2694749.66666667, 400530, 300637, 1759926, 784371, 1644341.33333333, 2054531.5, 239703.333333333, 3074869, 1143588.33333333, 63168.6666666667, 423062, 1354893.66666667, 169604, 110562.666666667, 25090, 140672.333333333, 173531.666666667, 40671, 370075.666666667, 82473.3333333333, 187492.666666667, 309984.333333333, 113787.666666667, 93361.6666666667, 142269.166666667, 179003.333333333, 244661.333333333, 151754.666666667, 548598.833333333, 1117559.5, 775374.333333333, 1002640, 244527, 600798, 17368.6666666667, 237515.666666667, 47599.6666666667, 87266.5, 561006.666666667, 6744154.16666667, 4181555.33333333, 11341209, 12986979, 1359312.66666667, 17666858.6666667, 144644.666666667, 2355843.16666667, 41890138.6666667, 133970.333333333, 114110.666666667, 124021.333333333, 208360.333333333, 6543.33333333333, 18754.3333333333, 15237.3333333333, 446034.333333333, 1050469.33333333, 11569710.6666667, 302367.333333333, 163573.333333333, 1064324, 10907174.3333333, 8312024.66666667, 24264809, 797923.333333333, 953073, 974526.666666667, 870937, 376157.5, 3739507.66666667, 13866442.1666667, 1214256.33333333, 387412.333333333, 130142, 11279567.6666667, 6411995.66666667, 1353513.33333333, 19411749.6666667, 18222098, 3377765, 1290837, 3380810, 870910.333333333, 15916711, 106819, 2900604.66666667, 7956447, 2826125, 1371032.66666667, 1778378, 2036743.33333333, 670967.333333333)))

	expect_that(peptide_inference.ProteinInference(UPS2_SRM_REP, transition_topx = 3, transition_strictness = "loose", transition_summary = "sum", consensus_proteins = FALSE, consensus_transitions = TRUE)$peptide_intensity,equals(c(25590.3333333333, 19487.8333333333, 94881.6666666667, 178646.666666667, 6015030.33333333, 627038.333333333, 11635244.6666667, 736551, 1073336.33333333, 11974505, 3786579, 218245.666666667, 24880.3333333333, 72675.1666666667, 65261.5, 218894, 5963012.66666667, 8408505, 22734358, 12761857.3333333, 20272967.3333333, 35512577.6666667, 10497302, 3865748.33333333, 241238.333333333, 16276914.3333333, 15558374.6666667, 6822498.83333333, 1277186.66666667, 11749037.6666667, 21737299.6666667, 1846759, 1660423, 7720861.33333333, 13014537.3333333, 14559.3333333333, 67646, 63101.3333333333, 15900608.3333333, 22536, 30531, 3684505, 59372.3333333333, 35372, 37802.3333333333, 1035790, 453902.333333333, 446332.333333333, 365018.666666667, 1809386.66666667, 9011571, 645838.666666667, 106781.333333333, 11963.3333333333, 65084.3333333333, 74667.6666666667, 63633, 17660313.3333333, 59667539.3333333, 24996562.3333333, 7886728.16666667, 2654978, 6584263.33333333, 66764209.8333333, 37522615, 2694749.66666667, 400530, 300637, 1759926, 784371, 1644341.33333333, 2054531.5, 239703.333333333, 3074869, 1143588.33333333, 63168.6666666667, 423062, 1354893.66666667, 169604, 110562.666666667, 25090, 140672.333333333, 173531.666666667, 40671, 370075.666666667, 82473.3333333333, 187492.666666667, 309984.333333333, 113787.666666667, 93361.6666666667, 142269.166666667, 179003.333333333, 244661.333333333, 151754.666666667, 548598.833333333, 1117559.5, 775374.333333333, 1002640, 244527, 600798, 17368.6666666667, 237515.666666667, 47599.6666666667, 87266.5, 561006.666666667, 6744154.16666667, 4181555.33333333, 11341209, 12986979, 1359312.66666667, 17666858.6666667, 144644.666666667, 2355843.16666667, 41890138.6666667, 19155.3333333333, 133970.333333333, 114110.666666667, 124021.333333333, 208360.333333333, 6543.33333333333, 18754.3333333333, 15237.3333333333, 446034.333333333, 1050469.33333333, 11569710.6666667, 302367.333333333, 163573.333333333, 1064324, 10907174.3333333, 8312024.66666667, 24264809, 797923.333333333, 953073, 974526.666666667, 870937, 376157.5, 3739507.66666667, 13866442.1666667, 1214256.33333333, 387412.333333333, 130142, 11279567.6666667, 6411995.66666667, 1353513.33333333, 19411749.6666667, 18222098, 3377765, 1290837, 3380810, 870910.333333333, 15916711, 106819, 2900604.66666667, 7956447, 2826125, 1371032.66666667, 1778378, 2036743.33333333, 670967.333333333, 25590.3333333333, 19487.8333333333, 94881.6666666667, 178646.666666667, 6015030.33333333, 627038.333333333, 11635244.6666667, 736551, 1073336.33333333, 11974505, 3786579, 218245.666666667, 24880.3333333333, 72675.1666666667, 65261.5, 218894, 5963012.66666667, 8408505, 22734358, 12761857.3333333, 20272967.3333333, 35512577.6666667, 10497302, 3865748.33333333, 241238.333333333, 16276914.3333333, 15558374.6666667, 6822498.83333333, 1277186.66666667, 11749037.6666667, 21737299.6666667, 1846759, 1660423, 7720861.33333333, 13014537.3333333, 14559.3333333333, 67646, 63101.3333333333, 15900608.3333333, 22536, 30531, 3684505, 59372.3333333333, 35372, 37802.3333333333, 1035790, 453902.333333333, 446332.333333333, 365018.666666667, 1809386.66666667, 9011571, 645838.666666667, 106781.333333333, 11963.3333333333, 65084.3333333333, 74667.6666666667, 63633, 17660313.3333333, 59667539.3333333, 24996562.3333333, 7886728.16666667, 2654978, 6584263.33333333, 66764209.8333333, 37522615, 2694749.66666667, 400530, 300637, 1759926, 784371, 1644341.33333333, 2054531.5, 239703.333333333, 3074869, 1143588.33333333, 63168.6666666667, 423062, 1354893.66666667, 169604, 110562.666666667, 25090, 140672.333333333, 173531.666666667, 40671, 370075.666666667, 82473.3333333333, 187492.666666667, 309984.333333333, 113787.666666667, 93361.6666666667, 142269.166666667, 179003.333333333, 244661.333333333, 151754.666666667, 548598.833333333, 1117559.5, 775374.333333333, 1002640, 244527, 600798, 17368.6666666667, 237515.666666667, 47599.6666666667, 87266.5, 561006.666666667, 6744154.16666667, 4181555.33333333, 11341209, 12986979, 1359312.66666667, 17666858.6666667, 144644.666666667, 2355843.16666667, 41890138.6666667, 19155.3333333333, 133970.333333333, 114110.666666667, 124021.333333333, 208360.333333333, 6543.33333333333, 18754.3333333333, 15237.3333333333, 446034.333333333, 1050469.33333333, 11569710.6666667, 302367.333333333, 163573.333333333, 1064324, 10907174.3333333, 8312024.66666667, 24264809, 797923.333333333, 953073, 974526.666666667, 870937, 376157.5, 3739507.66666667, 13866442.1666667, 1214256.33333333, 387412.333333333, 130142, 11279567.6666667, 6411995.66666667, 1353513.33333333, 19411749.6666667, 18222098, 3377765, 1290837, 3380810, 870910.333333333, 15916711, 106819, 2900604.66666667, 7956447, 2826125, 1371032.66666667, 1778378, 2036743.33333333, 670967.333333333, 25590.3333333333, 19487.8333333333, 94881.6666666667, 178646.666666667, 6015030.33333333, 627038.333333333, 11635244.6666667, 736551, 1073336.33333333, 11974505, 3786579, 218245.666666667, 24880.3333333333, 72675.1666666667, 65261.5, 218894, 5963012.66666667, 8408505, 22734358, 12761857.3333333, 20272967.3333333, 35512577.6666667, 10497302, 3865748.33333333, 241238.333333333, 16276914.3333333, 15558374.6666667, 6822498.83333333, 1277186.66666667, 11749037.6666667, 21737299.6666667, 1846759, 1660423, 7720861.33333333, 13014537.3333333, 14559.3333333333, 67646, 63101.3333333333, 22536, 30531, 3684505, 59372.3333333333, 35372, 37802.3333333333, 1035790, 453902.333333333, 446332.333333333, 365018.666666667, 1809386.66666667, 9011571, 645838.666666667, 106781.333333333, 11963.3333333333, 65084.3333333333, 74667.6666666667, 63633, 17660313.3333333, 59667539.3333333, 24996562.3333333, 7886728.16666667, 2654978, 6584263.33333333, 66764209.8333333, 37522615, 2694749.66666667, 400530, 300637, 1759926, 784371, 1644341.33333333, 2054531.5, 239703.333333333, 3074869, 1143588.33333333, 63168.6666666667, 423062, 1354893.66666667, 169604, 110562.666666667, 25090, 140672.333333333, 173531.666666667, 40671, 370075.666666667, 82473.3333333333, 187492.666666667, 309984.333333333, 113787.666666667, 93361.6666666667, 142269.166666667, 179003.333333333, 244661.333333333, 151754.666666667, 548598.833333333, 1117559.5, 775374.333333333, 1002640, 244527, 600798, 17368.6666666667, 237515.666666667, 47599.6666666667, 87266.5, 561006.666666667, 6744154.16666667, 4181555.33333333, 11341209, 12986979, 1359312.66666667, 17666858.6666667, 144644.666666667, 2355843.16666667, 41890138.6666667, 133970.333333333, 114110.666666667, 124021.333333333, 208360.333333333, 6543.33333333333, 18754.3333333333, 15237.3333333333, 446034.333333333, 1050469.33333333, 11569710.6666667, 302367.333333333, 163573.333333333, 1064324, 10907174.3333333, 8312024.66666667, 24264809, 797923.333333333, 953073, 974526.666666667, 870937, 376157.5, 3739507.66666667, 13866442.1666667, 1214256.33333333, 387412.333333333, 130142, 11279567.6666667, 6411995.66666667, 1353513.33333333, 19411749.6666667, 18222098, 3377765, 1290837, 3380810, 870910.333333333, 15916711, 106819, 2900604.66666667, 7956447, 2826125, 1371032.66666667, 1778378, 2036743.33333333, 670967.333333333)))

	expect_that(peptide_inference.ProteinInference(UPS2_SRM_REP, transition_topx = 3, transition_strictness = "loose", transition_summary = "sum", consensus_proteins = TRUE, consensus_transitions = FALSE)$peptide_intensity,equals(c(114462.666666667, 43780.5, 161978.666666667, 257581.666666667, 6991767, 2152459, 13113548.6666667, 913342.333333333, 1189364, 11974505, 4045589.66666667, 307874, 106052, 104507.833333333, 80093.8333333333, 568833.666666667, 8569949.66666667, 14146125, 28568480.6666667, 16652541.6666667, 21725352.6666667, 37676253.3333333, 17609389, 3951486.33333333, 437893.666666667, 17784180.6666667, 25882206.3333333, 7208210.5, 7942106.66666667, 15588755, 22443725.3333333, 2145055.33333333, 14716619, 7807785.66666667, 17482423.6666667, 19404.3333333333, 74258.3333333333, 63101.3333333333, 60670316.6666667, 22536, 46324, 3684505, 74352, 106152.666666667, 69104.6666666667, 1142502.66666667, 459295.666666667, 515593, 632950.333333333, 1809386.66666667, 9793640.33333333, 670395, 109830.666666667, 33767.8333333333, 86128.6666666667, 113786.333333333, 193290.333333333, 19889342.3333333, 73089941.6666667, 35516393.6666667, 24499944, 7979479.66666667, 6676686, 66764209.8333333, 38683439.6666667, 3279503.33333333, 548813.5, 649076.166666667, 2852016.66666667, 816513, 2303601.33333333, 3486194.5, 1501057.5, 3074869, 1192006.33333333, 63168.6666666667, 423062, 5000338.66666667, 169604, 329493, 39498.3333333333, 162739.666666667, 232968, 55588, 549538, 183531.666666667, 278020.333333333, 695690.333333333, 284204.666666667, 93361.6666666667, 142269.166666667, 206419, 274470.333333333, 391611.666666667, 934322.666666667, 2309451.5, 849879.333333333, 1297497.66666667, 270146, 653256, 96798.6666666667, 237515.666666667, 77191.6666666667, 90460.5, 589519.666666667, 8358047.16666667, 8841519.66666667, 11341209, 12986979, 1359312.66666667, 28657459.6666667, 298191.333333333, 2664756.83333333, 41890138.6666667, 19155.3333333333, 164348.666666667, 170885, 58722.3333333333, 256876, 235343.666666667, 6543.33333333333, 49325, 49716.6666666667, 702563.666666667, 1051786, 11580894.3333333, 2199387.66666667, 182419.333333333, 6442124, 16932987.8333333, 18537309.6666667, 26079521.3333333, 4608205, 1234254, 1090341.33333333, 889262, 452304, 3739507.66666667, 15002957.8333333, 1772142.66666667, 1429487, 171733, 22825749.3333333, 13705634, 1456951, 20128782.6666667, 24090440.6666667, 3377765, 1552218.33333333, 3756792.66666667, 3445441.33333333, 22574059.6666667, 1690357.66666667, 3014251, 17080367.3333333, 3852184.66666667, 2157911.66666667, 2397813.66666667, 5444638.66666667, 4467382.66666667, 114462.666666667, 43780.5, 161978.666666667, 257581.666666667, 6976591.66666667, 2152459, 13113548.6666667, 913342.333333333, 1189364, 11974505, 4045589.66666667, 307874, 101104.666666667, 104507.833333333, 80093.8333333333, 568833.666666667, 8569949.66666667, 25460282, 28568480.6666667, 16652541.6666667, 21725352.6666667, 37676253.3333333, 17609389, 3951486.33333333, 437893.666666667, 17784180.6666667, 25882206.3333333, 7208210.5, 8968420.66666667, 15588755, 22443725.3333333, 2145055.33333333, 3228019, 7720861.33333333, 17482423.6666667, 19404.3333333333, 74258.3333333333, 63101.3333333333, 15900608.3333333, 22536, 46324, 3684505, 74352, 110885, 49688.3333333333, 1142502.66666667, 459295.666666667, 515593, 632950.333333333, 1809386.66666667, 9086594.33333333, 670395, 120428.333333333, 54804.5, 119522.666666667, 79722, 193290.333333333, 19889342.3333333, 73089941.6666667, 24996562.3333333, 24499944, 7979479.66666667, 6676686, 66764209.8333333, 38683439.6666667, 4132338, 724432.666666667, 649076.166666667, 3371683.33333333, 832829.333333333, 1644341.33333333, 3486194.5, 1538105.83333333, 3074869, 1192006.33333333, 63168.6666666667, 423062, 1928893.66666667, 387414.333333333, 329493, 39498.3333333333, 162739.666666667, 232968, 40671, 549538, 232766, 278020.333333333, 507207, 284204.666666667, 93361.6666666667, 142269.166666667, 206419, 274470.333333333, 391611.666666667, 934322.666666667, 2309451.5, 849879.333333333, 1297497.66666667, 270146, 653256, 337449.333333333, 311226.333333333, 118129.333333333, 90460.5, 589519.666666667, 8358047.16666667, 8841519.66666667, 11341209, 12986979, 1359312.66666667, 28657459.6666667, 307334.333333333, 2664756.83333333, 43289065.1666667, 19155.3333333333, 164348.666666667, 170885, 147868.333333333, 256876, 236057, 6543.33333333333, 18754.3333333333, 28007, 702563.666666667, 1061301.66666667, 11569710.6666667, 2199387.66666667, 218830.666666667, 6958715, 16932987.8333333, 8312024.66666667, 26079521.3333333, 4224907.33333333, 1234254, 1090341.33333333, 930356.666666667, 452304, 7367058, 15002957.8333333, 1456780.66666667, 1228075.33333333, 171733, 22825749.3333333, 9780970.66666667, 1456951, 19411749.6666667, 24618285, 3377765, 1552218.33333333, 3756792.66666667, 1444923, 15916711, 1731657.33333333, 4051053.66666667, 17080367.3333333, 3852184.66666667, 2157911.66666667, 2397813.66666667, 3571881.33333333, 1253346.33333333, 25590.3333333333, 19487.8333333333, 94881.6666666667, 178646.666666667, 6991767, 627038.333333333, 11635244.6666667, 796256, 1073336.33333333, 11974505, 4008006, 218245.666666667, 54708, 91206.1666666667, 65261.5, 218894, 5963012.66666667, 13097569.6666667, 23976145, 12989128, 20272967.3333333, 35512577.6666667, 10497302, 3865748.33333333, 241238.333333333, 17553186.6666667, 15558374.6666667, 6822498.83333333, 3580687.33333333, 11749037.6666667, 21737299.6666667, 1846759, 1660423, 7807785.66666667, 13014537.3333333, 14559.3333333333, 67646, 63101.3333333333, 22536, 30531, 3684505, 59372.3333333333, 35372, 63825, 1035790, 453902.333333333, 446332.333333333, 365018.666666667, 1809386.66666667, 9011571, 645838.666666667, 106781.333333333, 51237.3333333333, 65084.3333333333, 74667.6666666667, 63633, 17660313.3333333, 63324580, 24996562.3333333, 7886728.16666667, 2654978, 6584263.33333333, 66764209.8333333, 37522615, 3641334.33333333, 724432.666666667, 300637, 2325724, 784371, 1644341.33333333, 2094716.83333333, 566856, 3074869, 1143588.33333333, 63168.6666666667, 423062, 4546262.33333333, 169604, 110562.666666667, 25090, 140672.333333333, 173531.666666667, 40671, 370075.666666667, 118165.333333333, 187492.666666667, 640630.333333333, 113787.666666667, 93361.6666666667, 142269.166666667, 179003.333333333, 244661.333333333, 151754.666666667, 548598.833333333, 1117559.5, 799533.333333333, 1002640, 244527, 600798, 275388, 311226.333333333, 101945.333333333, 87266.5, 561006.666666667, 6744154.16666667, 4181555.33333333, 11341209, 12986979, 1359312.66666667, 17666858.6666667, 144644.666666667, 2355843.16666667, 41890138.6666667, 133970.333333333, 114110.666666667, 89146, 124021.333333333, 236057, 6543.33333333333, 26169.3333333333, 15237.3333333333, 446034.333333333, 1061301.66666667, 11569710.6666667, 385647.333333333, 163573.333333333, 1064324, 10907174.3333333, 18537309.6666667, 24264809, 797923.333333333, 1232051.33333333, 974526.666666667, 926777.666666667, 376157.5, 7367058, 13866442.1666667, 1214256.33333333, 597599, 130142, 11279567.6666667, 12250508.6666667, 1353513.33333333, 20128782.6666667, 21496176.3333333, 3377765, 1290837, 3380810, 3576946.33333333, 19224674.6666667, 225355.333333333, 4051053.66666667, 8625285.33333333, 2826125, 1371032.66666667, 2099624.33333333, 2609102.66666667, 670967.333333333)))
})

# strictnessfilter.ProteinInference
test_that("strictnessfilter.ProteinInference", {
	expect_that(strictnessfilter.ProteinInference(c(1,2,3,4,NA), strictness="loose"),equals(c(1,2,3,4)))
	expect_that(strictnessfilter.ProteinInference(c(1,2,3,4,NA), strictness="strict"),equals(c(0)))
})

# trypsin.default
test_that("trypsin.default", {
	expect_that(trypsin.default(fasta=system.file("extdata","example.fasta",package="aLFQ"))$PROTEINA,equals(c("PEPTIDEK","ANDPEPTIDEA")))
})
