% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/do_fada_multiv_robust.R
\name{do_fada_multiv_robust}
\alias{do_fada_multiv_robust}
\title{Run the whole archetypoid analysis with the functional multivariate robust Frobenius norm}
\usage{
do_fada_multiv_robust(subset, numArchoid, numRep, huge, prob, compare = FALSE, PM,
                      method = "adjbox")
}
\arguments{
\item{subset}{Data to obtain archetypes. In fadalara this is a subset of the 
entire data frame.}

\item{numArchoid}{Number of archetypes/archetypoids.}

\item{numRep}{For each \code{numArch}, run the archetype algorithm \code{numRep} times.}

\item{huge}{Penalization to solve the convex least squares problem, 
see \code{\link[Anthropometry]{archetypoids}}.}

\item{prob}{Probability with values in [0,1].}

\item{compare}{Boolean argument to compute the non-robust residual sum of squares 
to compare these results with the ones provided by \code{\link{do_fada}}.}

\item{PM}{Penalty matrix obtained with \code{\link[fda]{eval.penalty}}.}

\item{method}{Method to compute the outliers. So far the only option allowed is 
'adjbox' for using adjusted boxplots for skewed distributions. The use of
tolerance intervals might also be explored in the future for the multivariate case.}
}
\value{
A list with the following elements:
\itemize{
\item cases: Final vector of archetypoids.
\item alphas: Alpha coefficients for the final vector of archetypoids.
\item rss: Residual sum of squares corresponding to the final vector of archetypoids.
\item rss_non_rob: If \code{compare=TRUE}, this is the residual sum of squares using
the non-robust Frobenius norm. Otherwise, NULL.
\item resid Vector of residuals.
\item outliers: Outliers.
\item local_rel_imp Matrix with the local (casewise) relative importance 
(in percentage) of each variable for the outlier identification. Only for 
the multivariate case. It is relative to the outlier observation itself. 
The other observations are not considered for computing this importance. 
This procedure works because the functional variables are in the same scale, 
after standardizing. Otherwise, it couldn't be interpreted like that.
\item margi_rel_imp Matrix with the marginal relative importance of each variable 
(in percentage) for the outlier identification. Only for the multivariate case. 
In this case, the other points are considered, since the value of the outlier 
observation is compared with the remaining points.
}
}
\description{
This function executes the entire procedure involved in the functional archetypoid 
analysis. Firstly, the initial vector of archetypoids is obtained using the 
functional archetypal algorithm and finally, the optimal vector of archetypoids is 
returned.
}
\examples{
\dontrun{
library(fda)
?growth
str(growth)
hgtm <- growth$hgtm
hgtf <- growth$hgtf[,1:39]

# Create array:
nvars <- 2
data.array <- array(0, dim = c(dim(hgtm), nvars))
data.array[,,1] <- as.matrix(hgtm)
data.array[,,2] <- as.matrix(hgtf)
rownames(data.array) <- 1:nrow(hgtm)
colnames(data.array) <- colnames(hgtm)
str(data.array)

# Create basis:
nbasis <- 10
basis_fd <- create.bspline.basis(c(1,nrow(hgtm)), nbasis)
PM <- eval.penalty(basis_fd)
# Make fd object:
temp_points <- 1:nrow(hgtm)
temp_fd <- Data2fd(argvals = temp_points, y = data.array, basisobj = basis_fd)

X <- array(0, dim = c(dim(t(temp_fd$coefs[,,1])), nvars))
X[,,1] <- t(temp_fd$coef[,,1]) 
X[,,2] <- t(temp_fd$coef[,,2])

# Standardize the variables:
Xs <- X
Xs[,,1] <- scale(X[,,1])
Xs[,,2] <- scale(X[,,2])

suppressWarnings(RNGversion("3.5.0"))
set.seed(2018)
res_fada <- do_fada_multiv_robust(subset = Xs, numArchoid = 3, numRep = 5, huge = 200, 
                                  prob = 0.75, compare = FALSE, PM = PM, method = "adjbox")
str(res_fada)
res_fada$cases
#[1]  8 24 29
res_fada$rss
#[1] 2.301741
}
                                  
}
\references{
Moliner, J. and Epifanio, I., Robust multivariate and functional archetypal analysis 
with application to financial time series analysis, 2019. 
\emph{Physica A: Statistical Mechanics and its Applications} \bold{519}, 195-208. 
\url{https://doi.org/10.1016/j.physa.2018.12.036}
}
\seealso{
\code{\link{stepArchetypesRawData_funct_multiv_robust}}, 
\code{\link{archetypoids_funct_multiv_robust}}
}
\author{
Guillermo Vinue, Irene Epifanio
}
