% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/create_query_data.R
\name{create_query_data}
\alias{create_query_data}
\title{Creates a queries dataset as input dataset to the \code{dataset_queries} argument in
\code{derive_vars_query()}}
\usage{
create_query_data(queries, version = NULL, get_terms_fun = NULL)
}
\arguments{
\item{queries}{List of queries

A list of \code{query()} objects is expected.}

\item{version}{Dictionary version

The dictionary version used for coding the terms should be specified.
If any of the queries is a basket (SMQ, SDG, ....) or a customized query
including a basket, the parameter needs to be specified.

\emph{Permitted Values}: A character string (the expected format is
company-specific)}

\item{get_terms_fun}{Function which returns the terms

For each query specified for the \code{queries} parameter referring to a basket
(i.e., those where the \code{definition} field is set to a \code{basket_select()}
object or a list which contains at least one \code{basket_select()} object) the
specified function is called to retrieve the terms defining the query.
This function is not provided by admiral as it is company specific, i.e.,
it has to be implemented at company level.

The function must return a dataset with all the terms defining the basket.
The output dataset must contain the following variables.
\itemize{
\item \code{SRCVAR}: the variable to be used for defining a term of the basket,
e.g., \code{AEDECOD}
\item \code{TERMNAME}: the name of the term if the variable \code{SRCVAR} is
referring to is character
\item \code{TERMID} the numeric id of the term if the variable \code{SRCVAR} is
referring to is numeric
\item \code{GRPNAME}: the name of the basket. The values must be the same for
all observations.
}

The function must provide the following parameters
\itemize{
\item \code{basket_select}: A \code{basket_select()} object.
\item \code{version}: The dictionary version. The value specified for the
\code{version} in the \code{create_query_data()} call is passed to this
parameter.
\item \code{keep_id}: If set to \code{TRUE}, the output dataset must contain the
\code{GRPID} variable. The variable must be set to the numeric id of the basket.
\item \code{temp_env}: A temporary environment is passed to this parameter. It can
be used to store data which is used for all baskets in the
\code{create_query_data()} call. For example if SMQs need to be read from a
database all SMQs can be read and stored in the environment when the first
SMQ is handled. For the other SMQs the terms can be retrieved from the
environment instead of accessing the database again.
}}
}
\value{
A dataset to be used as input dataset to the \code{dataset_queries}
argument in \code{derive_vars_query()}
}
\description{
Creates a queries dataset as input dataset to the \code{dataset_queries} argument
in the \code{derive_vars_query()} function as defined in the \href{../articles/queries_dataset.html}{Queries Dataset Documentation}.
}
\details{
For each \code{query()} object listed in the \code{queries} argument, the terms belonging
to the query (\code{SRCVAR}, \code{TERMNAME}, \code{TERMID}) are determined with respect
to the \code{definition} field of the query: if the definition field of the
\code{query()} object is
\itemize{
\item a \code{basket_select()} object, the terms are read from the basket
database by calling the function specified for the \code{get_terms_fun} parameter.
\item a data frame, the terms stored in the data frame are used.
\item a list of data frames and \code{basket_select()} objects, all terms from
the data frames and all terms read from the basket database referenced by the
\code{basket_select()} objects are collated.
}

The following variables (as described in \href{../articles/queries_dataset.html}{Queries Dataset Documentation}) are created:
\itemize{
\item \code{PREFIX}: Prefix of the variables to be created by
\code{derive_vars_query()} as specified by the \code{prefix} element.
\item \code{GRPNAME}: Name of the query as specified by the \code{name} element.
\item \code{GRPID}: Id of the query as specified by the \code{id} element. If the \code{id}
element is not specified for a query, the variable is set to \code{NA}. If the
\code{id} element is not specified for any query, the variable is not created.
\item \code{SCOPE}: scope of the query as specified by the \code{scope} element of
the \code{basket_select()} object. For queries not defined by a \code{basket_select()}
object, the variable is set to \code{NA}. If none of the queries is defined by a
\code{basket_select()} object, the variable is not created.
\item \code{SCOPEN}: numeric scope of the query. It is set to \code{1} if the
scope is broad. Otherwise it is set to \code{2}. If the \code{add_scope_num} element
equals \code{FALSE}, the variable is set to \code{NA}. If the \code{add_scope_num} element
equals \code{FALSE} for all baskets or none of the queries is an basket , the variable
is not created.
\item \code{SRCVAR}: Name of the variable used to identify the terms.
\item \code{TERMNAME}: Value of the term variable if it is a character variable.
\item \code{TERMID}: Value of the term variable if it is a numeric variable.
\item \code{VERSION}: Set to the value of the \code{version} argument. If it is not
specified, the variable is not created.
}
}
\examples{
library(tibble)
library(dplyr, warn.conflicts = FALSE)
library(admiral.test)
library(admiral)

# creating a query dataset for a customized query
cqterms <- tribble(
  ~TERMNAME,                     ~TERMID,
  "APPLICATION SITE ERYTHEMA", 10003041L,
  "APPLICATION SITE PRURITUS", 10003053L
) \%>\%
  mutate(SRCVAR = "AEDECOD")

cq <- query(
  prefix = "CQ01",
  name = "Application Site Issues",
  definition = cqterms
)

create_query_data(queries = list(cq))

# create a query dataset for SMQs
pregsmq <- query(
  prefix = "SMQ02",
  id = auto,
  definition = basket_select(
    name = "Pregnancy and neonatal topics (SMQ)",
    scope = "NARROW",
    type = "smq"
  )
)

bilismq <- query(
  prefix = "SMQ04",
  definition = basket_select(
    id = 20000121L,
    scope = "BROAD",
    type = "smq"
  )
)

# The get_terms function from admiral.test is used for this example.
# In a real application a company-specific function must be used.
create_query_data(
  queries = list(pregsmq, bilismq),
  get_terms_fun = admiral.test:::get_terms,
  version = "20.1"
)

# create a query dataset for SDGs
sdg <- query(
  prefix = "SDG01",
  id = auto,
  definition = basket_select(
    name = "5-aminosalicylates for ulcerative colitis",
    scope = NA_character_,
    type = "sdg"
  )
)

# The get_terms function from admiral.test is used for this example.
# In a real application a company-specific function must be used.
create_query_data(
  queries = list(sdg),
  get_terms_fun = admiral.test:::get_terms,
  version = "2019-09"
)

# creating a query dataset for a customized query including SMQs
# The get_terms function from admiral.test is used for this example.
# In a real application a company-specific function must be used.
create_query_data(
  queries = list(
    query(
      prefix = "CQ03",
      name = "Special issues of interest",
      definition = list(
        basket_select(
          name = "Pregnancy and neonatal topics (SMQ)",
          scope = "NARROW",
          type = "smq"
        ),
        cqterms
      )
    )
  ),
  get_terms_fun = admiral.test:::get_terms,
  version = "20.1"
)
}
\seealso{
\code{\link[=derive_vars_query]{derive_vars_query()}}, \code{\link[=query]{query()}}, \code{\link[=basket_select]{basket_select()}}, \href{../articles/queries_dataset.html}{Queries Dataset Documentation}

Creating auxiliary datasets: 
\code{\link{consolidate_metadata}()},
\code{\link{create_period_dataset}()},
\code{\link{create_single_dose_dataset}()}
}
\concept{create_aux}
\keyword{create_aux}
