% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/derive_var_extreme_date.R
\name{derive_var_extreme_dtm}
\alias{derive_var_extreme_dtm}
\title{Derive First or Last Datetime from Multiple Sources}
\usage{
derive_var_extreme_dtm(
  dataset,
  new_var,
  ...,
  source_datasets,
  mode,
  subject_keys = vars(STUDYID, USUBJID)
)
}
\arguments{
\item{dataset}{Input dataset

The variables specified by \code{subject_keys} are required.}

\item{new_var}{Name of variable to create}

\item{...}{Source(s) of dates. One or more \code{date_source()} objects are
expected.}

\item{source_datasets}{A named \code{list} containing datasets in which to search
for the first or last date}

\item{mode}{Selection mode (first or last)

If \code{"first"} is specified, the first date for each subject is selected. If
\code{"last"} is specified, the last date for each subject is selected.

Permitted Values:  \code{"first"}, \code{"last"}}

\item{subject_keys}{Variables to uniquely identify a subject

A list of quosures where the expressions are symbols as returned by
\code{vars()} is expected.}
}
\value{
The input dataset with the new variable added.
}
\description{
Add the first or last datetime from multiple sources to the dataset, e.g.,
the last known alive datetime (\code{LSTALVDTM}).
}
\details{
The following steps are performed to create the output dataset:
\enumerate{
\item For each source dataset the observations as specified by the \code{filter}
element are selected. Then for each patient the first or last observation
(with respect to \code{date} and \code{mode}) is selected.
\item The new variable is set to the variable specified by the \code{date} element.
If the date variable is a date variable, the time is imputed as specified
by the \code{time_imputation} element. If the source variable is a character
variable, it is converted to a datetime. If the date is incomplete, it is
imputed as specified by the \code{date_imputation} and \code{time_imputation}
element.
\item The variables specified by the \code{traceability_vars} element are added.
\item The selected observations of all source datasets are combined into a
single dataset.
\item For each patient the first or last observation (with respect to the new
variable and \code{mode}) from the single dataset is selected and the new
variable is merged to the input dataset.
}
}
\examples{
library(dplyr, warn.conflicts = FALSE)
library(admiral.test)
data("admiral_dm")
data("admiral_ae")
data("admiral_lb")
data("admiral_adsl")

# derive last known alive datetime (LSTALVDTM)
ae_start <- date_source(
  dataset_name = "ae",
  date = AESTDTC,
  date_imputation = "first",
  time_imputation = "first"
)
ae_end <- date_source(
  dataset_name = "ae",
  date = AEENDTC,
  date_imputation = "first",
  time_imputation = "first"
)
lb_date <- date_source(
  dataset_name = "lb",
  date = LBDTC,
  filter = nchar(LBDTC) >= 10,
  time_imputation = "first"
)
adsl_date <- date_source(dataset_name = "adsl", date = TRTEDTM)

admiral_dm \%>\%
  derive_var_extreme_dtm(
    new_var = LSTALVDTM,
    ae_start, ae_end, lb_date, adsl_date,
    source_datasets = list(
      adsl = admiral_adsl,
      ae = admiral_ae, lb = admiral_lb
    ),
    mode = "last"
  ) \%>\%
  select(USUBJID, LSTALVDTM)

# derive last alive datetime and traceability variables
ae_start <- date_source(
  dataset_name = "ae",
  date = AESTDTC,
  date_imputation = "first",
  time_imputation = "first",
  traceability_vars = vars(
    LALVDOM = "AE",
    LALVSEQ = AESEQ,
    LALVVAR = "AESTDTC"
  )
)

ae_end <- date_source(
  dataset_name = "ae",
  date = AEENDTC,
  date_imputation = "first",
  time_imputation = "first",
  traceability_vars = vars(
    LALVDOM = "AE",
    LALVSEQ = AESEQ,
    LALVVAR = "AEENDTC"
  )
)
lb_date <- date_source(
  dataset_name = "lb",
  date = LBDTC,
  filter = nchar(LBDTC) >= 10,
  time_imputation = "first",
  traceability_vars = vars(
    LALVDOM = "LB",
    LALVSEQ = LBSEQ,
    LALVVAR = "LBDTC"
  )
)

adsl_date <- date_source(
  dataset_name = "adsl",
  date = TRTEDTM,
  traceability_vars = vars(
    LALVDOM = "ADSL",
    LALVSEQ = NA_integer_,
    LALVVAR = "TRTEDTM"
  )
)

admiral_dm \%>\%
  derive_var_extreme_dtm(
    new_var = LSTALVDTM,
    ae_start, ae_end, lb_date, adsl_date,
    source_datasets = list(
      adsl = admiral_adsl,
      ae = admiral_ae, lb = admiral_lb
    ),
    mode = "last"
  ) \%>\%
  select(USUBJID, LSTALVDTM, LALVDOM, LALVSEQ, LALVVAR)
}
\seealso{
\code{\link[=date_source]{date_source()}}, \code{\link[=derive_var_extreme_dt]{derive_var_extreme_dt()}},
\code{\link[=derive_vars_merged_dt]{derive_vars_merged_dt()}}, \code{\link[=derive_vars_merged_dtm]{derive_vars_merged_dtm()}},
\code{\link[=derive_vars_merged]{derive_vars_merged()}}
}
\author{
Stefan Bundfuss, Thomas Neitmann
}
\keyword{adsl}
\keyword{derivation}
