% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/clustering.R
\name{adproclus_low_dim}
\alias{adproclus_low_dim}
\title{Low dimensional ADPROCLUS}
\usage{
adproclus_low_dim(
  data,
  nclusters,
  ncomponents,
  start_allocation = NULL,
  nrandomstart = 3,
  nsemirandomstart = 3,
  save_all_starts = FALSE,
  seed = NULL
)
}
\arguments{
\item{data}{Object-by-variable data matrix of class \code{matrix} or
\code{data.frame}.}

\item{nclusters}{Number of clusters to be used. Must be a positive integer.}

\item{ncomponents}{Number of components (dimensions) to which the profiles
should be restricted. Must be a positive integer.}

\item{start_allocation}{Optional matrix of binary values as starting
allocation for first run. Default is \code{NULL}.}

\item{nrandomstart}{Number of random starts (see \code{\link{get_random}}).
Can be zero. Increase for better results, though longer computation time.
Some research finds 500 starts to be a useful reference.}

\item{nsemirandomstart}{Number of semi-random starts
(see \code{\link{get_semirandom}})). Can be zero.
Increase for better results, though longer computation time.
Some research finds 500 starts to be a useful reference.}

\item{save_all_starts}{logical. If \code{TRUE}, the results of all algorithm
starts are returned. By default, only the best solution is retained.}

\item{seed}{Integer. Seed for the random number generator.
Default: NULL, meaning no reproducibility}
}
\value{
\code{adproclus_low_dim()} returns a list with the following
  components, which describe the best model (from the multiple starts):
  \describe{
  \item{\code{model}}{matrix. The obtained overlapping clustering model
  \eqn{M} of the same size as \code{data}.}
  \item{\code{model_lowdim}}{matrix. The obtained low dimensional clustering
  model \eqn{AC} of size \eqn{I \times S}}
  \item{\code{A}}{matrix. The membership matrix \eqn{A} of the
  clustering model. Clusters are sorted by size.}
  \item{\code{P}}{matrix. The profile matrix \eqn{P} of the
  clustering model.}
  \item{\code{c}}{matrix. The profile values in terms of the low dimensional
  components.}
  \item{\code{B}}{Variables-by-components matrix.
  Base vectors connecting low dimensional components with original variables.
  matrix. Warning: for computing
  \eqn{P} use \eqn{B'}.}
  \item{\code{sse}}{numeric. The
  residual sum of squares of the clustering model, which is minimized by the
  ALS algorithm.}
  \item{\code{totvar}}{numeric. The total sum of squares
  of \code{data}.}
  \item{\code{explvar}}{numeric. The proportion of variance
  in \code{data} that is accounted for by the clustering model.}
  \item{\code{iterations}}{numeric. The number of iterations of the
  algorithm.}
  \item{\code{timer}}{numeric. The amount of time (in seconds) the complete
  algorithm ran for.}
  \item{\code{timer_one_run}}{numeric. The amount of time (in seconds) the
  relevant single start ran for.}
  \item{\code{initial_start}}{list. A list containing the initial
  membership matrix, as well as the type of start that was used
  to obtain the clustering solution. (as returned by \code{\link{get_random}}
  or \code{\link{get_semirandom}})}
  \item{\code{runs}}{list. Each element represents one model obtained
  from one of the multiple starts.
  Each element contains all of the above information.}
  \item{\code{parameters}}{list. Containing the parameters used for the
  model.}}
}
\description{
Perform \strong{low dimensional} additive profile clustering (ADPROCLUS) on
object by variable data. Use case: data to cluster consists of a large set of
variables, where it can be useful to interpret the cluster profiles in terms
of a smaller set of components that represent the original variables well.
}
\details{
In this function, an extension by Depril et al. (2012) of
Mirkins (1987, 1990) additive profile clustering method is used to obtain a
low dimensional overlapping clustering model of the object by variable data
provided by \code{data}.
More precisely, the low dimensional ADPROCLUS model approximates an
\eqn{I \times J} object by variable data matrix \eqn{X} by an
\eqn{I \times J} model matrix \eqn{M}. For \eqn{K} overlapping
clusters, \eqn{M} can be decomposed into an \eqn{I \times K}
binary cluster membership matrix \eqn{A} and a \eqn{K \times J}
real-valued cluster profile matrix \eqn{P} s.t. \eqn{M = AP.}
With the simultaneous dimension reduction, \eqn{P} is restricted
to be of reduced rank \eqn{S < min(K,J)}, such that it can be decomposed
into \eqn{P = CB',} with \eqn{C} a \eqn{K \times S} matrix and
\eqn{B} a \eqn{J \times S} matrix. Now, a row in
\eqn{C} represents the profile values associated with the
respective cluster in terms of the \eqn{S} components, while
the entries of \eqn{B} can be used to interpret the components
in terms of the complete set of variables. In particular, the aim of an
ADPROCLUS analysis is therefore, given a number of clusters \eqn{K} and a
number of dimensions \eqn{S}, to estimate a model matrix \eqn{M}
that reconstructs data matrix
\eqn{X} as close as possible in a least squares sense and
simultaneously reduce the dimensions of the data.
For a detailed illustration of the low dimensional ADPROCLUS model and
associated loss function, see Depril et al. (2012).

\strong{Warning:} Computation time increases exponentially with increasing
number of clusters, \eqn{K}. We recommend to determine the computation time
of a single start for each specific dataset and \eqn{K} before increasing the
number of starts.
}
\examples{
# Loading a test dataset into the global environment
x <- stackloss

# Low dimensional clustering with K = 3 clusters
# where the resulting profiles can be characterized in S = 1 dimensions
clust <- adproclus_low_dim(x, 3, ncomponents = 1)

}
\references{
Depril, D., Van Mechelen, I., & Wilderjans, T. F.
  (2012). Lowdimensional additive overlapping clustering.
  \emph{Journal of classification, 29,} 297-320.
}
\seealso{
\describe{
  \item{\code{\link{adproclus}}}{for full dimensional ADPROCLUS}
  \item{\code{\link{get_random}}}{for generating random starts}
  \item{\code{\link{get_semirandom}}}{for generating semi-random starts}
  \item{\code{\link{get_rational}}}{for generating rational starts}
}
}
