% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Sequence.R
\name{collapseDuplicates}
\alias{collapseDuplicates}
\title{Remove duplicate DNA sequences and combine annotations}
\usage{
collapseDuplicates(data, id = "SEQUENCE_ID", seq = "SEQUENCE_IMGT",
  text_fields = NULL, num_fields = NULL, seq_fields = NULL,
  add_count = FALSE, ignore = c("N", "-", ".", "?"), sep = ",",
  verbose = FALSE)
}
\arguments{
\item{data}{data.frame containing Change-O columns. The data.frame 
must contain, at a minimum, a unique identifier column 
and a column containg a character vector of DNA sequences.}

\item{id}{name of the column containing sequence identifiers.}

\item{seq}{name of the column containing DNA sequences.}

\item{text_fields}{character vector of textual columns to collapse. The textual 
annotations of duplicate sequences will be merged into a single 
string with each unique value alphabetized and delimited by 
\code{sep}.}

\item{num_fields}{vector of numeric columns to collapse. The numeric annotations
of duplicate sequences will be summed.}

\item{seq_fields}{vector of nucletoide sequence columns to collapse. The sequence 
with the fewest numer of non-informative characters will be retained. 
Where a non-informative character is one of \code{c("N", "-", ".", "?")}. 
Note, this is distinct from the \code{seq} parameter which is used to 
determine duplicates.}

\item{add_count}{if \code{TRUE} add the column \code{COLLAPSE_COUNT} that 
indicates the number of sequences that were collapsed to build 
each unique entry.}

\item{ignore}{vector of characters to ignore when testing for equality.}

\item{sep}{character to use for delimiting collapsed annotations in the 
\code{text_fields} columns. Defines both the input and output 
delimiter.}

\item{verbose}{if \code{TRUE} report the number input, discarded and output 
sequences; if \code{FALSE} process sequences silently.}
}
\value{
A modified \code{data} data.frame with duplicate sequences removed and 
          annotation fields collapsed.
}
\description{
\code{collapseDuplicates} identifies duplicate DNA sequences, allowing for ambiguous 
characters, removes the duplicate entries, and combines any associated annotations.
}
\details{
\code{collapseDuplicates} identifies duplicate sequences in the \code{seq} column by
testing for character identity, with consideration of IUPAC ambiguous nucleotide codes. 
A cluster of sequences are considered duplicates if they are all equivalent, and no 
member of the cluster is equivalent to a sequence in a different cluster. 

Textual annotations, specified by \code{text_fields}, are collapsed by taking the unique
set of values within in each duplicate cluster and delimiting those values by \code{sep}.
Numeric annotations, specified by \code{num_fields}, are collapsed by summing all values 
in the duplicate cluster. Sequence annotations, specified by \code{seq_fields}, are 
collapsed by retaining the first sequence with the fewest number of N characters.

Columns that are not specified in either \code{text_fields}, \code{num_fields}, or 
\code{seq_fields} will be retained, but the value will be chosen from a random entry 
amongst all sequences in a cluster of duplicates.

An ambiguous sequence is one that can be assigned to two different clusters, wherein
the ambiguous sequence is equivalent to two sequences which are themselves 
non-equivalent. Ambiguous sequences arise due to ambiguous characters at positions that
vary across sequences, and are discarded along with their annotations. Thus, ambiguous
sequences are removed as duplicates of some sequence, but do not create a potential
false-positive annotation merger. Ambiguous sequences are not included in the 
\code{COLLAPSE_COUNT} annotation that is added when \code{add_count=TRUE}.
}
\examples{
# Example Change-O data.frame
df <- data.frame(SEQUENCE_ID=LETTERS[1:4],
                 SEQUENCE_IMGT=c("CCCCTGGG", "CCCCTGGN", "NAACTGGN", "NNNCTGNN"),
                 TYPE=c("IgM", "IgG", "IgG", "IgA"),
                 SAMPLE=c("S1", "S1", "S2", "S2"),
                 COUNT=1:4,
                 stringsAsFactors=FALSE)

# Annotations are not parsed if neither text_fields nor num_fields is specified
# The retained sequence annotations will be random
collapseDuplicates(df, verbose=TRUE)

# Unique text_fields annotations are combined into a single string with ","
# num_fields annotations are summed
# Ambiguous duplicates are discarded
collapseDuplicates(df, text_fields=c("TYPE", "SAMPLE"), num_fields="COUNT", 
                   verbose=TRUE)

# Use alternate delimiter for collapsing textual annotations
collapseDuplicates(df, text_fields=c("TYPE", "SAMPLE"), num_fields="COUNT", 
                   sep="/", verbose=TRUE)

# Add count of duplicates
collapseDuplicates(df, text_fields=c("TYPE", "SAMPLE"), num_fields="COUNT", 
                   add_count=TRUE, verbose=TRUE)

# Masking ragged ends may impact duplicate removal
df$SEQUENCE_IMGT <- maskSeqEnds(df$SEQUENCE_IMGT)
collapseDuplicates(df, text_fields=c("TYPE", "SAMPLE"), num_fields="COUNT", 
                   add_count=TRUE, verbose=TRUE)

}
\seealso{
Equality is tested with \code{\link{testSeqEqual}}.  For IUPAC ambiguous 
          character codes see \code{\link{IUPAC_DNA}}.

Other sequence manipulation functions: \code{\link{maskSeqEnds}},
  \code{\link{maskSeqGaps}}
}

