#' AMMI Stability Index
#'
#' \code{ASI.AMMI} computes the AMMI Stability Index (ASI)
#' \insertCite{jambhulkar_ammi_2014,jambhulkar_genotype_2015,jambhulkar_stability_2017}{ammistability}
#' considering the first two interaction principal components (IPCs) in the AMMI
#' model. Using ASI, the Simultaneous Selection Index for Yield and Stability
#' (SSI) is also calculated according to the argument \code{ssi.method}.
#' \loadmathjax
#'
#' The AMMI Stability Index (\mjseqn{ASI})
#' \insertCite{jambhulkar_ammi_2014,jambhulkar_genotype_2015,jambhulkar_stability_2017}{ammistability}
#' is computed as follows:
#'
#' \mjsdeqn{ASI = \sqrt{\left [ PC_{1}^{2} \times \theta_{1}^{2} \right ]+\left
#' [ PC_{2}^{2} \times \theta_{2}^{2} \right ]}}
#'
#' Where, \mjseqn{PC_{1}} and \mjseqn{PC_{2}} are the scores of 1st and 2nd IPCs
#' respectively; and \mjseqn{\theta_{1}} and \mjseqn{\theta_{2}} are percentage
#' sum of squares explained by the 1st and 2nd principal component interaction
#' effect respectively.
#'
#' @param model The AMMI model (An object of class \code{AMMI} generated by
#'   \code{\link[agricolae]{AMMI}}).
#' @param ssi.method The method for the computation of simultaneous selection
#'   index. Either \code{"farshadfar"} or \code{"rao"} (See
#'   \code{\link[ammistability]{SSI}}).
#' @param a The ratio of the weights given to the stability components for
#'   computation of SSI when \code{method = "rao"} (See
#'   \code{\link[ammistability]{SSI}}).
#'
#' @return A data frame with the following columns:  \item{ASI}{The ASI values.}
#'   \item{SSI}{The computed values of simultaneous selection index for yield
#'   and stability.} \item{rASI}{The ranks of ASI values.} \item{rY}{The ranks
#'   of the mean yield of genotypes.} \item{means}{The mean yield of the
#'   genotypes.}
#'
#'   The names of the genotypes are indicated as the row names of the data
#'   frame.
#'
#' @importFrom methods is
#' @importFrom stats aggregate
#' @importFrom agricolae AMMI
#' @export
#'
#' @references
#'
#' \insertAllCited{}
#'
#' @seealso \code{\link[agricolae]{AMMI}}, \code{\link[ammistability]{SSI}}
#'
#' @examples
#' library(agricolae)
#' data(plrv)
#'
#' # AMMI model
#' model <- with(plrv, AMMI(Locality, Genotype, Rep, Yield, console = FALSE))
#'
#' # ANOVA
#' model$ANOVA
#'
#' # IPC F test
#' model$analysis
#'
#' # Mean yield and IPC scores
#' model$biplot
#'
#' # G*E matrix (deviations from mean)
#' array(model$genXenv, dim(model$genXenv), dimnames(model$genXenv))
#'
#' # With default ssi.method (farshadfar)
#' ASI.AMMI(model)
#'
#' # With  ssi.method = "rao"
#' ASI.AMMI(model, ssi.method = "rao")
#'
#' # Changing the ratio of weights for Rao's SSI
#' ASI.AMMI(model, ssi.method = "rao", a = 0.43)
#'
ASI.AMMI <- function(model, ssi.method = c("farshadfar", "rao"), a = 1) {

  # Check model class
  if (!is(model, "AMMI")) {
    stop('"model" is not of class "AMMI"')
  }

  ssi.method <- match.arg(ssi.method)

  # Fetch response (Yield)
  yresp <- setdiff(colnames(model$means), c("ENV", "GEN", "RESIDUAL"))

  A <- model$biplot[, 1:4]
  A <- A[A[, 1] == "GEN", -c(1, 2)]

  th1 <- model$analysis["PC1", ]$percent / 100
  th2 <- model$analysis["PC2", ]$percent / 100

  ASI <- sqrt(((A[, "PC1"]^2) * (th1^2)) + ((A[, "PC2"]^2) * (th2^2)))

  B <- model$means
  W <- aggregate(B[, yresp], by = list(model$means$GEN), FUN = mean, na.rm = TRUE)
  SSI_ASI <- SSI(y = W$x, sp = ASI, gen = W$Group.1,
                 method = ssi.method, a = a)
  ranking <- SSI_ASI
  colnames(ranking) <- c("ASI", "SSI", "rASI", "rY", "means")

  return(ranking)

}
