% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/shrink.heur.R
\name{shrink.heur}
\alias{shrink.heur}
\title{Shrinkage After Estimation Using Heuristic Formulae}
\usage{
shrink.heur(dataset, model, DF, int = TRUE, int.adj = FALSE)
}
\arguments{
\item{dataset}{a dataset for regression analysis. Data should be in the form
of a matrix, with the outcome variable as the final column. Application of the
\code{\link{datashape}} function beforehand is recommended, especially if
 categorical predictors are present. For regression with an intercept included
 a column vector of 1s should be included before the dataset (see examples)}

\item{model}{type of regression model. Either "linear" or "logistic".}

\item{DF}{the number of degrees of freedom or number of predictors in the model.
If DF is missing the value will be automatically estimated. This
may be inaccurate for complex models with non-linear terms.}

\item{int}{logical. If TRUE the model will include a regression intercept.}

\item{int.adj}{logical. If TRUE the regression intercept will be re-estimated
after shrinkage of the regression coefficients. If FALSE the regression
intercept will be re-estimated as described by Harrell 2001.}
}
\value{
\code{shrink.heur} returns a list containing the following:

\item{raw.coeff}{the raw regression model coefficients, pre-shrinkage.}

\item{shrunk.coeff}{the shrunken regression model coefficients}

\item{lambda}{the heuristic shrinkage factor}

\item{DF}{the number of degrees of freedom or number of predictors in the model}
}
\description{
Shrink regression coefficients using heuristic formulae, first described
by Van Houwelingen and Le Cessie (Stat. Med., 1990)
}
\details{
This function can be used to estimate shrunken regression coefficients based on
heuristic formulae (see References). A linear or logistic regression model with
an intercept is fitted to the data, and a shrinkage factor is estimated. The
shrinkage factor is then applied to the regression coefficients. If
\code{int.adj == FALSE} the intercept value is estimated as described in
Harrell 2001.If \code{int.adj == TRUE} the intercept value will be re-estimated
by refitting the model with the shrunken coefficients.

The heuristic formula work by applying the
number of model degrees of freedom (or the number of predictors) as a penalty,
and so as the model becomes more complex, the necessary shrinkage increases, and
the shrinkage factor becomes closer to zero.
}
\note{
Warning: In poorly fitting models that includea large number of predictors
      the number of degrees of freedom may approch or exceed the model chi square.
      In such cases the shrinkage factor will be very small or even negative,
      and a different model building strategy is recommended.
}
\examples{
## Example 1: Linear regression using the iris dataset
## shrinkage using a heuristic formula
data(iris)
iris.data <- as.matrix(iris[, 1:4])
iris.data <- cbind(1, iris.data)
set.seed(123)
shrink.heur(dataset = iris.data, model = "linear")

## Example 2: logistic regression using a subset of the mtcars data
## shrinkage using a heuristic formula
data(mtcars)
mtc.data <- cbind(1,datashape(mtcars, y = 8, x = c(1,6,9)))
head(mtc.data)
set.seed(321)
shrink.heur(dataset = mtc.data, model = "logistic", DF = 3,
int = TRUE, int.adj = TRUE)

}
\references{
Harrell, F. E. \emph{"Regression modeling strategies: with applications
             to linear models, logistic regression, and survival analysis."} \emph{Springer}, (2001).

Harrell, F. E., Kerry L. Lee, and Daniel B. Mark. \emph{"Tutorial in
           biostatistics multivariable prognostic models: issues in developing models,
           evaluating assumptions and adequacy, and measuring and reducing errors."}
           \emph{Statistics in medicine} (1996) 15:361-387.

Steyerberg, E. \emph{"Clinical Prediction Models"} \emph{Springer} (2009)

Van Houwelingen, J. C. and Le Cessie, S., \emph{"Predictive value of statistical models."}
            \emph{Statistics in medicine} (1990) 9:1303:1325.
}

