% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/getSurfaceHorizonDepth.R
\name{getSurfaceHorizonDepth}
\alias{getSurfaceHorizonDepth}
\alias{getMineralSoilSurfaceDepth}
\alias{getPlowLayerDepth}
\title{Determine thickness of horizons (continuous from surface) matching a pattern}
\usage{
getSurfaceHorizonDepth(
  p,
  pattern,
  hzdesgn = hzdesgnname(p, required = TRUE),
  simplify = TRUE
)

getMineralSoilSurfaceDepth(
  p,
  hzdesgn = hzdesgnname(p, required = TRUE),
  pattern = "O",
  simplify = TRUE
)

getPlowLayerDepth(
  p,
  hzdesgn = hzdesgnname(p, required = TRUE),
  pattern = "^Ap[^b]*",
  simplify = TRUE
)
}
\arguments{
\item{p}{a SoilProfileCollection}

\item{pattern}{a regular expression pattern to match for all horizons to be considered part of the "surface".}

\item{hzdesgn}{column name containing horizon designation. Default: \code{hzdesgnname(p, required = TRUE)}.}

\item{simplify}{logical. Return single profile results as vector (default: \code{TRUE}) or \code{data.frame} (\code{FALSE})}
}
\value{
a numeric value corresponding to the bottom depth of the last horizon matching 'pattern' that is contiguous with other matching horizons up to the soil surface. If \code{length(p) > 1} then a \emph{data.frame} containing profile ID, horizon ID, top or bottom depths, horizon designation and pattern.
}
\description{
Find the thickness of horizon designations, or any other character patterns, that are continuous from the soil surface (depth = 0 or shallowest depth in profile).
}
\details{
The horizon designation to match is specified with the regular expression pattern 'pattern'. All horizons matching that pattern, that are continuous from the soil surface, count towards the depth / thickness value that is ultimately returned. For instance: horizon designations: A1-A2-A3-C-Ab , would return A3 bottom depth given \code{pattern = "^A[1-9]*$"}.

\code{getSurfaceHorizonDepth} is used by \code{getPlowLayerDepth} for matching Ap horizons; and, it is used by \code{getMineralSoilSurfaceDepth} to find the thickness of O horizons in lieu of lab data.
}
\examples{
library(aqp)
data(sp1, package = 'aqp')
depths(sp1) <- id ~ top + bottom
site(sp1) <- ~ group

p <- sp1[1]
q <- sp1[2]

# look at horizon designations in p and q
p$name
q$name

# thickness of all surface horizons containing A
getSurfaceHorizonDepth(p, pattern = 'A', hzdesgn = 'name')

# thickness of all surface horizons that start with A
getSurfaceHorizonDepth(p, pattern = '^A', hzdesgn = 'name')

# thickness of all surface horizons that start with A, and the A is not followed by B
getSurfaceHorizonDepth(p, pattern = '^A[^B]*', hzdesgn = 'name')

# thickness of all surface horizons that start with A
#  followed by a number from _2_ to 9 (returns ZERO)
getSurfaceHorizonDepth(p, pattern = '^A[2-9]*', hzdesgn = 'name')

# getPlowLayerDepth matches first two horizons in fake Ap horizon data with "buried Ap"
p$aphorizons <- c("Ap1","Ap2","AB", rep('C', nrow(p) - 4), "Apb")
getPlowLayerDepth(p, hzdesgn = 'aphorizons')

# getMineralSoilSurfaceDepthmatches first 3 horizons in fake O horizon data
p$ohorizons <- c("Oi1","Oi2","Oe", rep('C', nrow(p) - 4), "2C")
getMineralSoilSurfaceDepth(p, hzdesgn='ohorizons')

# matches first Oi horizon with original horizon designations of pedon 2
getMineralSoilSurfaceDepth(q, hzdesgn='name')

}
\author{
Andrew G. Brown
}
