library("arules")
library("testthat")

context("interestMeasures")

options(digits = 2)

data <- list(c("A", "B"),
  c("A", "B", "C", "G"),
  c("C", "D"),
  c("C", "D"),
  c("E", "F"))

trans <- transactions(data)


##################################################################
# Test the original example from
# Edward R. Omiecinski. Alternative interest measures for mining
# associations in databases. IEEE Transactions on Knowledge and
# Data Engineering, 15(1):57-69, Jan/Feb 2003.

# complains about low support
fsets <-
    eclat(
      trans,
      parameter = list(supp = 0),
      control = list(verb = FALSE)
    )

# add all-confidence
quality(fsets)$allConfidence <-
  interestMeasure(fsets, measure = "allConfidence", trans)
#inspect(fsets[order(size(fsets))])

# check
ac <-
  c(
    1.00,
    0.67,
    0.33,
    0.33,
    0.50,
    0.33,
    0.33,
    0.50,
    0.50,
    0.33,
    0.33,
    1.00,
    0.33,
    0.60,
    0.40,
    0.40,
    0.20,
    0.40,
    0.20,
    0.20
  )
expect_equal(round(quality(fsets)$allConfidence, 2), ac)

###################################################################
## test all measures for itemsets
m1 <- interestMeasure(fsets, transactions = trans)

## now recalculate the measures using the transactions
m2 <- interestMeasure(fsets, transactions = trans, reuse = FALSE)
expect_equal(m1, m2)

## check if single itemset returns a single row
m3 <- interestMeasure(fsets[1], transactions = trans)
expect_equal(nrow(m3), 1L)

## check for empty itemset
m4 <- interestMeasure(fsets[0], transactions = trans)
expect_equal(nrow(m4), 0L)

m5 <- interestMeasure(fsets[0], transactions = trans, reuse = FALSE)
expect_equal(nrow(m5), 0L)

###################################################################
# test measures for rules

rules <- apriori(trans,
  parameter = list(supp = 0.01, conf = 0.5),
  control = list(verb = FALSE))

## calculate all measures (just to see if one creates an error)
m1 <- interestMeasure(rules, transactions = trans)

## ruleset without quality data.frame
rules2 <- rules
quality(rules2) <- quality(rules)[, 0]
mr2 <- interestMeasure(rules2, transactions = trans)

## check if single rule returns a single row
m2 <- interestMeasure(rules[1], transactions = trans)
expect_equal(nrow(m2), 1L)

## coverage
expect_equal(coverage(rules), support(lhs(rules), trans = trans))
expect_equal(coverage(rules, trans = trans, reuse = FALSE),
  support(lhs(rules), trans = trans))

## check for empty ruleset
m4 <- interestMeasure(rules[0], transactions = trans)
expect_equal(nrow(m4), 0L)

m5 <- interestMeasure(rules[0], transactions = trans, reuse = TRUE)
expect_equal(nrow(m5), 0L)

## is.redundant (this test does not help much)!
context("is.redundant")

red <- is.redundant(rules)
imp <- interestMeasure(rules, measure = "improvement")
expect_equal(red, imp <= 0)

#inspect(rules[!red])
#inspect(rules[red])


context("support")

s_tid <- support(rules, trans, control = list(method = "tidlist"))
s_ptree <- support(rules, trans, control = list(method = "ptree"))
expect_equal(s_tid, s_ptree)
expect_equal(s_tid, quality(rules)$support)

## FIXME: test others

context("test with previous version")
data("Adult")
## Mine association rules.
rules <- apriori(Adult,
  parameter = list(
    supp = 0.5,
    conf = 0.9,
    target = "rules"
  ), control = list(verb = FALSE))

m_r <- interestMeasure(rules, transactions = Adult, reuse = TRUE)
m <- interestMeasure(rules, transactions = Adult, reuse = FALSE)
expect_equal(m_r, m)

#dput(round(m_r, 3))
m_previous <- structure(
  list(
    support = c(
      0.917,
      0.953,
      0.544,
      0.561,
      0.605,
      0.633,
      0.641,
      0.664,
      0.788,
      0.782,
      0.814,
      0.822,
      0.855,
      0.871,
      0.871,
      0.519,
      0.519,
      0.542,
      0.542,
      0.531,
      0.556,
      0.541,
      0.566,
      0.57,
      0.543,
      0.547,
      0.567,
      0.569,
      0.59,
      0.611,
      0.611,
      0.719,
      0.719,
      0.749,
      0.749,
      0.74,
      0.74,
      0.779,
      0.779,
      0.511,
      0.511,
      0.511,
      0.508,
      0.518,
      0.518,
      0.52,
      0.52,
      0.541,
      0.541,
      0.68,
      0.68,
      0.68
    ),
    confidence = c(
      0.917,
      0.953,
      0.929,
      0.958,
      0.905,
      0.947,
      0.924,
      0.956,
      0.922,
      0.914,
      0.952,
      0.916,
      0.953,
      0.949,
      0.913,
      0.955,
      0.926,
      0.92,
      0.905,
      0.903,
      0.946,
      0.904,
      0.946,
      0.942,
      0.914,
      0.921,
      0.955,
      0.922,
      0.955,
      0.953,
      0.92,
      0.913,
      0.92,
      0.95,
      0.921,
      0.947,
      0.91,
      0.948,
      0.912,
      0.944,
      0.919,
      0.903,
      0.94,
      0.954,
      0.913,
      0.951,
      0.917,
      0.952,
      0.918,
      0.946,
      0.908,
      0.919
    ),
    lift = c(
      1,
      1,
      1.013,
      1.005,
      0.987,
      0.993,
      1.007,
      1.003,
      1.027,
      0.997,
      0.998,
      0.998,
      0.999,
      0.996,
      0.996,
      1.002,
      1.009,
      1.026,
      1.059,
      0.984,
      0.992,
      0.985,
      0.993,
      0.988,
      1.019,
      1.004,
      1.002,
      1.005,
      1.002,
      1,
      1.003,
      0.995,
      1.025,
      0.997,
      1.026,
      0.994,
      0.992,
      0.995,
      0.994,
      0.991,
      1.024,
      1.056,
      0.987,
      1,
      1.017,
      0.998,
      1,
      0.998,
      1.001,
      0.992,
      0.99,
      1.024
    ),
    count = c(
      44807,
      46560,
      26550,
      27384,
      29553,
      30922,
      31326,
      32431,
      38493,
      38184,
      39742,
      40146,
      41752,
      42525,
      42525,
      25357,
      25357,
      26450,
      26450,
      25950,
      27177,
      26404,
      27651,
      27825,
      26540,
      26728,
      27717,
      27789,
      28803,
      29851,
      29851,
      35140,
      35140,
      36585,
      36585,
      36164,
      36164,
      38066,
      38066,
      24976,
      24976,
      24976,
      24832,
      25307,
      25307,
      25421,
      25421,
      26447,
      26447,
      33232,
      33232,
      33232
    ),
    coverage = c(
      1,
      1,
      0.585,
      0.585,
      0.668,
      0.668,
      0.694,
      0.694,
      0.855,
      0.855,
      0.855,
      0.897,
      0.897,
      0.917,
      0.953,
      0.544,
      0.561,
      0.588,
      0.598,
      0.588,
      0.588,
      0.598,
      0.598,
      0.605,
      0.594,
      0.594,
      0.594,
      0.617,
      0.617,
      0.641,
      0.664,
      0.788,
      0.782,
      0.788,
      0.814,
      0.782,
      0.814,
      0.822,
      0.855,
      0.542,
      0.556,
      0.566,
      0.541,
      0.543,
      0.567,
      0.547,
      0.567,
      0.569,
      0.59,
      0.719,
      0.749,
      0.74
    ),
    rhsSupport = c(
      0.917,
      0.953,
      0.917,
      0.953,
      0.917,
      0.953,
      0.917,
      0.953,
      0.897,
      0.917,
      0.953,
      0.917,
      0.953,
      0.953,
      0.917,
      0.953,
      0.917,
      0.897,
      0.855,
      0.917,
      0.953,
      0.917,
      0.953,
      0.953,
      0.897,
      0.917,
      0.953,
      0.917,
      0.953,
      0.953,
      0.917,
      0.917,
      0.897,
      0.953,
      0.897,
      0.953,
      0.917,
      0.953,
      0.917,
      0.953,
      0.897,
      0.855,
      0.953,
      0.953,
      0.897,
      0.953,
      0.917,
      0.953,
      0.917,
      0.953,
      0.917,
      0.897
    ),
    leverage = c(
      0,
      0,
      0.007,
      0.003,
      -0.008,
      -0.004,
      0.005,
      0.002,
      0.021,
      -0.003,
      -0.001,
      -0.001,
      -0.001,
      -0.004,
      -0.004,
      0.001,
      0.005,
      0.014,
      0.03,
      -0.008,
      -0.004,
      -0.008,
      -0.004,
      -0.007,
      0.01,
      0.002,
      0.001,
      0.003,
      0.001,
      0,
      0.002,
      -0.004,
      0.018,
      -0.002,
      0.019,
      -0.005,
      -0.006,
      -0.004,
      -0.005,
      -0.005,
      0.012,
      0.027,-0.007,
      0,
      0.009,
      -0.001,
      0,
      -0.001,
      0,
      -0.005,
      -0.007,
      0.016
    ),
    hyperLift = c(
      1,
      1,
      1.01,
      1.003,
      0.984,
      0.992,
      1.005,
      1.002,
      1.026,
      0.995,
      0.997,
      0.997,
      0.998,
      0.995,
      0.995,
      1,
      1.007,
      1.023,
      1.055,
      0.982,
      0.99,
      0.982,
      0.991,
      0.986,
      1.016,
      1.001,
      1,
      1.002,
      1,
      0.998,
      1.001,
      0.993,
      1.024,
      0.996,
      1.024,
      0.992,
      0.99,
      0.994,
      0.993,
      0.988,
      1.021,
      1.052,
      0.984,
      0.998,
      1.014,
      0.996,
      0.997,
      0.996,
      0.998,
      0.991,
      0.988,
      1.022
    ),
    hyperConfidence = c(
      0,
      0,
      1,
      1,
      0,
      0,
      1,
      1,
      1,
      0,
      0,
      0,
      0.01,
      0,
      0,
      0.978,
      1,
      1,
      1,
      0,
      0,
      0,
      0,
      0,
      1,
      1,
      0.983,
      1,
      0.998,
      0.298,
      1,
      0,
      1,
      0,
      1,
      0,
      0,
      0,
      0,
      0,
      1,
      1,
      0,
      0.61,
      1,
      0.006,
      0.412,
      0.028,
      0.779,
      0,
      0,
      1
    ),
    fishersExactTest = c(
      1,
      1,
      0,
      0,
      1,
      1,
      0,
      0,
      0,
      1,
      1,
      1,
      0.99,
      1,
      1,
      0.022,
      0,
      0,
      0,
      1,
      1,
      1,
      1,
      1,
      0,
      0,
      0.017,
      0,
      0.002,
      0.702,
      0,
      1,
      0,
      1,
      0,
      1,
      1,
      1,
      1,
      1,
      0,
      0,
      1,
      0.39,
      0,
      0.994,
      0.588,
      0.972,
      0.221,
      1,
      1,
      0
    ),
    improvement = c(
      0.917,
      0.953,
      0.012,
      0.005,
      -0.012,
      -0.006,
      0.007,
      0.003,
      0.922,
      -0.003,
      -0.002,
      -0.001,-0.001,
      -0.004,
      -0.004,
      -0.003,
      -0.003,
      -0.001,
      0.905,
      -0.014,-0.007,
      -0.014,
      -0.007,
      -0.012,
      -0.007,
      -0.003,
      -0.002,
      -0.002,-0.001,
      -0.004,
      -0.003,
      -0.004,
      -0.001,
      -0.003,
      -0.001,
      -0.006,-0.007,
      -0.005,
      -0.006,
      -0.009,
      -0.003,
      -0.002,
      -0.013,
      -0.003,-0.009,
      -0.005,
      -0.007,
      -0.005,
      -0.006,
      -0.008,
      -0.009,
      -0.003
    ),
    chiSquared = c(
      NA,
      NA,
      124.024,
      38.278,
      194.725,
      85.079,
      62.203,
      25.802,
      1847.852,
      35.656,
      17.552,
      12.358,
      5.138,
      215.573,
      215.573,
      4.175,
      60.714,
      403.069,
      1471.306,
      188.513,
      88.095,
      184.183,
      81.673,
      231.929,
      224.338,
      11.604,
      4.589,
      20.647,
      8.588,
      0.259,
      11.924,
      48.398,
      993.053,
      33.023,
      1240.572,
      150.033,
      154.846,
      131.101,
      122.002,
      105.056,
      310.185,
      1195.368,
      211.9,
      0.091,
      169.967,
      6.287,
      0.042,
      3.57,
      0.617,
      161.375,
      157.068,
      699.642
    ),
    cosine = c(
      0.958,
      0.976,
      0.742,
      0.751,
      0.773,
      0.793,
      0.804,
      0.816,
      0.9,
      0.883,
      0.901,
      0.906,
      0.924,
      0.931,
      0.931,
      0.721,
      0.724,
      0.745,
      0.757,
      0.723,
      0.743,
      0.73,
      0.75,
      0.75,
      0.744,
      0.741,
      0.754,
      0.756,
      0.769,
      0.782,
      0.783,
      0.846,
      0.859,
      0.864,
      0.877,
      0.858,
      0.857,
      0.88,
      0.88,
      0.712,
      0.724,
      0.735,
      0.708,
      0.72,
      0.726,
      0.721,
      0.721,
      0.735,
      0.736,
      0.822,
      0.821,
      0.835
    ),
    conviction = c(
      1,
      1,
      1.165,
      1.119,
      0.871,
      0.883,
      1.086,
      1.074,
      1.31,
      0.964,
      0.966,
      0.982,
      0.985,
      0.917,
      0.953,
      1.04,
      1.116,
      1.29,
      1.528,
      0.852,
      0.862,
      0.856,
      0.869,
      0.799,
      1.199,
      1.044,
      1.038,
      1.057,
      1.05,
      0.992,
      1.038,
      0.948,
      1.287,
      0.943,
      1.291,
      0.883,
      0.918,
      0.902,
      0.936,
      0.838,
      1.267,
      1.498,
      0.785,
      1.006,
      1.18,
      0.955,
      0.997,
      0.967,
      1.01,
      0.86,
      0.901,
      1.265
    ),
    gini = c(
      NA,
      NA,
      0,
      0,
      0.001,
      0,
      0,
      0,
      0.007,
      0,
      0,
      0,
      0,
      0,
      0.001,
      0,
      0,
      0.002,
      0.007,
      0.001,
      0,
      0.001,
      0,
      0,
      0.001,
      0,
      0,
      0,
      0,
      0,
      0,
      0,
      0.004,
      0,
      0.005,
      0,
      0,
      0,
      0,
      0,
      0.001,
      0.006,
      0,
      0,
      0.001,
      0,
      0,
      0,
      0,
      0,
      0,
      0.003
    ),
    oddsRatio = c(
      NA,
      NA,
      1.441,
      1.304,
      0.587,
      0.634,
      1.31,
      1.256,
      3.84,
      0.736,
      0.756,
      0.816,
      0.843,
      0,
      0,
      1.092,
      1.291,
      1.815,
      2.683,
      0.618,
      0.652,
      0.619,
      0.66,
      0.476,
      1.561,
      1.12,
      1.097,
      1.164,
      1.136,
      0.977,
      1.126,
      0.739,
      2.611,
      0.719,
      2.963,
      0.451,
      0.534,
      0.435,
      0.535,
      0.634,
      1.69,
      2.45,
      0.516,
      1.013,
      1.474,
      0.897,
      0.993,
      0.921,
      1.027,
      0.489,
      0.584,
      2.222
    ),
    phi = c(
      NA,
      NA,
      0.05,
      0.028,
      -0.063,
      -0.042,
      0.036,
      0.023,
      0.195,
      -0.027,-0.019,
      -0.016,
      -0.01,
      -0.066,
      -0.066,
      0.009,
      0.035,
      0.091,
      0.174,-0.062,
      -0.042,
      -0.061,
      -0.041,
      -0.069,
      0.068,
      0.015,
      0.01,
      0.021,
      0.013,
      -0.002,
      0.016,
      -0.031,
      0.143,
      -0.026,
      0.159,
      -0.055,
      -0.056,-0.052,
      -0.05,
      -0.046,
      0.08,
      0.156,
      -0.066,
      0.001,
      0.059,
      -0.011,-0.001,
      -0.009,
      0.004,
      -0.057,
      -0.057,
      0.12
    ),
    doc = c(
      NA,
      NA,
      0.028,
      0.012,
      -0.037,
      -0.019,
      0.021,
      0.011,
      0.168,
      -0.021,
      -0.011,-0.014,
      -0.007,
      -0.051,
      -0.087,
      0.004,
      0.02,
      0.056,
      0.125,
      -0.035,-0.018,
      -0.034,
      -0.018,
      -0.03,
      0.042,
      0.009,
      0.004,
      0.012,
      0.006,-0.001,
      0.009,
      -0.021,
      0.105,
      -0.013,
      0.124,
      -0.028,
      -0.04,
      -0.029,-0.039,
      -0.02,
      0.049,
      0.111,
      -0.028,
      0.001,
      0.036,
      -0.005,
      -0.001,-0.004,
      0.002,
      -0.027,
      -0.036,
      0.083
    ),
    RLD = c(
      NA,
      NA,
      0.141,
      0.106,
      0.299,
      0.268,
      0.079,
      0.069,
      0.237,
      0.219,
      0.208,
      0.157,
      0.137,
      1,
      1,
      0.038,
      0.104,
      0.225,
      0.345,
      0.247,
      0.229,
      0.25,
      0.225,
      0.385,
      0.166,
      0.042,
      0.036,
      0.054,
      0.047,
      0.014,
      0.037,
      0.202,
      0.223,
      0.227,
      0.226,
      0.474,
      0.392,
      0.503,
      0.404,
      0.228,
      0.21,
      0.333,
      0.323,
      0.006,
      0.152,
      0.056,
      0.004,
      0.044,
      0.01,
      0.416,
      0.326,
      0.21
    ),
    imbalance = c(
      0.083,
      0.047,
      0.347,
      0.377,
      0.254,
      0.288,
      0.23,
      0.263,
      0.044,
      0.063,
      0.099,
      0.02,
      0.056,
      0.036,
      0.036,
      0.419,
      0.372,
      0.327,
      0.282,
      0.338,
      0.37,
      0.327,
      0.36,
      0.352,
      0.32,
      0.335,
      0.366,
      0.311,
      0.343,
      0.317,
      0.261,
      0.131,
      0.12,
      0.166,
      0.087,
      0.172,
      0.105,
      0.132,
      0.063,
      0.419,
      0.362,
      0.318,
      0.419,
      0.419,
      0.348,
      0.414,
      0.363,
      0.392,
      0.339,
      0.236,
      0.171,
      0.164
    ),
    kulczynski = c(
      0.959,
      0.977,
      0.761,
      0.773,
      0.782,
      0.806,
      0.812,
      0.827,
      0.9,
      0.883,
      0.903,
      0.906,
      0.925,
      0.931,
      0.931,
      0.75,
      0.746,
      0.762,
      0.769,
      0.741,
      0.765,
      0.746,
      0.77,
      0.77,
      0.76,
      0.759,
      0.775,
      0.771,
      0.787,
      0.797,
      0.793,
      0.849,
      0.861,
      0.868,
      0.878,
      0.862,
      0.859,
      0.883,
      0.881,
      0.74,
      0.744,
      0.751,
      0.737,
      0.749,
      0.745,
      0.749,
      0.742,
      0.76,
      0.754,
      0.83,
      0.825,
      0.839
    ),
    collectiveStrength = c(
      0,
      0,
      2100.469,
      1109.947,
      885.158,
      537.551,
      1505.665,
      821.045,
      2209.719,
      444.05,
      257.474,
      343.017,
      199.964,
      0,
      0,
      1095.646,
      2071.171,
      2983.257,
      5296.758,
      1183.735,
      703.514,
      1151.931,
      690.502,
      528.595,
      2702.14,
      1763.053,
      981.469,
      1710.534,
      948.714,
      805.671,
      1476.354,
      660.623,
      2315.901,
      366.007,
      2236.759,
      250.406,
      431.434,
      193.157,
      331.092,
      783.815,
      3036.8,
      5290.847,
      680.099,
      1049.923,
      2754.658,
      967.708,
      1737.545,
      934.617,
      1684.277,
      358.06,
      645.367,
      2372.283
    ),
    jaccard = c(
      0.917,
      0.953,
      0.567,
      0.573,
      0.617,
      0.64,
      0.661,
      0.675,
      0.817,
      0.789,
      0.818,
      0.828,
      0.858,
      0.871,
      0.871,
      0.531,
      0.541,
      0.574,
      0.594,
      0.545,
      0.565,
      0.554,
      0.574,
      0.576,
      0.573,
      0.567,
      0.579,
      0.589,
      0.601,
      0.621,
      0.63,
      0.73,
      0.75,
      0.755,
      0.779,
      0.744,
      0.747,
      0.783,
      0.785,
      0.52,
      0.543,
      0.562,
      0.516,
      0.53,
      0.547,
      0.531,
      0.54,
      0.552,
      0.561,
      0.686,
      0.69,
      0.711
    ),
    kappa = c(
      0,
      0,
      0.032,
      0.014,
      -0.046,-0.024,
      0.027,
      0.014,
      0.191,
      -0.026,
      -0.016,
      -0.016,
      -0.009,-0.063,
      -0.063,
      0.004,
      0.021,
      0.063,
      0.139,
      -0.039,
      -0.021,
      -0.04,-0.021,
      -0.035,
      0.048,
      0.01,
      0.005,
      0.014,
      0.007,
      -0.001,
      0.011,-0.027,
      0.129,
      -0.019,
      0.15,
      -0.04,
      -0.051,
      -0.04,
      -0.048,
      -0.021,
      0.053,
      0.121,
      -0.03,
      0.001,
      0.04,
      -0.005,
      -0.001,
      -0.004,
      0.002,-0.036,
      -0.046,
      0.103
    ),
    mutualInformation = c(
      NA,
      NA,
      0.004,
      0.002,
      0.007,
      0.005,
      0.002,
      0.001,
      0.046,
      0.001,
      0.001,
      0,
      0,
      NA,
      NA,
      0,
      0.002,
      0.012,
      0.036,
      0.007,
      0.005,
      0.007,
      0.005,
      0.013,
      0.007,
      0,
      0,
      0.001,
      0,
      0,
      0,
      0.002,
      0.027,
      0.002,
      0.033,
      0.009,
      0.006,
      0.008,
      0.005,
      0.006,
      0.01,
      0.029,
      0.012,
      0,
      0.005,
      0,
      0,
      0,
      0,
      0.01,
      0.006,
      0.02
    ),
    lambda = c(
      0,
      0,
      0,
      0,
      0,
      0,
      0,
      0,
      0,
      0,
      0,
      0,
      0,
      0,
      0,
      0,
      0,
      0,
      0,
      0,
      0,
      0,
      0,
      0,
      0,
      0,
      0,
      0,
      0,
      0,
      0,
      0,
      0,
      0,
      0,
      0,
      0,
      0,
      0,
      0,
      0,
      0,
      0,
      0,
      0,
      0,
      0,
      0,
      0,
      0,
      0,
      0
    ),
    jMeasure = c(
      0,
      0,
      0.001,
      0,
      0.001,
      0,
      0,
      0,
      0.003,
      0,
      0,
      0,
      0,
      0,
      0,
      0,
      0,
      0.002,
      0.007,
      0.001,
      0,
      0.001,
      0,
      0.001,
      0.001,
      0,
      0,
      0,
      0,
      0,
      0,
      0,
      0.002,
      0,
      0.003,
      0,
      0,
      0,
      0,
      0,
      0.002,
      0.006,
      0.001,
      0,
      0.001,
      0,
      0,
      0,
      0,
      0,
      0,
      0.002
    ),
    laplace = c(
      0.917,
      0.953,
      0.929,
      0.958,
      0.905,
      0.947,
      0.924,
      0.956,
      0.922,
      0.914,
      0.952,
      0.916,
      0.953,
      0.949,
      0.913,
      0.955,
      0.926,
      0.92,
      0.905,
      0.903,
      0.946,
      0.904,
      0.946,
      0.941,
      0.914,
      0.921,
      0.955,
      0.922,
      0.955,
      0.953,
      0.92,
      0.913,
      0.92,
      0.95,
      0.921,
      0.947,
      0.91,
      0.948,
      0.912,
      0.944,
      0.919,
      0.903,
      0.94,
      0.954,
      0.913,
      0.951,
      0.917,
      0.952,
      0.918,
      0.946,
      0.908,
      0.919
    ),
    certainty = c(
      0,
      0,
      0.141,
      0.106,
      -0.148,
      -0.133,
      0.079,
      0.069,
      0.237,
      -0.037,
      -0.035,
      -0.018,-0.016,
      -0.09,
      -0.049,
      0.038,
      0.104,
      0.225,
      0.345,
      -0.173,
      -0.16,-0.168,
      -0.151,
      -0.251,
      0.166,
      0.042,
      0.036,
      0.054,
      0.047,
      -0.008,
      0.037,
      -0.054,
      0.223,
      -0.061,
      0.226,
      -0.132,
      -0.09,
      -0.109,
      -0.069,-0.193,
      0.21,
      0.333,
      -0.274,
      0.006,
      0.152,
      -0.047,
      -0.003,
      -0.034,
      0.01,
      -0.162,
      -0.109,
      0.21
    ),
    addedValue = c(
      0,
      0,
      0.012,
      0.005,-0.012,
      -0.006,
      0.007,
      0.003,
      0.024,
      -0.003,
      -0.002,
      -0.001,-0.001,
      -0.004,
      -0.004,
      0.002,
      0.009,
      0.023,
      0.05,
      -0.014,
      -0.007,-0.014,
      -0.007,
      -0.012,
      0.017,
      0.004,
      0.002,
      0.004,
      0.002,
      0,
      0.003,
      -0.004,
      0.023,
      -0.003,
      0.023,
      -0.006,
      -0.007,
      -0.005,-0.006,
      -0.009,
      0.022,
      0.048,
      -0.013,
      0,
      0.016,
      -0.002,
      0,
      -0.002,
      0.001,
      -0.008,
      -0.009,
      0.022
    ),
    maxconfidence = c(
      1,
      1,
      0.929,
      0.958,
      0.905,
      0.947,
      0.924,
      0.956,
      0.922,
      0.914,
      0.952,
      0.916,
      0.953,
      0.949,
      0.949,
      0.955,
      0.926,
      0.92,
      0.905,
      0.903,
      0.946,
      0.904,
      0.946,
      0.942,
      0.914,
      0.921,
      0.955,
      0.922,
      0.955,
      0.953,
      0.92,
      0.913,
      0.92,
      0.95,
      0.921,
      0.947,
      0.91,
      0.948,
      0.912,
      0.944,
      0.919,
      0.903,
      0.94,
      0.954,
      0.913,
      0.951,
      0.917,
      0.952,
      0.918,
      0.946,
      0.908,
      0.919
    ),
    rulePowerFactor = c(
      0.842,
      0.909,
      0.505,
      0.537,
      0.548,
      0.6,
      0.593,
      0.635,
      0.726,
      0.715,
      0.774,
      0.753,
      0.814,
      0.826,
      0.795,
      0.496,
      0.481,
      0.498,
      0.49,
      0.48,
      0.526,
      0.488,
      0.536,
      0.536,
      0.497,
      0.504,
      0.542,
      0.524,
      0.563,
      0.582,
      0.563,
      0.657,
      0.662,
      0.712,
      0.69,
      0.701,
      0.674,
      0.739,
      0.711,
      0.483,
      0.47,
      0.462,
      0.478,
      0.494,
      0.473,
      0.495,
      0.477,
      0.515,
      0.497,
      0.643,
      0.618,
      0.625
    ),
    ralambondrainy = c(
      0.083,
      0.047,
      0.042,
      0.024,
      0.063,
      0.035,
      0.053,
      0.03,
      0.067,
      0.073,
      0.041,
      0.075,
      0.043,
      0.047,
      0.083,
      0.024,
      0.042,
      0.047,
      0.057,
      0.057,
      0.032,
      0.058,
      0.032,
      0.035,
      0.051,
      0.047,
      0.027,
      0.048,
      0.027,
      0.03,
      0.053,
      0.069,
      0.062,
      0.039,
      0.065,
      0.041,
      0.073,
      0.043,
      0.075,
      0.03,
      0.045,
      0.055,
      0.032,
      0.025,
      0.049,
      0.027,
      0.047,
      0.027,
      0.048,
      0.039,
      0.069,
      0.06
    ),
    confirmedConfidence = c(
      0.835,
      0.907,
      0.858,
      0.917,
      0.81,
      0.894,
      0.848,
      0.913,
      0.843,
      0.829,
      0.903,
      0.832,
      0.905,
      0.898,
      0.827,
      0.91,
      0.852,
      0.841,
      0.81,
      0.806,
      0.892,
      0.807,
      0.892,
      0.883,
      0.829,
      0.842,
      0.91,
      0.844,
      0.911,
      0.906,
      0.841,
      0.826,
      0.841,
      0.901,
      0.841,
      0.894,
      0.82,
      0.896,
      0.823,
      0.889,
      0.838,
      0.807,
      0.881,
      0.907,
      0.826,
      0.902,
      0.834,
      0.903,
      0.836,
      0.891,
      0.817,
      0.838
    ),
    sebag = c(
      11.105,
      20.403,
      13.098,
      22.954,
      9.542,
      17.895,
      12.142,
      21.987,
      11.775,
      10.672,
      19.674,
      10.891,
      20.073,
      18.635,
      10.539,
      21.255,
      12.51,
      11.575,
      9.538,
      9.318,
      17.444,
      9.366,
      17.59,
      16.102,
      10.684,
      11.641,
      21.207,
      11.795,
      21.463,
      20.238,
      11.57,
      10.48,
      11.544,
      19.175,
      11.589,
      17.903,
      10.107,
      18.301,
      10.327,
      16.944,
      11.348,
      9.337,
      15.796,
      20.525,
      10.501,
      19.45,
      11.072,
      19.707,
      11.225,
      17.417,
      9.911,
      11.334
    ),
    counterexample = c(
      0.91,
      0.951,
      0.924,
      0.956,
      0.895,
      0.944,
      0.918,
      0.955,
      0.915,
      0.906,
      0.949,
      0.908,
      0.95,
      0.946,
      0.905,
      0.953,
      0.92,
      0.914,
      0.895,
      0.893,
      0.943,
      0.893,
      0.943,
      0.938,
      0.906,
      0.914,
      0.953,
      0.915,
      0.953,
      0.951,
      0.914,
      0.905,
      0.913,
      0.948,
      0.914,
      0.944,
      0.901,
      0.945,
      0.903,
      0.941,
      0.912,
      0.893,
      0.937,
      0.951,
      0.905,
      0.949,
      0.91,
      0.949,
      0.911,
      0.943,
      0.899,
      0.912
    ),
    casualSupport = c(
      1.752,
      1.86,
      1.419,
      1.49,
      1.459,
      1.551,
      1.506,
      1.587,
      1.619,
      1.626,
      1.726,
      1.664,
      1.766,
      1.777,
      1.705,
      1.448,
      1.395,
      1.392,
      1.34,
      1.392,
      1.478,
      1.4,
      1.487,
      1.488,
      1.39,
      1.418,
      1.494,
      1.438,
      1.516,
      1.534,
      1.476,
      1.568,
      1.555,
      1.663,
      1.582,
      1.652,
      1.585,
      1.69,
      1.621,
      1.434,
      1.364,
      1.312,
      1.43,
      1.446,
      1.366,
      1.447,
      1.391,
      1.467,
      1.411,
      1.595,
      1.529,
      1.518
    ),
    casualConfidence = c(
      1,
      1,
      1,
      1,
      1,
      1,
      1,
      1,
      1,
      1,
      1,
      1,
      1,
      1,
      1,
      1,
      1,
      1,
      1,
      1,
      1,
      1,
      1,
      1,
      1,
      1,
      1,
      1,
      1,
      1,
      1,
      1,
      1,
      1,
      1,
      1,
      1,
      1,
      1,
      1,
      1,
      1,
      1,
      1,
      1,
      1,
      1,
      1,
      1,
      1,
      1,
      1
    ),
    leastContradiction = c(
      1,
      1,
      0.593,
      0.588,
      0.66,
      0.664,
      0.699,
      0.697,
      0.878,
      0.852,
      0.854,
      0.896,
      0.897,
      0.913,
      0.949,
      0.545,
      0.566,
      0.603,
      0.633,
      0.579,
      0.584,
      0.589,
      0.594,
      0.598,
      0.605,
      0.597,
      0.595,
      0.62,
      0.619,
      0.641,
      0.666,
      0.784,
      0.802,
      0.786,
      0.835,
      0.777,
      0.807,
      0.818,
      0.85,
      0.536,
      0.57,
      0.598,
      0.533,
      0.544,
      0.577,
      0.546,
      0.567,
      0.568,
      0.59,
      0.714,
      0.742,
      0.758
    ),
    centeredConfidence = c(
      0,
      0,
      0.012,
      0.005,
      -0.012,
      -0.006,
      0.007,
      0.003,
      0.024,
      -0.003,
      -0.002,
      -0.001,
      -0.001,
      -0.004,-0.004,
      0.002,
      0.009,
      0.023,
      0.05,
      -0.014,
      -0.007,
      -0.014,-0.007,
      -0.012,
      0.017,
      0.004,
      0.002,
      0.004,
      0.002,
      0,
      0.003,-0.004,
      0.023,
      -0.003,
      0.023,
      -0.006,
      -0.007,
      -0.005,
      -0.006,-0.009,
      0.022,
      0.048,
      -0.013,
      0,
      0.016,
      -0.002,
      0,
      -0.002,
      0.001,
      -0.008,
      -0.009,
      0.022
    ),
    varyingLiaison = c(
      0,
      0,
      0.013,
      0.005,
      -0.013,
      -0.007,
      0.007,
      0.003,
      0.027,
      -0.003,
      -0.002,-0.002,
      -0.001,
      -0.004,
      -0.004,
      0.002,
      0.009,
      0.026,
      0.059,-0.016,
      -0.008,
      -0.015,
      -0.007,
      -0.012,
      0.019,
      0.004,
      0.002,
      0.005,
      0.002,
      0,
      0.003,
      -0.005,
      0.025,
      -0.003,
      0.026,
      -0.006,-0.008,
      -0.005,
      -0.006,
      -0.009,
      0.024,
      0.056,
      -0.013,
      0,
      0.017,
      -0.002,
      0,
      -0.002,
      0.001,
      -0.008,
      -0.01,
      0.024
    ),
    yuleQ = c(
      NA,
      NA,
      0.181,
      0.132,
      -0.26,
      -0.224,
      0.134,
      0.113,
      0.587,
      -0.152,-0.139,
      -0.102,
      -0.085,
      -1,
      -1,
      0.044,
      0.127,
      0.289,
      0.457,-0.236,
      -0.211,
      -0.235,
      -0.204,
      -0.355,
      0.219,
      0.056,
      0.046,
      0.076,
      0.064,
      -0.011,
      0.059,
      -0.15,
      0.446,
      -0.164,
      0.495,-0.378,
      -0.303,
      -0.394,
      -0.303,
      -0.224,
      0.257,
      0.42,
      -0.319,
      0.006,
      0.192,
      -0.054,
      -0.003,
      -0.041,
      0.013,
      -0.343,
      -0.263,
      0.379
    ),
    yuleY = c(
      NA,
      NA,
      0.091,
      0.066,
      -0.132,
      -0.113,
      0.067,
      0.057,
      0.324,
      -0.076,
      -0.07,
      -0.051,
      -0.043,
      -1,
      -1,
      0.022,
      0.064,
      0.148,
      0.242,
      -0.12,
      -0.107,
      -0.119,
      -0.103,
      -0.183,
      0.111,
      0.028,
      0.023,
      0.038,
      0.032,
      -0.006,
      0.03,
      -0.075,
      0.235,
      -0.082,
      0.265,
      -0.196,
      -0.155,
      -0.205,
      -0.155,
      -0.113,
      0.131,
      0.22,
      -0.164,
      0.003,
      0.097,
      -0.027,
      -0.002,
      -0.021,
      0.007,
      -0.177,
      -0.134,
      0.197
    ),
    lerman = c(
      0,
      0,
      2.062,
      0.861,-2.309,
      -1.148,
      1.254,
      0.607,
      5.242,
      -0.653,
      -0.345,
      -0.324,-0.157,
      -0.912,
      -0.912,
      0.298,
      1.484,
      4.126,
      9.256,
      -2.532,-1.302,
      -2.472,
      -1.238,
      -2.069,
      3.056,
      0.624,
      0.295,
      0.808,
      0.392,
      -0.066,
      0.575,
      -0.92,
      4.715,
      -0.572,
      4.869,
      -1.237,-1.544,
      -1.044,
      -1.21,
      -1.5,
      3.757,
      8.671,
      -2.133,
      0.044,
      2.746,
      -0.365,
      -0.039,
      -0.268,
      0.145,
      -1.454,
      -1.805,
      4.316
    ),
    implicationIndex = c(
      0,
      0,
      -6.871,
      -3.891,
      7.696,
      5.185,-4.177,
      -2.743,
      -15.504,
      2.178,
      1.557,
      1.078,
      0.709,
      4.12,
      3.04,
      -1.348,
      -4.947,
      -12.203,
      -22.48,
      8.438,
      5.88,
      8.238,
      5.592,
      9.344,
      -9.038,
      -2.078,
      -1.332,
      -2.693,
      -1.77,
      0.298,-1.917,
      3.067,
      -13.945,
      2.583,
      -14.402,
      5.587,
      5.145,
      4.717,
      4.031,
      6.776,
      -11.112,
      -21.058,
      9.633,
      -0.199,
      -8.122,
      1.647,
      0.13,
      1.211,
      -0.482,
      6.569,
      6.013,
      -12.766
    ),
    importance = c(
      0.264,
      0.28,
      0.013,
      0.005,
      -0.017,
      -0.008,
      0.01,
      0.005,
      0.087,
      -0.01,-0.005,
      -0.007,
      -0.003,
      -0.023,
      -0.039,
      0.002,
      0.009,
      0.027,
      0.064,
      -0.016,
      -0.008,
      -0.016,
      -0.008,
      -0.014,
      0.02,
      0.004,
      0.002,
      0.006,
      0.003,
      0,
      0.004,
      -0.01,
      0.052,
      -0.006,
      0.063,-0.013,
      -0.019,
      -0.013,
      -0.018,
      -0.009,
      0.024,
      0.057,
      -0.013,
      0,
      0.018,
      -0.002,
      0,
      -0.002,
      0.001,
      -0.012,
      -0.017,
      0.041
    ),
    stdLift = c(
      1,
      1,
      0.291,
      0.477,
      0.051,
      0.243,
      0.239,
      0.354,
      0.217,
      0.113,
      0.115,
      0.086,
      0.089,
      0,
      0,
      0.44,
      0.26,
      0.205,
      0.051,
      0.031,
      0.317,
      0.035,
      0.311,
      0.243,
      0.144,
      0.209,
      0.427,
      0.218,
      0.412,
      0.354,
      0.204,
      0.129,
      0.203,
      0.164,
      0.206,
      0.115,
      0.1,
      0.089,
      0.086,
      0.274,
      0.19,
      0.033,
      0.207,
      0.418,
      0.13,
      0.427,
      0.172,
      0.412,
      0.182,
      0.164,
      0.084,
      0.189
    ),
    boost = c(
      Inf,
      Inf,
      1.013,
      1.005,
      0.987,
      0.993,
      1.007,
      1.003,
      Inf,
      0.997,
      0.998,
      0.998,
      0.999,
      0.996,
      0.996,
      0.997,
      0.997,
      0.999,
      Inf,
      0.984,
      0.992,
      0.985,
      0.993,
      0.988,
      0.992,
      0.997,
      0.998,
      0.998,
      0.999,
      0.996,
      0.996,
      0.995,
      0.998,
      0.997,
      0.999,
      0.994,
      0.992,
      0.995,
      0.994,
      0.991,
      0.997,
      0.998,
      0.987,
      0.997,
      0.991,
      0.994,
      0.993,
      0.995,
      0.994,
      0.992,
      0.99,
      0.997
    ),
    table.n11 = c(
      44807,
      46560,
      26550,
      27384,
      29553,
      30922,
      31326,
      32431,
      38493,
      38184,
      39742,
      40146,
      41752,
      42525,
      42525,
      25357,
      25357,
      26450,
      26450,
      25950,
      27177,
      26404,
      27651,
      27825,
      26540,
      26728,
      27717,
      27789,
      28803,
      29851,
      29851,
      35140,
      35140,
      36585,
      36585,
      36164,
      36164,
      38066,
      38066,
      24976,
      24976,
      24976,
      24832,
      25307,
      25307,
      25421,
      25421,
      26447,
      26447,
      33232,
      33232,
      33232
    ),
    table.n01 = c(
      0,
      0,
      18257,
      19176,
      15254,
      15638,
      13481,
      14129,
      5339,
      6623,
      6818,
      4661,
      4808,
      4035,
      2282,
      21203,
      19450,
      17382,
      15312,
      18857,
      19383,
      18403,
      18909,
      18735,
      17292,
      18079,
      18843,
      17018,
      17757,
      16709,
      14956,
      9667,
      8692,
      9975,
      7247,
      10396,
      8643,
      8494,
      6741,
      21584,
      18856,
      16786,
      21728,
      21253,
      18525,
      21139,
      19386,
      20113,
      18360,
      13328,
      11575,
      10600
    ),
    table.n10 = c(
      4035,
      2282,
      2027,
      1193,
      3097,
      1728,
      2580,
      1475,
      3269,
      3578,
      2020,
      3686,
      2080,
      2282,
      4035,
      1193,
      2027,
      2285,
      2773,
      2785,
      1558,
      2819,
      1572,
      1728,
      2484,
      2296,
      1307,
      2356,
      1342,
      1475,
      2580,
      3353,
      3044,
      1908,
      3157,
      2020,
      3578,
      2080,
      3686,
      1474,
      2201,
      2675,
      1572,
      1233,
      2410,
      1307,
      2296,
      1342,
      2356,
      1908,
      3353,
      2932
    ),
    table.n00 = c(
      0,
      0,
      2008,
      1089,
      938,
      554,
      1455,
      807,
      1741,
      457,
      262,
      349,
      202,
      0,
      0,
      1089,
      2008,
      2725,
      4307,
      1250,
      724,
      1216,
      710,
      554,
      2526,
      1739,
      975,
      1679,
      940,
      807,
      1455,
      682,
      1966,
      374,
      1853,
      262,
      457,
      202,
      349,
      808,
      2809,
      4405,
      710,
      1049,
      2600,
      975,
      1739,
      940,
      1679,
      374,
      682,
      2078
    ),
    relativeRisk = c(NaN, NaN, 1.031, 1.013, 0.961, 0.981, 1.024, 1.011, 1.222, 
      0.977, 0.988, 0.984, 0.993, 0.949, 0.913, 1.004, 1.022, 1.065, 
      1.16, 0.963, 0.981, 0.963, 0.982, 0.969, 1.048, 1.009, 1.004, 
      1.013, 1.006, 0.999, 1.01, 0.977, 1.128, 0.986, 1.156, 0.971, 
      0.958, 0.971, 0.959, 0.98, 1.056, 1.14, 0.971, 1.001, 1.041, 
      0.995, 0.999, 0.996, 1.002, 0.972, 0.962, 1.099)
  ),
  row.names = c(NA,-52L),
  class = "data.frame"
)

if (!all(setequal(names(m_previous), names(m_r))))
  warning("Not all interestMeasures are tested! Missing data for: ", paste(setdiff(names(m_r), names(m_previous)), collapse = ", "))
expect_equivalent(m_previous, round(m_r[names(m_previous)], 3))
