#' Extracts the backbone of a weighted network using results from a null model
#'
#' `backbone.extract` returns a binary or signed adjacency matrix
#'      containing the backbone that retains only the significant edges.
#'
#' @param positive Matrix: proportion of times the projected matrix values were above the corresponding matrices generated by \link{hyperg}, \link{fdsm}, or \link{sdsm}.
#' @param negative Matrix: proportion of times the projected matrix values were below the corresponding matrices generated by \link{hyperg}, \link{fdsm}, or \link{sdsm}. If supplied, a signed backbone will be returned; if not supplied, a binary backbone will be returned. Default is set to NULL.
#' @param alpha Real: Precision of significance test (one-tailed if only the positive matrix supplied, two-tailed if positive and negative matrices supplied)
#'
#' @return backbone Matrix: Binary or signed adjacency matrix of backbone graph.
#' @export
#'
#' @examples
#' probs <- sdsm(davis, 100)
#' bb <- backbone.extract(probs$positive, probs$negative)
backbone.extract <- function(positive, negative = NULL, alpha = 0.05){

  #Argument Checks
  if ((alpha >= 1) | (alpha <= 0)) {stop("alpha must be between 0 and 1")}
  if (length(negative) != 0) {alpha <- alpha / 2}  #Use a two-tailed test for signed backbones

  #Convert values to matrix
  SignedPositive <- as.matrix((positive<=alpha)+0)
  SignedNegative <- as.matrix((negative<=alpha)+0)
  SignedNegative[SignedNegative==1] <- -1

  #Create backbone matrix
  if (length(negative) == 0) {backbone <- SignedPositive
  } else {backbone <- SignedPositive + SignedNegative}
  diag(backbone) <- 0
  rownames(backbone) <- rownames(positive)
  colnames(backbone) <- rownames(positive)
  return(backbone)
}
