% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bain_methods.R
\name{bain}
\alias{bain}
\title{Bayes factors for informative hypotheses}
\usage{
bain(x, hypothesis, fraction = 1, ...)
}
\arguments{
\item{x}{An R object containing the outcome of a statistical analysis.
Currently, the following objects can be processed: \code{lm()},
\code{t_test()}, \code{lavaan} objects created with the
\code{sem()}, \code{cfa()}, and \code{growth()} functions, and named
vector objects. See the vignette for elaborations.}

\item{hypothesis}{A character string containing the informative hypotheses
to evaluate. See the vignette for elaborations.}

\item{fraction}{A number representing the fraction of information
in the data used to construct the prior distribution
(see the tutorial DOI: 10.1037/met0000201): The default value 1 denotes the
minimal fraction, 2 denotes twice the minimal fraction, etc.}

\item{...}{Additional arguments. See the vignette for elaborations.}
}
\value{
The main output resulting from analyses with \code{bain} are
Bayes factors and posterior model probabilities associated with the
hypotheses that are evaluated. See the \strong{tutorial} and the
\strong{vignette} for further elaborations.
}
\description{
\code{bain} is an acronym for "Bayesian informative hypotheses evaluation".
It uses the Bayes factor to evaluate hypotheses specified using equality and
inequality constraints among (linear combinations of) parameters in a wide
range of statistical models. A tutorial by Hoijtink, Mulder, van Lissa,
and Gu (2018), was published in
\href{https://www.doi.org/10.1037\%2Fmet0000201}{Psychological Methods}.
The preprint of that tutorial is available at
\href{https://psyarxiv.com/v3shc/}{DOI:10.31234/osf.io/v3shc}, or on the bain
website at
\url{https://informative-hypotheses.sites.uu.nl/software/bain/}
\strong{Users are
advised to read the tutorial AND the vignette that is provided
with this package before using} \code{bain}.
}
\examples{
# USING BAIN WITH A LM OBJECT: Bayesian ANOVA
# make a factor of variable site
sesamesim$site <- as.factor(sesamesim$site)
# execute an analysis of variance using lm() which, due to the -1, returns
# estimates of the means per group
anov <- lm(postnumb~site-1,sesamesim)
# take a look at the estimated means and their names
coef(anov)
# set a seed value
set.seed(100)
# use the names to formulate and test hypotheses with bain
results <- bain(anov, "site1=site2=site3=site4=site5; site2>site5>site1>
site3>site4")
#
# USING BAIN WITH A NAMED VECTOR: Bayesian ANOVA

# make a factor of variable site
sesamesim$site <- as.factor(sesamesim$site)
# execute an analysis of variance using lm() which, due to the -1, returns
# estimates of the means per group
anov <- lm(postnumb~site-1,sesamesim)
# collect the estimates means in a vector
estimate <- coef(anov)
# give names to the estimates in anov
names(estimate) <- c("site1", "site2", "site3","site4","site5")
# create a vector containing the sample sizes of each group
ngroup <- table(sesamesim$site)
# compute the variance of the means and collect them in a list
var <- summary(anov)$sigma**2
cov1 <- matrix(var/ngroup[1], nrow=1, ncol=1)
cov2 <- matrix(var/ngroup[2], nrow=1, ncol=1)
cov3 <- matrix(var/ngroup[3], nrow=1, ncol=1)
cov4 <- matrix(var/ngroup[4], nrow=1, ncol=1)
cov5 <- matrix(var/ngroup[5], nrow=1, ncol=1)
covlist <- list(cov1, cov2, cov3, cov4,cov5)
# set a seed value
set.seed(100)
# test hypotheses with bain. Note that there are multiple groups
# characterized by one mean, therefore group_parameters=1. Note that
# there are no joint parameters, therefore, joint_parameters=0.
results <- bain(estimate,
"site1=site2=site3=site4=site5; site2>site5>site1>site3>site4",
n=ngroup,Sigma=covlist,group_parameters=1,joint_parameters = 0)

# SEE THE TUTORIAL AND VIGNETTE FOR MANY ADDITIONAL EXAMPLES


}
\references{
See the vignette for additional references.

Hoijtink, H., Mulder, J., van Lissa, C., and Gu, X. (2018). A tutorial on
testing hypotheses using the Bayes factor. \emph{Psychological Methods.}
DOI: 10.1037/met0000201
}
\author{
The main authors of the bain package are Xin Gu, Caspar
van Lissa, Herbert Hoijtink and Joris Mulder. Contributions
were made by Marlyne Bosman and Camiel van Zundert.
Contact information can be
found on the bain website at
\url{https://informative-hypotheses.sites.uu.nl/software/bain/}
}
