% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bdpregression.R
\name{bdpregression}
\alias{bdpregression}
\alias{bdpregression,ANY-method}
\title{Bayesian Discount Prior: Regression Analysis}
\usage{
bdpregression(formula = formula, family = "gaussian", data = data,
  prior_mean = 0, prior_scale = NULL, prior_df = 1,
  prior_mean_for_intercept = 0, prior_scale_for_intercept = NULL,
  prior_df_for_intercept = 1, alpha_max = 1, fix_alpha = FALSE,
  number_mcmc = 10000, weibull_scale = 0.135, weibull_shape = 3,
  two_side = TRUE, method = "fixed")
}
\arguments{
\item{formula}{an object of class "formula." See "Details" for
more information, including specfication of historical and treatment
data status.}

\item{family}{a description of the error distribution and link
function to be used in the model. For bdpregression this can
be a character string naming a family function, a family
function or the result of a call to a family function.}

\item{data}{an optional data frame, list or environment
(or object coercible by as.data.frame to a data frame)
containing the variables in the model. If not found in data,
the variables are taken from environment(formula), typically
the environment from which bdpregression is called.}

\item{prior_mean}{prior mean for the coefficients; default is 0.
Can be a vector of length equal to the number of covariates
(not counting the intercept, if any). If it is a scalar, it is
expanded to the length of this vector.}

\item{prior_scale}{prior scale for the coefficients: default is NULL;
if NULL, for a logistic regression model, prior_scale is 2.5; for a
 probit model, prior scale is 2.5*1.6. Can be a vector of length equal
 to the number of predictors (not counting the intercept, if any). If
 it is a scalar, it is expanded to the length of this vector.}

\item{prior_df}{prior degrees of freedom for the coefficients. For
t distribution default is 1 (Cauchy). Set to Inf to get normal prior
distributions. Can be a vector of length equal to the number of
predictors (not counting the intercept, if any). If it is a scalar,
it is expanded to the length of this vector.}

\item{prior_mean_for_intercept}{prior mean for the intercept: default
is 0.}

\item{prior_scale_for_intercept}{prior scale for the intercept:
default is NULL; for a logit model, prior scale for intercept is 10;
for probit model, prior scale for intercept is rescaled as 10*1.6.}

\item{prior_df_for_intercept}{prior degrees of freedom for the
intercept: default is 1.}

\item{alpha_max}{scalar. Maximum weight the discount function can apply.
Default is 1. For a two-arm trial, users may specify a vector of two values
where the first value is used to weight the historical treatment group and
the second value is used to weight the historical control group.}

\item{fix_alpha}{logical. Fix alpha at alpha_max? Default value is FALSE.}

\item{number_mcmc}{scalar. Number of Markov Chain Monte Carlo (MCMC)
simulations. Default is 10000.}

\item{weibull_scale}{scalar. Scale parameter of the Weibull discount function
used to compute alpha, the weight parameter of the historical data. Default
value is 0.135. For a two-arm trial, users may specify a vector of two
values where the first value is used to estimate the weight of the
historical treatment group and the second value is used to estimate the
weight of the historical control group.}

\item{weibull_shape}{scalar. Shape parameter of the Weibull discount function
used to compute alpha, the weight parameter of the historical data. Default
value is 3. For a two-arm trial, users may specify a vector of two values
where the first value is used to estimate the weight of the historical
treatment group and the second value is used to estimate the weight of the
historical control group.}

\item{two_side}{logical. Indicator of two-sided test for the discount
function. Default value is TRUE.}

\item{method}{character. Analysis method with respect to estimation of the weight
paramter alpha. Default value "\code{fixed}" estimates alpha once and holds it fixed
throughout the analysis. Alternative method "\code{mc}" estimates alpha for each
Monte Carlo iteration. Currently, only the default method "\code{fixed}" is
supported.}
}
\value{
\code{bdpregression} returns an object of class "bdpregression".
  The functions \code{summary} and \code{print} are used to obtain and
  print a summary of the results, including user inputs. The \code{plot}
  function displays visual outputs as well.

An object of class "\code{bdpregression}" is a list containing at least
the following components:
\describe{
 \item{\code{posterior_treatment}}{
   list. Entries contain values related to the treatment group:}
   \itemize{
     \item{\code{alpha_discount}}{
       numeric. Alpha value, the weighting parameter of the historical data.}
     \item{\code{p_hat}}{
       numeric. The posterior probability of the stochastic comparison
       between the current and historical data.}
     \item{\code{posterior_regression}}{
       list. Contains results for the augmented regression analysis. Entries
       are similar to the output of \code{glm} and \code{bayesglm} (from the
        \code{arm} package).}
     \item{\code{posterior_flat_regression}}{
       list. Contains entries similar to \code{posterior_regression} corresponding
       to estimates of the unweighted current data.}
     \item{\code{prior_regression}}{
       to estimates of the historical data.}
   }
 \item{\code{args1}}{
   list. Entries contain user inputs. In addition, the following elements
   are ouput:}
   \itemize{
     \item{\code{df_t} and \code{df_c}}{
       dataframe. Input data parsed into internally used treatment and control
       data frames.
     }
     \item{\code{arm2}}{
       logical. Used internally to indicate one-arm or two-arm analysis.
     }
  }
}
}
\description{
\code{bdpregression} is used to estimate the adjusted intercept
  (single arm trial; OPC) or treatment effect (two-arm trial; RCT) for
  data using the regression analysis implementation of the
  Bayesian discount prior.
}
\details{
\code{bdpregression} uses a two-stage approach for determining the
  strength of historical data in estimation of an adjusted mean or covariate effect.
  In the first stage, a Weibull distribution function is used as a
  \emph{discount function} that defines the maximum strength of the
  historical data (via \code{weibull_shape}, \code{weibull_scale}, and
  \code{alpha_max}) and discounts based on disagreement with the current data.
  Disagreement between current and historical data is determined by stochastically
  comparing the respective posterior distributions under noninformative priors.
  With a single arm regression analysis, the comparison is the
  proability (\code{p}) that the current adjusted intercept is less than the
  historical adjusted intercept. The comparison metric \code{p} is then
  input into the Weibull discount function and the final strength of the
  historical data is returned (alpha).

  In the second stage, posterior estimation is performed where the discount
  function parameter, \code{alpha}, is used as a fixed value for all posterior
  estimation procedures.

 Two-arm (RCT) analyses are not currently available with this release.
}
\examples{
\dontrun{
# One-arm trial (OPC) example - linear regression
# Simulate regression data for a single arm (OPC) trial

set.seed(3408)
historical <- c(rep(1,50),rep(0,50))
x1         <- c(rnorm(50), rnorm(50))
y          <- c(rnorm(50), rnorm(50)+0.2)

fit1 <- bdpregression(y ~ x1 + historical)
summary(fit1)
print(fit1)
plot(fit1, type="discount")


# One-arm trial (OPC) example - logistic regression
set.seed(3408)
historical <- c(rep(1,100),rep(0,100))
x1         <- c(rep(0,50), rep(1,50), rep(0,50), rep(1,50))
y          <- rbinom(200,1,plogis(1 + 0.5*x1 + 0.1*historical))

fit2 <- bdpregression(y ~ x1 + historical,
                      family = binomial)
print(fit2)


# One-arm trial (OPC) example - Poisson regression
set.seed(3408)
historical <- c(rep(1,100),rep(0,100))
x1         <- c(rep(0,50), rep(1,50), rep(0,50), rep(1,50))
y          <- rpois(200,exp(1 + 0.5*x1 + 0.5*historical))

fit3 <- bdpregression(y ~ x1 + historical,
                      family = poisson)
summary(fit3)

# Refit the Poisson data ignoring historical
fit4 <- bdpregression(y ~ x1, family = poisson)
summary(fit4)


# Place data in a dataframe and carry out linear regression
set.seed(3408)
df <- data.frame(historical = c(rep(1,50),rep(0,50)),
                 x1         = c(rnorm(50), rnorm(50)),
                 y          = c(rnorm(50), rnorm(50)+0.2))

fit5 <- bdpregression(y ~ x1 + historical, data=df)
summary(fit5)


# Two-arm trials are not yet implemented.
}

}

