% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/estimate_density.R
\name{estimate_density}
\alias{estimate_density}
\alias{estimate_density.data.frame}
\alias{estimate_density.brmsfit}
\title{Density Estimation}
\usage{
estimate_density(x, ...)

\method{estimate_density}{data.frame}(
  x,
  method = "kernel",
  precision = 2^10,
  extend = FALSE,
  extend_scale = 0.1,
  bw = "SJ",
  ci = NULL,
  select = NULL,
  by = NULL,
  rvar_col = NULL,
  ...
)

\method{estimate_density}{brmsfit}(
  x,
  method = "kernel",
  precision = 2^10,
  extend = FALSE,
  extend_scale = 0.1,
  bw = "SJ",
  effects = "fixed",
  component = "conditional",
  parameters = NULL,
  ...
)
}
\arguments{
\item{x}{Vector representing a posterior distribution, or a data frame of such
vectors. Can also be a Bayesian model. \strong{bayestestR} supports a wide range
of models (see, for example, \code{methods("hdi")}) and not all of those are
documented in the 'Usage' section, because methods for other classes mostly
resemble the arguments of the \code{.numeric} or \code{.data.frame}methods.}

\item{...}{Currently not used.}

\item{method}{Density estimation method. Can be \code{"kernel"} (default), \code{"logspline"}
or \code{"KernSmooth"}.}

\item{precision}{Number of points of density data. See the \code{n} parameter in \code{density}.}

\item{extend}{Extend the range of the x axis by a factor of \code{extend_scale}.}

\item{extend_scale}{Ratio of range by which to extend the x axis. A value of \code{0.1}
means that the x axis will be extended by \code{1/10} of the range of the data.}

\item{bw}{See the eponymous argument in \code{density}. Here, the default has been
changed for \code{"SJ"}, which is recommended.}

\item{ci}{The confidence interval threshold. Only used when \code{method = "kernel"}.
This feature is experimental, use with caution.}

\item{select}{Character vector of column names. If \code{NULL} (the default), all
numeric variables will be selected. Other arguments from
\code{datawizard::extract_column_names()} (such as \code{exclude}) can also be used.}

\item{by}{Optional character vector. If not \code{NULL} and input is a data frame,
density estimation is performed for each group (subsets) indicated by \code{by}.
See examples.}

\item{rvar_col}{A single character - the name of an \code{rvar} column in the data
frame to be processed. See example in \code{\link[=p_direction]{p_direction()}}.}

\item{effects}{Should variables for fixed effects (\code{"fixed"}), random effects
(\code{"random"}) or both (\code{"all"}) be returned? Only applies to mixed models. May
be abbreviated.

For models of from packages \strong{brms} or \strong{rstanarm} there are additional
options:
\itemize{
\item \code{"fixed"} returns fixed effects.
\item \code{"random_variance"} return random effects parameters (variance and
correlation components, e.g. those parameters that start with \code{sd_} or
\code{cor_}).
\item \code{"grouplevel"} returns random effects group level estimates, i.e. those
parameters that start with \code{r_}.
\item \code{"random"} returns both \code{"random_variance"} and \code{"grouplevel"}.
\item \code{"all"} returns fixed effects and random effects variances.
\item \code{"full"} returns all parameters.
}}

\item{component}{Which type of parameters to return, such as parameters for
the conditional model, the zero-inflated part of the model, the dispersion
term, etc. See details in section \emph{Model Components}. May be abbreviated.
Note that the \emph{conditional} component also refers to the \emph{count} or \emph{mean}
component - names may differ, depending on the modeling package. There are
three convenient shortcuts (not applicable to \emph{all} model classes):
\itemize{
\item \code{component = "all"} returns all possible parameters.
\item If \code{component = "location"}, location parameters such as \code{conditional},
\code{zero_inflated}, \code{smooth_terms}, or \code{instruments} are returned (everything
that are fixed or random effects - depending on the \code{effects} argument -
but no auxiliary parameters).
\item For \code{component = "distributional"} (or \code{"auxiliary"}), components like
\code{sigma}, \code{dispersion}, \code{beta} or \code{precision} (and other auxiliary
parameters) are returned.
}}

\item{parameters}{Regular expression pattern that describes the parameters
that should be returned. Meta-parameters (like \code{lp__} or \code{prior_}) are
filtered by default, so only parameters that typically appear in the
\code{summary()} are returned. Use \code{parameters} to select specific parameters
for the output.}
}
\description{
This function is a wrapper over different methods of density estimation. By
default, it uses the base R \code{density} with by default uses a different smoothing
bandwidth (\code{"SJ"}) from the legacy default implemented the base R \code{density}
function (\code{"nrd0"}). However, Deng and Wickham suggest that \code{method = "KernSmooth"}
is the fastest and the most accurate.
}
\note{
There is also a \href{https://easystats.github.io/see/articles/bayestestR.html}{\code{plot()}-method} implemented in the \href{https://easystats.github.io/see/}{\pkg{see}-package}.
}
\section{Model components}{


Possible values for the \code{component} argument depend on the model class.
Following are valid options:
\itemize{
\item \code{"all"}: returns all model components, applies to all models, but will only
have an effect for models with more than just the conditional model
component.
\item \code{"conditional"}: only returns the conditional component, i.e. "fixed
effects" terms from the model. Will only have an effect for models with
more than just the conditional model component.
\item \code{"smooth_terms"}: returns smooth terms, only applies to GAMs (or similar
models that may contain smooth terms).
\item \code{"zero_inflated"} (or \code{"zi"}): returns the zero-inflation component.
\item \code{"location"}: returns location parameters such as \code{conditional},
\code{zero_inflated}, or \code{smooth_terms} (everything that are fixed or random
effects - depending on the \code{effects} argument - but no auxiliary
parameters).
\item \code{"distributional"} (or \code{"auxiliary"}): components like \code{sigma},
\code{dispersion}, \code{beta} or \code{precision} (and other auxiliary parameters) are
returned.
}

For models of class \code{brmsfit} (package \strong{brms}), even more options are
possible for the \code{component} argument, which are not all documented in detail
here. See also \href{https://easystats.github.io/insight/reference/find_parameters.BGGM.html}{\code{?insight::find_parameters}}.
}

\examples{
\dontshow{if (require("logspline") && require("KernSmooth") && require("mclust") && require("emmeans") && require("rstanarm") && require("brms")) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
library(bayestestR)

set.seed(1)
x <- rnorm(250, mean = 1)

# Basic usage
density_kernel <- estimate_density(x) # default method is "kernel"

hist(x, prob = TRUE)
lines(density_kernel$x, density_kernel$y, col = "black", lwd = 2)
lines(density_kernel$x, density_kernel$CI_low, col = "gray", lty = 2)
lines(density_kernel$x, density_kernel$CI_high, col = "gray", lty = 2)
legend("topright",
  legend = c("Estimate", "95\% CI"),
  col = c("black", "gray"), lwd = 2, lty = c(1, 2)
)

# Other Methods
density_logspline <- estimate_density(x, method = "logspline")
density_KernSmooth <- estimate_density(x, method = "KernSmooth")
density_mixture <- estimate_density(x, method = "mixture")

hist(x, prob = TRUE)
lines(density_kernel$x, density_kernel$y, col = "black", lwd = 2)
lines(density_logspline$x, density_logspline$y, col = "red", lwd = 2)
lines(density_KernSmooth$x, density_KernSmooth$y, col = "blue", lwd = 2)
lines(density_mixture$x, density_mixture$y, col = "green", lwd = 2)

# Extension
density_extended <- estimate_density(x, extend = TRUE)
density_default <- estimate_density(x, extend = FALSE)

hist(x, prob = TRUE)
lines(density_extended$x, density_extended$y, col = "red", lwd = 3)
lines(density_default$x, density_default$y, col = "black", lwd = 3)

# Multiple columns
head(estimate_density(iris))
head(estimate_density(iris, select = "Sepal.Width"))

# Grouped data
head(estimate_density(iris, by = "Species"))
head(estimate_density(iris$Petal.Width, by = iris$Species))
\donttest{
# rstanarm models
# -----------------------------------------------
library(rstanarm)
model <- suppressWarnings(
  stan_glm(mpg ~ wt + gear, data = mtcars, chains = 2, iter = 200, refresh = 0)
)
head(estimate_density(model))

library(emmeans)
head(estimate_density(emtrends(model, ~1, "wt", data = mtcars)))

# brms models
# -----------------------------------------------
library(brms)
model <- brms::brm(mpg ~ wt + cyl, data = mtcars)
estimate_density(model)
}
\dontshow{\}) # examplesIf}
}
\references{
Deng, H., & Wickham, H. (2011). Density estimation in R. Electronic publication.
}
