% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mediation.R
\name{mediation}
\alias{mediation}
\alias{mediation.brmsfit}
\alias{mediation.stanmvreg}
\title{Summary of Bayesian multivariate-response mediation-models}
\usage{
mediation(model, ...)

\method{mediation}{brmsfit}(
  model,
  treatment,
  mediator,
  response = NULL,
  centrality = "median",
  ci = 0.95,
  method = "ETI",
  ...
)

\method{mediation}{stanmvreg}(
  model,
  treatment,
  mediator,
  response = NULL,
  centrality = "median",
  ci = 0.95,
  method = "ETI",
  ...
)
}
\arguments{
\item{model}{A \code{brmsfit} or \code{stanmvreg} object.}

\item{...}{Not used.}

\item{treatment}{Character, name of the treatment variable (or direct effect)
in a (multivariate response) mediator-model. If missing, \code{mediation()}
tries to find the treatment variable automatically, however, this may fail.}

\item{mediator}{Character, name of the mediator variable in a (multivariate
response) mediator-model. If missing, \code{mediation()} tries to find the
treatment variable automatically, however, this may fail.}

\item{response}{A named character vector, indicating the names of the response
variables to be used for the mediation analysis. Usually can be \code{NULL},
in which case these variables are retrieved automatically. If not \code{NULL},
names should match the names of the model formulas,
\code{names(insight::find_response(model, combine = TRUE))}. This can be
useful if, for instance, the mediator variable used as predictor has a different
name from the mediator variable used as response. This might occur when the
mediator is transformed in one model, but used "as is" as response variable
in the other model. Example: The mediator \code{m} is used as response variable,
but the centered version \code{m_center} is used as mediator variable. The
second response variable (for the treatment model, with the mediator as
additional predictor), \code{y}, is not transformed. Then we could use
\code{response} like this: \code{mediation(model, response = c(m = "m_center", y = "y"))}.}

\item{centrality}{The point-estimates (centrality indices) to compute.  Character (vector) or list with one or more of these options: \code{"median"}, \code{"mean"}, \code{"MAP"} or \code{"all"}.}

\item{ci}{Value or vector of probability of the CI (between 0 and 1)
to be estimated. Default to \code{.89} (89\%) for Bayesian models and \code{.95} (95\%) for frequentist models.}

\item{method}{Can be \link[=eti]{'ETI'} (default), \link[=hdi]{'HDI'} or \link[=si]{'SI'}.}
}
\value{
A data frame with direct, indirect, mediator and
  total effect of a multivariate-response mediation-model, as well as the
  proportion mediated. The effect sizes are median values of the posterior
  samples (use \code{centrality} for other centrality indices).
}
\description{
\code{mediation()} is a short summary for multivariate-response
  mediation-models, i.e. this function computes average direct and average
  causal mediation effects of multivariate response models.
}
\details{
\code{mediation()} returns a data frame with information on the
      \emph{direct effect} (mean value of posterior samples from \code{treatment}
      of the outcome model), \emph{mediator effect} (mean value of posterior
      samples from \code{mediator} of the outcome model), \emph{indirect effect}
      (mean value of the multiplication of the posterior samples from
      \code{mediator} of the outcome model and the posterior samples from
      \code{treatment} of the mediation model) and the total effect (mean
      value of sums of posterior samples used for the direct and indirect
      effect). The \emph{proportion mediated} is the indirect effect divided
      by the total effect.
      \cr \cr
      For all values, the 89\% credible intervals are calculated by default.
      Use \code{ci} to calculate a different interval.
      \cr \cr
      The arguments \code{treatment} and \code{mediator} do not necessarily
      need to be specified. If missing, \code{mediation()} tries to find the
      treatment and mediator variable automatically. If this does not work,
      specify these variables.
      \cr \cr
      The direct effect is also called \emph{average direct effect} (ADE),
      the indirect effect is also called \emph{average causal mediation effects}
      (ACME). See also \cite{Tingley et al. 2014} and \cite{Imai et al. 2010}.
}
\note{
There is an \code{as.data.frame()} method that returns the posterior samples of the effects, which can be used for further processing in the different \pkg{bayestestR} package.
}
\examples{
\dontrun{
library(mediation)
library(brms)
library(rstanarm)

# load sample data
data(jobs)
set.seed(123)

# linear models, for mediation analysis
b1 <- lm(job_seek ~ treat + econ_hard + sex + age, data = jobs)
b2 <- lm(depress2 ~ treat + job_seek + econ_hard + sex + age, data = jobs)
# mediation analysis, for comparison with Stan models
m1 <- mediate(b1, b2, sims = 1000, treat = "treat", mediator = "job_seek")

# Fit Bayesian mediation model in brms
f1 <- bf(job_seek ~ treat + econ_hard + sex + age)
f2 <- bf(depress2 ~ treat + job_seek + econ_hard + sex + age)
m2 <- brm(f1 + f2 + set_rescor(FALSE), data = jobs, cores = 4, refresh = 0)

# Fit Bayesian mediation model in rstanarm
m3 <- stan_mvmer(
  list(
    job_seek ~ treat + econ_hard + sex + age + (1 | occp),
    depress2 ~ treat + job_seek + econ_hard + sex + age + (1 | occp)
  ),
  data = jobs,
  cores = 4,
  refresh = 0
)

summary(m1)
mediation(m2, centrality = "mean", ci = .95)
mediation(m3, centrality = "mean", ci = .95)
}
}
\references{
\itemize{
\item Imai, K., Keele, L. and Tingley, D. (2010) A General Approach to Causal Mediation Analysis, Psychological Methods, Vol. 15, No. 4 (December), pp. 309-334.
\item Tingley, D., Yamamoto, T., Hirose, K., Imai, K. and Keele, L. (2014). mediation: R package for Causal Mediation Analysis, Journal of Statistical Software, Vol. 59, No. 5, pp. 1-38.
}
}
\seealso{
The \pkg{mediation} package for a causal mediation analysis in
  the frequentist framework.
}
