% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/classification.R
\name{LL.ROC}
\alias{LL.ROC}
\title{ROC curves for the Livingston and Lewis approach.}
\usage{
LL.ROC(
  x = NULL,
  reliability,
  min = 0,
  max = 1,
  truecut,
  true.model = "4P",
  error.model = "Binomial",
  failsafe = TRUE,
  l = 0,
  u = 1,
  AUC = FALSE,
  maxJ = FALSE,
  maxAcc = FALSE,
  locate = NULL,
  raw.out = FALSE,
  grainsize = 100
)
}
\arguments{
\item{x}{A vector of observed results.}

\item{reliability}{The reliability coefficient of the test.}

\item{min}{The minimum possible value to attain on the observed-score scale. Default is 0 (assuming \code{x} represent proportions).}

\item{max}{The maximum possible value to attain on the observed-score scale. Default is 1 (assuming \code{x} represent proportions).}

\item{truecut}{The true point along the x-scale that marks the categorization-threshold.}

\item{true.model}{The probability distribution to be fitted to the moments of the true-score distribution. Options are \code{"4P"} (default) and \code{"2P"}, referring to four- and two-parameter Beta distributions. The \code{"4P"} method produces a four-parameter Beta distribution with the same first four moments (mean, variance, skewness, and kurtosis) as the estimated true-score distribution, while the \code{"2P"} method produces a two-parameter Beta distribution with the first two moments (mean and variance) as the estimated true-score distribution.}

\item{error.model}{The probability distribution to be used for producing the sampling distributions at different points of the true-score scale. Options are \code{binomial} and \code{beta}. The binomial distribution is discrete, and is the distribution used originally by Livingston and Lewis. Use of the binomial distribution involves a rounding of the effective test length to the nearest integer value. The Beta distribution is continuous, and does not involve rounding of the effective test length.}

\item{failsafe}{If true-model == "4P": Whether to engage a fail-safe reverting to a two-parameter true-score distribution solution should the four-parameter fitting procedure produce impermissible results. Default is TRUE (engage fail-safe in the event of impermissible estimates).}

\item{l}{If \code{true.model == "2P"} or \code{failsafe == TRUE}: The lower-bound location parameter of the two-parameter true-score distribution solution.}

\item{u}{If \code{true.model == "2P"} or \code{failsafe == TRUE}: The upper-bound location parameter of the two-parameter true-score distribution solution.}

\item{AUC}{Calculate and include the area under the curve? Default is \code{FALSE}.}

\item{maxJ}{Logical. Mark the point along the curve where Youden's J statistic is maximized? Default is \code{FALSE}.}

\item{maxAcc}{Logical. Mark the point along the curve where the Accuracy statistic is maximized? Default is \code{FALSE}.}

\item{locate}{Ask the function to locate the cut-point at which sensitivity or NPV is greater than or equal to some value, or specificity or PPV is lesser than or equal to some value. Take as input a character-vector of length 2, with the first argument being which index is to be found (e.g., "sensitivity"), and the second argument the value to locate (e.g., "0.75"). For example: c("sensitivity", "0.75").}

\item{raw.out}{Give raw coordinates as output rather than plot? Default is \code{FALSE}}

\item{grainsize}{Specify the number of cutoff-points for which the ROC curve is to be calculated. The greater this number the greater the accuracy. Default is 100 points.}
}
\value{
A plot tracing the ROC curve for the test, or matrix of coordinates if raw.out is \code{TRUE}.
}
\description{
Generate a ROC curve plotting the false-positive rate against the true-positive rate at different cut-off values across the observed proportion-score scale.
}
\examples{
# Generate some fictional data. Say, 1000 individuals take a test with a
# maximum score of 100 and a minimum score of 0.
set.seed(1234)
testdata <- rbinom(1000, 100, rBeta.4P(1000, 0.25, 0.75, 5, 3))
hist(testdata / 100, xlim = c(0, 1), freq = FALSE)

# Suppose the cutoff value for attaining a pass is 50 items correct.
# Suppose further that the reliability of the test-scores were estimated to
# 0.75. To produce a plot with an ROC curve using LL.ROC(), along with the
# AUC statistics and the points at which Youden's J. is maximized:
LL.ROC(x = testdata, reliability = 0.7, truecut = 50, min = 0, max = 100,
AUC = TRUE, maxJ = TRUE)
# Or to locate the point at which accuracy is maximized:
LL.ROC(x = testdata, reliability = 0.7, truecut = 50, min = 0, max = 100,
maxAcc = TRUE)

# Using the example data above, the function can be instructed to locate an
# operational cut-point at which sensitivity or specificity is equal to or
# greater than some specified value by specifying the "locate" argument with
# c("statistic", value). For example, to locate the operational cut-point at
# which sensitivity is first equal to or greater than 0.9:
LL.ROC(x = testdata, reliability = 0.7, min = 0, max = 100, truecut = 50,
locate = c("sensitivity", 0.9))
# For Negative Predictive value, the point at which it is equal or greater:
LL.ROC(x = testdata, reliability = 0.7, min = 0, max = 100, truecut = 50,
locate = c("NPV", 0.9))
# For specificity, the point at which it is equal to or less than 0.9:
LL.ROC(x = testdata, reliability = 0.7, min = 0, max = 100, truecut = 50,
locate = c("specificity", 0.9))
# For Positive Predictive Value, the point at which it is equal or less:
LL.ROC(x = testdata, reliability = 0.7, min = 0, max = 100, truecut = 50,
locate = c("PPV", 0.90))
}
