% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/SVD.R
\name{big_SVD}
\alias{big_SVD}
\title{Partial SVD}
\usage{
big_SVD(X, fun.scaling, ind.row = rows_along(X), ind.col = cols_along(X),
  k = 10, block.size = block_size(nrow(X)))
}
\arguments{
\item{X}{A \link[=FBM-class]{FBM}.}

\item{fun.scaling}{A function that returns a named list of
\code{mean} and \code{sd} for every column, to scale each of their elements
such as followed: \deqn{\frac{X_{i,j} - mean_j}{sd_j}.}}

\item{ind.row}{An optional vector of the row indices that are used.
If not specified, all rows are used. \strong{Don't use negative indices.}}

\item{ind.col}{An optional vector of the column indices that are used.
If not specified, all columns are used. \strong{Don't use negative indices.}}

\item{k}{Number of singular vectors/values to compute. Default is \code{10}.}

\item{block.size}{Maximum number of columns read at once.
Default uses \link{block_size}.}
}
\value{
A named list (an S3 class "big_SVD") of
\itemize{
\item \code{d}, the singular values,
\item \code{u}, the left singular vectors,
\item \code{v}, the right singular vectors,
\item \code{center}, the centering vector,
\item \code{scale}, the scaling vector.
}

Note that to obtain the Principal Components, you must use
\link[=predict.big_SVD]{predict} on the result. See examples.
}
\description{
An algorithm for partial SVD (or PCA) of a Filebacked Big Matrix through the
eigen decomposition of the covariance between variables (primal)
or observations (dual). \strong{Use this algorithm only if there is one dimension
that is much smaller than the other. Otherwise use \link{big_randomSVD}.}
}
\details{
To get \eqn{X = U \cdot D \cdot V^T},
\itemize{
\item if the number of observations is small, this function computes
\eqn{K_(2) = X \cdot X^T \approx U \cdot D^2 \cdot U^T} and then
\eqn{V = X^T \cdot U \cdot D^{-1}},
\item if the number of variable is small, this function computes
\eqn{K_(1) = X^T \cdot X \approx V \cdot D^2 \cdot V^T} and then
\eqn{U = X \cdot V \cdot D^{-1}},
\item if both dimensions are large, use \link{big_randomSVD} instead.
}
}
\examples{
set.seed(1)

X <- big_attachExtdata()
n <- nrow(X)

# Using only half of the data
ind <- sort(sample(n, n/2))

test <- big_SVD(X, fun.scaling = big_scale(), ind.row = ind)
str(test)
plot(test$u)

pca <- prcomp(X[ind, ], center = TRUE, scale. = TRUE)

# same scaling
all.equal(test$center, pca$center)
all.equal(test$scale,  pca$scale)

# scores and loadings are the same or opposite
# except for last eigenvalue which is equal to 0
# due to centering of columns
scores <- test$u \%*\% diag(test$d)
class(test)
scores2 <- predict(test) # use this function to predict scores
all.equal(scores, scores2)
dim(scores)
dim(pca$x)
tail(pca$sdev)
plot(scores2, pca$x[, 1:ncol(scores2)])
plot(test$v[1:100, ], pca$rotation[1:100, 1:ncol(scores2)])

# projecting on new data
X2 <- sweep(sweep(X[-ind, ], 2, test$center, '-'), 2, test$scale, '/')
scores.test <- X2 \%*\% test$v
ind2 <- setdiff(rows_along(X), ind)
scores.test2 <- predict(test, X, ind.row = ind2) # use this
all.equal(scores.test, scores.test2)
scores.test3 <- predict(pca, X[-ind, ])
plot(scores.test2, scores.test3[, 1:ncol(scores.test2)])
}
\seealso{
\link[stats:prcomp]{prcomp}
}
