% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/blocks.r
\name{blocks}
\alias{blocks}
\title{Block designs}
\usage{
blocks(treatments, replicates = 1, rows = NULL, columns = NULL,
  model = NULL, searches = NULL, seed = sample(10000, 1), jumps = 1)
}
\arguments{
\item{treatments}{either a data frame with columns for individual treatment factors or a partition
of the total required number of treatments into sets of equally replicated treatments.}

\item{replicates}{either a single replication number, not necessarily integral, if the \code{treatments} parameter is a data frame, or a set of 
replication numbers, one per replication set, if the \code{treatments} parameter is a partition.}

\item{rows}{the numbers of rows nested in each higher-level block for each level of nesting from the top block downwards. The top-level block is a
single super-block which does not need to be defined unless a completely randomized design is required. The default number of blocks is the hcf
of the replication numbers, which gives a maximal set of orthogonal row blocks.}

\item{columns}{the numbers of columns nested in each higher-level block for each level of nesting from the top block downwards. The \code{rows} and 
\code{columns} parameters must be of equal length unless the \code{columns} parameter is null, in which case the columns block design degenerates to a
single column block for each level of nesting, which gives a simple nested row blocks design.}

\item{model}{a model equation for the treatment factors in the design where the equation is defined using the model.matrix notation
in the {\link[stats]{model.matrix}} package. If undefined, the model is a full factorial treatment design.}

\item{searches}{the maximum number of local optima searched for a design optimization. The default is 1 plus the floor of 10000 divided by the number of plots.}

\item{seed}{an integer initializing the random number generator. The default is a random seed.}

\item{jumps}{the number of pairwise random treatment swaps used to escape a local maxima. The default is a single swap.}
}
\value{
\item{Treatments}{The treatment factors defined by the \code{treatments} inputs in standard factorial order.}
\item{model.matrix}{The model.matrix used to define the \code{treatments} design.}
\item{Design}{Data frame giving the optimized block and treatment factors in plot order.}
\item{Plan}{Data frame for single factor designs showing a plan view of the treatment design in the bottom stratum of the design. A NULL plan is returned for multi-factor designs.}
\item{BlocksEfficiency}{The D-efficiencies and the A-efficiencies (unstructured designs) of the blocks in each stratum of the design together with A-efficiency upper-bounds, where available.}
\item{DesignEfficiency}{The generalized variance of the complete factorial design divided by the generalized variance of the fractional factorial design.}
\item{seed}{Numerical seed for random number generator.}
\item{searches}{Maximum number of searches in each stratum.}
\item{jumps}{Number of random treatment swaps to escape a local maxima.}
}
\description{
Constructs randomized nested block designs for factorial or fractional factorial or unstructured treatment designs with any
feasible depth of nesting and up to two crossed block structures in each level of nesting.
}
\details{
Constructs randomized nested block designs with arbitrary depth of nesting for factorial or fractional
factorial  or unstructured treatment designs. The treatment model can be any arbitrary combination of quantitative
or qualitative factorial model terms or can be a single set of unstructured treatments. 
 
The \code{treatments} parameter defines the treatment factors of the design and can be either a data frame with
a column for each factor and a row for each factor combination (see examples) or a set of P cardinal numbers for an unstructured treatment set
where each cardinal represents a set of equally replicated treatments and the sum of the cardinals is the total number of treatments
(see examples). 

If the \code{treatments} parameter is a data frame, the treatment factors can be any mixture of qualitative or quantitative level factors and the 
treatment model can be any feasible model defined by the \code{models} formula of the \code{\link[stats]{model.matrix}} package (see examples). 

Quantitative factors can be modelled either by raw or by orthogonal polynomials. Orthogonal polynomials are numerically more stable
than raw polynomials and are usually the best choice at the design stage. Polynomial models can be fitted at the analysis stage either by raw or
by orthogonal polynomials regardless of the type of polynomial fitted at the design stage.
  
The \code{replicates} parameter defines the required replication for the treatments design.  
If the \code{treatments} parameter is a data frame, the replication number must be a single number, not necessarily integral, representing any
required multiple or any required fraction of the \code{treatments} data frame. If the \code{treatments} parameter is a set of cardinal numbers, the 
\code{replicates} parameter must be a matching set of replication numbers, one for each equally replicated treatments sets (see examples).

If the \code{treatments} parameter is a data frame and if the replication number is non-integral, the algorithm will find a
D-optimal or near D-optimal fraction of the required size for the fractional part of replication number, assuming the required design is non-singular.

The \code{rows} parameter, if any, defines the nested row blocks for each level of nesting taken in order from the highest to the lowest. The
first number, if any, is the number of nested row blocks in the first-level of nesting, the second number, if any, is the number of nested row blocks in
the second-level of nesting and so on for the required feasible depth of nesting.

The \code{columns} parameter, if any, defines the nested column blocks for each level of nesting taken in order from the highest to the lowest.
The first number, if any, is the number of nested column blocks in the first-level of nesting, the second, if any, is the number of nested column blocks in
the second-level of nesting and so on for the required feasible depth of nesting.

The \code{rows} and \code{columns} parameters, if defined, must be of equal length and if a simple nested blocks design is required for
any particular level of nesting, the number of columns for that level should be set to unity. Any required combination of simple or
crossed blocks can be obtained by appropriate choice of the levels of the \code{rows} and \code{columns} parameters.
 
If the \code{columns} parameter is undefined, a single crossed block is assumed for each level of nesting. 

If both the \code{rows} parameter and the \code{columns} parameter are null, the default block design will be a set of orthogonal
main blocks equal in number to the highest common factor of the replication numbers. If the \code{rows} parameter is defined but the \code{columns}
parameter is null, the design will have simple nested blocks for each level of nesting, as defined by the \code{rows} parameter.

Block sizes are always as nearly equal as possible and will never differ by more than a single plot for any particular block classification. 
Row blocks and column blocks must always contain at least two plots per block and this restriction will constrain the permitted numbers of 
rows and columns in the various nested levels of a block design.

Unreplicated treatments are allowed and any simple nested block design can be augmented by any number of single unreplicated treatments 
to give augmented blocks that never differ in size by more than a single plot. General crossed block designs are more complex and currently 
the algorithm will only accommodate single unreplicated treatments in a crossed block design if the block sizes of the replicated part of 
the design are all equal in each nested level of the design.

For any particular level of nesting, the algorithm first optimizes the row blocks conditional on any higher-level blocks
and then optimizes the columns blocks, if any, conditional on the rows blocks.

The efficiency factor for a fractional factorial design is the generalized variance of the complete factorial design divided by the generalized variance of
the fractional factorial design where the generalized variance of a design is the (1/p)th power of the determinant of the crossed-product of the p-dimensional
model matrix divided by the number of observations in the design. 

Special designs:

Trojan designs are row-and-column designs for p replicates of v*p treatments arranged in p-rows and p-columns where v < p and 
where every row x column intersection contains v plots. Trojan designs have orthogonal rows and columns and optimal rows x columns
blocks and exist whenever p is prime or prime-power. The \code{blocksdesign} constructs these designs algebraically from mutually
orthogonal Latin squares (MOLS).

Square lattice designs are efficient resolvable incomplete block designs for r replicates of p*p treatments arranged in blocks of size p where
r < p+2 for prime or prime power p or r < 4 for general p. \code{blocksdesign} constructs lattice designs algebraically from Latin squares or MOLS.

Lattice designs and Trojan designs based on prime-power MOLS require the \code{\link[crossdes]{MOLS}} package.

All other designs are constructed algorithmically.

Comment:

Row-and-column designs may contain useful treatment information in the individual row-by-column intersection blocks but \code{blocksdesign} does not currently
optimize the efficiency of these blocks except for the special case of Trojan designs.

Row-and-column design with 2 complete treatment replicates, 2 complete rows and 2 complete columns will always confound one treatment contrast in the
rows-by-columns interaction. For these designs, it is impossible to nest a non-singular block design in the rows-by-columns intersections and instead
we suggest a randomized nested blocks design with four incomplete main blocks.

Outputs:

The principle design outputs comprise:
\itemize{
 \item  A data frame showing the allocation of treatments to blocks with successive nested strata arranged in standard block order. \cr
 \item  A table showing the replication number of each treatment in the design. \cr
 \item  An efficiency factor for fractional factorial treatment designs. \cr
 \item  A table showing the block levels and the achieved D-efficiency and A-efficiency (unstructured treatment designs only) factors for each stratum together
  with A-efficiency upper bounds, where available. \cr
 \item  A plan showing the allocation of treatments to blocks or to rows and to columns in the bottom stratum of the design (unstructured treatment
 designs only).\cr
}
}
\examples{

## The number of searches in the following examples have been limited for fast execution.  
## In practice, the number of searches may need to be increased for optimum results.
## Designs should be rebuilt several times to check that a near-optimum design has been found.  


## Factorial designs defined by a treatments data frame and a factorial model equation.

# Main effects of five 2-level factors in a half-fraction of a 4 x 4 row-and column design.
GF = expand.grid(F1=factor(1:2),F2=factor(1:2),F3=factor(1:2),F4=factor(1:2),F5=factor(1:2))
blocks(treatments=GF,model="~ F1+F2+F3+F4+F5",replicates=.5,rows=4,columns=4,searches=20)

# Quadratic regression for one 6-level numeric factor in 2 randomized blocks
blocks(treatments=expand.grid(X=1:6),model=" ~ poly(X,2)",rows=2,searches=5) 

# Second-order model for five qualitative 2-level factors in 4 randomized blocks
GF=expand.grid(F1=factor(1:2),F2=factor(1:2),F3=factor(1:2),F4=factor(1:2),F5=factor(1:2))
blocks(treatments=GF,model=" ~ (F1+F2+F3+F4+F5)*(F1+F2+F3+F4+F5)",rows=4,searches=5)

# First-order model for 1/3rd fraction of four qualitative 3-level factors in 3  blocks
GF=expand.grid(F1=factor(1:3),F2=factor(1:3),F3=factor(1:3),F4=factor(1:3))
blocks(treatments=GF,model=" ~ (F1+F2+F3+F4)",replicates=(1/3),rows=3,searches=5)

# Second-order model for a 1/3rd fraction of five qualitative 3-level factors in 3 blocks
GF=expand.grid( F1=factor(1:3), F2=factor(1:3), F3=factor(1:3), F4=factor(1:3), F5=factor(1:3) )
modelform=" ~ (F1+F2+F3+F4+F5)*(F1+F2+F3+F4+F5)"
blocks(treatments=GF,model=modelform,rows=3,replicates=(1/3),searches=1)

# Second-order model for two qualitative and two quantitative level factors in 4 randomized blocks
GF=expand.grid(F1=factor(1:2),F2=factor(1:3),V1=1:3,V2=1:4)
modelform=" ~ F1 + F2 + poly(V1,2) +  poly(V2,2) + (poly(V1,1)+F1+F2):(poly(V2,1)+F1+F2) "
blocks(treatments=GF,model=modelform,rows=4,searches=5)

# Plackett and Burman design for eleven 2-level factors in 12 runs (needs large number of searches)
GF=expand.grid(F1=factor(1:2),F2=factor(1:2),F3=factor(1:2),F4=factor(1:2),F5=factor(1:2),
F6=factor(1:2),F7=factor(1:2),F8=factor(1:2),F9=factor(1:2),F10=factor(1:2),F11=factor(1:2))
\dontrun{blocks(treatments=GF,model="~ F1+F2+F3+F4+F5+F6+F7+F8+F9+F10+F11",replicates=(12/2048))}


## Unstructured treatments partitioned into equally replicated treatment sets

# 3 treatments x 2 replicates + 2 treatments x 4 replicates 
blocks(treatments=c(3,2),replicates=c(2,4),searches=10)

# 4 treatments x 4 replicates with 2 main rows each containing two complete replicates
blocks(treatments=4,replicates=4,rows=2)

# 50 treatments x 4 replicates with 4 main blocks and 5 nested sub-blocks in each main block
blocks(treatments=50,replicates=4,rows=c(4,5))

# as above but with 20 single replicate treatments giving one extra treatment per sub-block
blocks(treatments=c(50,20),replicates=c(4,1),rows=c(4,5))

# 6 replicates of 6 treatments in 4 blocks of size 9 (non-binary block design)
blocks(treatments=6,replicates=6,rows=4)

# 4 replicates of 13 treatments arranged in a 13 x 4 Youden rectangle
blocks(treatments=13,replicates=4,rows=13,columns=4)

# 64 treatments x 2 replicates with nested 8 x 8 row-and-column designs in two main blocks
blocks(treatments=64,replicates=2,rows=c(2,8),columns=c(1,8),searches=10)

# 64 treatments x 2 replicates with two main blocks and a 4 x 4 row-and-column in each main block
blocks(treatments=64,replicates=2,rows=c(2,4),columns=c(1,4),searches=10)

# 128 treatments x 2 replicates with two main blocks and 3 levels of nesting
\dontrun{blocks(128,2,c(2,2,2,2))}


}
\references{
Sailer, M. O. (2013). crossdes: Construction of Crossover Designs. R package version 1.1-1. https://CRAN.R-project.org/package=crossdes

Edmondson R. N. (1998). Trojan square and incomplete Trojan square designs for crop research. Journal of Agricultural Science, Cambridge, 131, pp.135-142

Cochran, W.G., and G.M. Cox. 1957. Experimental Designs, 2nd ed., Wiley, New York.
}
