% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bootstrapFP.R
\name{bootstrapFP}
\alias{bootstrapFP}
\title{Bootstrap algorithms for Finite Population sampling}
\usage{
bootstrapFP(
  y,
  pik,
  B,
  D = 1,
  method,
  design,
  x = NULL,
  s = NULL,
  distribution = "uniform"
)
}
\arguments{
\item{y}{vector of sample values}

\item{pik}{vector of sample first-order inclusion probabilities}

\item{B}{scalar, number of bootstrap replications}

\item{D}{scalar, number of replications for the double bootstrap (when applicable)}

\item{method}{a string indicating the bootstrap method to be used, see Details for more}

\item{design}{sampling procedure to be used for sample selection.
Either a string indicating the name of the sampling design or a function;
see section "Details" for more information.}

\item{x}{vector of length N with values of the auxiliary variable for all population units,
only required if method "ppHotDeck" is chosen}

\item{s}{logical vector of length N, TRUE for units in the sample, FALSE otherwise. 
Alternatively, a vector of length n with the indices of the sample units.
Only required for "ppHotDeck" method.}

\item{distribution}{required only for \code{method='generalised'}, a string
indicating the distribution to use for the Generalised bootstrap. 
Available options are "uniform", "normal", "exponential" and "lognormal"}
}
\value{
The bootstrap variance of the Horvitz-Thompson estimator.
}
\description{
Bootstrap variance estimation for finite population sampling.
}
\details{
Argument \code{design} accepts either a string indicating the sampling design
to use to draw samples or a function.
Accepted designs are "brewer", "tille", "maxEntropy", "poisson",
"sampford", "systematic", "randomSystematic".
The user may also pass a function as argument; such function should take as input
the parameters passed to argument \code{design_pars} and return either a logical
vector or a vector of 0 and 1,  where \code{TRUE} or \code{1} indicate sampled
units and \code{FALSE} or \code{0} indicate non-sample units.
The length of such vector must be equal to the length of \code{x}
if \code{units} is not specified, otherwise it must have the same length of \code{units}.

\code{method} must be a string indicating the bootstrap method to use.
A list of the currently available methods follows, the sampling design they
they should be used with is indicated in square brackets.
The prefix "pp" indicates a pseudo-population method, the prefix "d"
represents a direct method, and the prefix "w" inicates a weights method.
For more details on these methods see Mashreghi et al. (2016).

\itemize{
    \item "ppGross" [SRSWOR]
    \item "ppBooth" [SRSWOR]
    \item "ppChaoLo85" [SRSWOR]
    \item "ppChaoLo94" [SRSWOR]
    \item "ppBickelFreedman" [SRSWOR]
    \item "ppSitter" [SRSWOR]
    
    \item "ppHolmberg" [UPSWOR]
    \item "ppChauvet"  [UPSWOR]
    \item "ppHotDeck"  [UPSWOR]
    
    \item "dEfron" [SRSWOR]
    \item "dMcCarthySnowden" [SRSWOR]
    \item "dRaoWu" [SRSWOR]
    \item "dSitter" [SRSWOR]
    \item "dAntalTille_UPS" [UPSWOR]
    
    \item "wRaoWuYue"    [SRSWOR]
    \item "wChipperfieldPreston"    [SRSWOR]
    \item "wGeneralised" [any]

}
}
\examples{

library(bootstrapFP)

### Generate population data ---
N   <- 20; n <- 5
x   <- rgamma(N, scale=10, shape=5)
y   <- abs( 2*x + 3.7*sqrt(x) * rnorm(N) )
pik <- n * x/sum(x)

### Draw a dummy sample ---
s  <- sample(N, n)

### Estimate bootstrap variance ---
bootstrapFP(y = y[s], pik = n/N, B=100, method = "ppSitter")
bootstrapFP(y = y[s], pik = pik[s], B=10, method = "ppHolmberg", design = 'brewer')
bootstrapFP(y = y[s], pik = pik[s], B=10, D=10, method = "ppChauvet")
bootstrapFP(y = y[s], pik = n/N, B=10, method = "dRaoWu")
bootstrapFP(y = y[s], pik = n/N, B=10, method = "dSitter")
bootstrapFP(y = y[s], pik = pik[s], B=10, method = "dAntalTille_UPS", design='brewer')
bootstrapFP(y = y[s], pik = n/N, B=10, method = "wRaoWuYue") 
bootstrapFP(y = y[s], pik = n/N, B=10, method = "wChipperfieldPreston")
bootstrapFP(y = y[s], pik = pik[s], B=10, method = "wGeneralised", distribution = 'normal')



}
\references{
Mashreghi Z.; Haziza D.; Léger C., 2016. A survey of bootstrap methods in 
finite population sampling. Statistics Surveys 10 1-52.
}
