#' Calculate graph vulnerability
#'
#' This function calculates the \emph{vulnerability} of the vertices of a graph.
#' Here, vulnerability is considered to be the proportional drop in global
#' efficiency when a given node is removed from the graph. The vulnerability of
#' the graph is considered the maximum across all vertices.
#'
#' @param g The igraph graph object of interest
#' @param use.parallel Logical indicating whether or not to use \emph{foreach}
#'   (default: TRUE)
#' @param weighted Logical indicating whether weighted efficiency should be
#' calculated (default: FALSE)
#' @export
#'
#' @return A vector of length equal to the number of vertices in \emph{g}
#'
#' @seealso \code{\link{graph.efficiency}}
#' @author Christopher G. Watson, \email{cgwatson@@bu.edu}
#' @references Latora V., Marchiori M. (2005) \emph{Variability and protection
#' of infrastructure networks}. Physical Review E, 71:015103.

vulnerability <- function(g, use.parallel=TRUE, weighted=FALSE) {
  stopifnot(is_igraph(g))
  if (isTRUE(weighted)) {
    weights <- NULL
    g.attr <- 'E.global.wt'
  } else {
    weights <- NA
    g.attr <- 'E.global'
  }
  if (!g.attr %in% graph_attr_names(g)) {
    E.global <- graph.efficiency(g, 'global', weights)
  } else {
    E.global <- graph_attr(g, g.attr)
  }
  if (!'degree' %in% vertex_attr_names(g)) {
    degs <- degree(g)
  } else {
    degs <- V(g)$degree
  }
  vuln <- rep(0, vcount(g))
  inds <- which(degs > 0)
  if (isTRUE(use.parallel)) {
    vuln[inds] <- foreach(i=seq_along(inds), .combine='c') %dopar% {
      g.tmp <- delete.vertices(g, inds[i])
      E.global.tmp <- graph.efficiency(g.tmp, 'global', weights)
      1 - (E.global.tmp / E.global)
    }
  } else {
    for (i in inds) {
      g.tmp <- delete.vertices(g, i)
      E.global.tmp <- graph.efficiency(g.tmp, 'global', weights)
      vuln[i] <- 1 - (E.global.tmp / E.global)
    }
  }
  return(vuln)
}
